﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nnt { namespace testing { namespace detail {

template <typename InputIt>
struct IteratorTraits final
{
    using difference_type = typename InputIt::difference_type;
    using value_type = typename InputIt::value_type;
};

template <typename T>
struct IteratorTraits<T*> final
{
    using difference_type = ptrdiff_t;
    using value_type = T;
};

template <typename T>
struct IteratorTraits<const T*> final
{
    using difference_type = ptrdiff_t;
    using value_type = T;
};

template <typename Container>
class BackInsertIterator final
{
private:
    Container* m_pContainer;

public:
    explicit BackInsertIterator(Container& container) NN_NOEXCEPT
        : m_pContainer(&container)
    {
    }

    BackInsertIterator& operator=(
        const typename Container::value_type& value) NN_NOEXCEPT
    {
        m_pContainer->push_back(value);
        return *this;
    }

    BackInsertIterator& operator*() NN_NOEXCEPT
    {
        return *this;
    }

    BackInsertIterator& operator++() NN_NOEXCEPT
    {
        return *this;
    }

    BackInsertIterator operator++(int) NN_NOEXCEPT
    {
        return *this;
    }
};

class Iterator final
{
public:
    template <typename T>
    static auto Begin(T& c) NN_NOEXCEPT -> decltype(c.begin())
    {
        return c.begin();
    }

    template <typename T>
    static auto Begin(const T& c) NN_NOEXCEPT -> decltype(c.begin())
    {
        return c.begin();
    }

    template <typename T, size_t N>
    static T* Begin(T (&array)[N]) NN_NOEXCEPT
    {
        return array;
    }

    template <typename T>
    static auto End(T& c) NN_NOEXCEPT -> decltype(c.end())
    {
        return c.end();
    }

    template <typename T>
    static auto End(const T& c) NN_NOEXCEPT -> decltype(c.end())
    {
        return c.end();
    }

    template <typename T, size_t N>
    static T* End(T (&array)[N]) NN_NOEXCEPT
    {
        return array + N;
    }

    template <typename InputIt>
    static typename IteratorTraits<InputIt>::difference_type Distance(
        InputIt head, InputIt tail) NN_NOEXCEPT
    {
        typename IteratorTraits<InputIt>::difference_type n = 0;
        for (; head != tail; ++head) { ++n; }
        return n;
    }

    template <typename T>
    static typename IteratorTraits<T*>::difference_type Distance(
        T* head, T* tail) NN_NOEXCEPT
    {
        return tail - head;
    }

    template <typename T>
    static typename IteratorTraits<const T*>::difference_type Distance(
        const T* head, const T* tail) NN_NOEXCEPT
    {
        return tail - head;
    }

    template <typename Container>
    static BackInsertIterator<Container> BackInserter(
        Container& container) NN_NOEXCEPT
    {
        return BackInsertIterator<Container>(container);
    }

private:
    Iterator();

    NN_DISALLOW_COPY(Iterator);
    NN_DISALLOW_MOVE(Iterator);
};

}}} // namespace nnt::testing::detail
