﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @namespace   nn::cdr
 * @brief       Library to analyze nxdmp files
 * @details     Core Dump Report (CDR) is a utility for reading NxDmp files.
 *              It can be used to extract basic information from a NxDmp file
 *              including: the exception code and description, the loaded modules,
 *              the threads list, and the call stacks for each thread.
 */

#pragma once

//==============================================================================
//  INCLUDES
//==============================================================================

#include <stdint.h>

#ifdef _CDR_STATIC_LIB
#define CDR_DLL_DECL
#else
#define CDR_DLL_DECL __declspec( dllexport )
#endif

//==============================================================================

namespace nn {

namespace cdr {

/** @brief Size of the thread name buffer, including the null-termination.*/
const int32_t ThreadNameSize = 256;

/** @brief Size of the module id buffer, including the null-termination.*/
const int32_t ModuleIdSize = 32;

/** @brief Size of the function name buffer, including the null-termination.*/
const int32_t FunctionNameSize = 2048;

/** @brief Size of the file name buffer, including the null-termination.*/
const int32_t FilePathSize = 1024;

/** @brief Size of the exception description buffer, including the null-termination.*/
const int32_t ExceptionBufferSize = 512;

/** @brief A unique id for specifying the thread in the core dump file.*/
typedef uint64_t ThreadId;

/** @brief Core dump report error codes.*/
enum Error : int32_t
{
    Error_Ok                                = 0,    ///< No error.
    Error_FileNotFound                      = 2,    ///< Unable to find the coredump file.
    Error_FileNoPermissions                 = 3,    ///< Permission error occurred while opening the file.
    Error_FileIoError                       = 4,    ///< IO problem opening the file.
    Error_FileCorrupt                       = 5,    ///< Coredump file is corrupt.
    Error_FileNotOpen                       = 6,    ///< No coredump file open for this query.
    Error_FileNotCoredump                   = 7,    ///< Not a coredump file.
    Error_ThreadNotFound                    = 8,    ///< Invalid thread id or index specified.
    Error_ModuleNotFound                    = 9,    ///< Invalid module index specified.
    Error_SymbolModuleNotLoaded             = 10,   ///< Couldn't find or load the symbols for the specified module.
    Error_SymbolNotFound                    = 11,   ///< Couldn't find the specified symbol.
    Error_SymbolMemoryNotFound              = 12,   ///< Couldn't find the specified memory. This can occur with "quick" coredump files.
    Error_SymbolBufferTooSmall              = 13,   ///< The specified buffer is too small.
    Error_MemoryNotFound                    = 14,   ///< Couldn't find the specified memory.
    Error_InternalError                     = 15,   ///< An "internal error" occurred.
    Error_UnableToAllocateMemory            = 16,   ///< Unable to allocate memory for an operation.
    Error_InvalidAddress                    = 17,   ///< An invalid address was used when querying for source information.
    Error_BufferTooSmall                    = 18,   ///< The supplied buffer is too small.
    Error_InvalidParameter                  = 19,   ///< One or more parameters is invalid.
};

/** @brief Exception codes.*/
enum ExceptionCode : int32_t
{
    ExceptionCode_UndefinedInstruction      = 0,    ///< Undefined instruction.
    ExceptionCode_AccessViolationInstruct   = 1,    ///< Access violation.
    ExceptionCode_AccessViolationData       = 2,    ///< Access data violation.
    ExceptionCode_DatatypeMissaligned       = 3,    ///< Datatype misalignment.
    ExceptionCode_AttachBreak               = 4,    ///< Attach break.
    ExceptionCode_Breakpoint                = 5,    ///< Breakpoint.
    ExceptionCode_UserBreak                 = 6,    ///< User break.
    ExceptionCode_DebuggerBreak             = 7,    ///< Debugger break.
    ExceptionCode_UndefinedSystemCall       = 8,    ///< Undefined system call.
    ExceptionCode_MemorySystemError         = 9,    ///< Memory system error.
    ExceptionCode_StackOverflow             = 201,  ///< Stack overflow.
};

/** @brief Contains thread information.*/
struct Thread
{
    Thread()
    {
        memset( this, 0, sizeof( Thread ) );
    }
    ThreadId    id;                     ///< The unique thread id for the target thread.
    int8_t      core;                   ///< Current core running the thread.
    char        name[ThreadNameSize];   ///< Name of the thread.
};

/** @brief Contains module information.*/
struct Module
{
    Module()
    {
        memset( this, 0, sizeof( Module ) );
    }

    char        filePath[FilePathSize]; ///< Null-terminated, UTF-8 file name and path.
    char        id[ModuleIdSize];       ///< A build generated identifier for the module.
    uint64_t    loadAddress;            ///< The module load address in target memory.
    uint64_t    size;                   ///< Size in bytes of the module.
    bool        isSymbolsLoaded;        ///< True if debug symbols have been loaded for the module.
};

/** @brief Contains information for a call stack entry.*/
struct CallStackEntry
{
    CallStackEntry()
    {
        memset( this, 0, sizeof( CallStackEntry ) );
    }
    char        function[FunctionNameSize]; ///< Function name.
    char        source[FilePathSize];       ///< Source file name.
    int32_t     moduleIndex;                ///< Module index. More information about the module can be obtained through GetModules().
    int32_t     lineNumber;                 ///< Line number.
    uint64_t    instructionAddress;         ///< Instruction pointer address.
    bool        isSymbolsLoaded;            ///< True if debug symbols were loaded for the module for this entry.
};

/** @brief A container for storing a null-terminated, UTF-8 file path.*/
struct FilePath
{
    FilePath()
    {
        memset( this, 0, sizeof( FilePath ) );
    }
    char        filePath[FilePathSize];
};

/** @brief A container for storing the exception code and description.*/
struct Exception
{
    Exception()
    {
        memset( this, 0, sizeof( Exception ) );
    }
    char            description[ExceptionBufferSize];
    ExceptionCode   code;
};

//------------------------------------------------------------------------------
/**
* @brief        Opens the specified core dump file.
* @param[in]    pDumpFilePath File path to the nxdmp file.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error Open( const char* pDumpFilePath );

//------------------------------------------------------------------------------
/**
* @brief        Closes the core dump file, if a file is open.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error Close( void );

//------------------------------------------------------------------------------
/**
* @brief        Adds the specified path to the list of known symbol paths.
* @param[in]    pPath Path to one or more of the symbol files.
* @details      Searches the path for module symbols and adds any located
*               symbols to the list of known symbols. AddSymbolPath can be
*               called before or after Open. Cached symbol paths can be used
*               for multiple NXDMP files. Use ClearSymbolPaths to clear this
*               cache.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error AddSymbolPath( const char* pPath );

//------------------------------------------------------------------------------
/**
* @brief        Adds the specified Symbol Store path to the list of known
*               symbol paths.
* @param[in]    pPath Path to symbol store.
* @details      NX Symbol Store is a tool for archiving debug symbols.
*               AddSymbolStorePath() can be used to link archived debug
*               symbols to Core Dump Report. AddSymbolStorePath() searches
*               through the specified path and symbol store cache and adds any
*               symbols to the list of known symbols. AddSymbolStorePath can be
*               called before or after Open. Cached symbols can be used for
*               multiple NXDMP files. Use ClearSymbolPaths to clear this cache.
*               See @confluencelink{279139350, NXSymStore document}
*               for details about NX Symbol Store.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error AddSymbolStorePath( const char* pPath );

//------------------------------------------------------------------------------
/**
* @brief        Clears the list of known symbol paths.
* @details      Clears all paths specified by AddSymbolPath() and
*               AddSymbolStorePath().
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error ClearSymbolPaths( void );

//------------------------------------------------------------------------------
/**
* @brief        Gets the number of threads from the nxdmp file.
* @param[out]   pOutThreadCount Number of threads.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetThreadCount( int32_t* pOutThreadCount );

//------------------------------------------------------------------------------
/**
* @brief        Acquires an array of thread information.
* @param[out]   pOutThreadInfos An array of thread information.
* @param[out]   pOutBufferSize The size, in number of elements, of the returned
*               preallocated buffer.
* @param[in]    bufferSize The size, in number of elements, of the client
*               preallocated buffer.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetThreads( Thread* pOutThreadInfos, int32_t* pOutBufferSize, int32_t bufferSize );

//------------------------------------------------------------------------------
/**
* @brief        Gets the number of modules from the nxdmp file.
* @param[out]   pOutModuleCount The number of modules.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetModuleCount( int32_t* pOutModuleCount );

//------------------------------------------------------------------------------
/**
* @brief        Acquires an array of module information.
* @param[out]   pOutModuleInfos Contains module information, such as the module
*               name and load address.
* @param[out]   pOutBufferSize The size, in number of elements, of the returned
*               preallocated buffer.
* @param[in]    bufferSize The size, in number of elements, of the client
*               preallocated buffer.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetModules( Module* pOutModuleInfos, int32_t* pOutBufferSize, int32_t bufferSize );

//------------------------------------------------------------------------------
/**
* @brief        Gets the number of call stack entries for the specified thread.
* @param[out]   pOutCallStackEntryCount Number of call stack entries.
* @param[in]    threadId The desired thread id. The thread id can be found within
*               the nn::cdr::Thread struct.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetCallStackEntryCount( int32_t* pOutCallStackEntryCount, ThreadId threadId );

//------------------------------------------------------------------------------
/**
* @brief        Acquires an array of call stack entries.
* @param[out]   pOutEntries The call stack entry, which includes data such as the
*               function name, line number, and instruction pointer address.
* @param[out]   pOutBufferSize The size, in number of elements, of the returned
*               preallocated buffer.
* @param[in]    bufferSize The size, in number of elements, of the client
*               preallocated buffer.
* @param[in]    threadId The desired thread id. The thread id can be found
*               within the nn::cdr::Thread struct.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetCallStackEntries( CallStackEntry* pOutEntries, int32_t* pOutBufferSize, int32_t bufferSize, ThreadId threadId );

//------------------------------------------------------------------------------
/**
* @brief        Gets the file name and line number for the specified address.
* @param[out]   pOutFilePath The preallocated buffer to store the file path.
* @param[out]   pOutLineNumber The line number that matches the specified address.
* @param[in]    address The address to convert to a file path and line number.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetSource( FilePath* pOutFilePath, int* pOutLineNumber, uint64_t address );

//------------------------------------------------------------------------------
/**
* @brief        Gets the exception code and description.
* @param[out]   pOutException The preallocated buffer to store the exception
*               description and code.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetException( Exception* pOutException );

//==============================================================================

}

}
