﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @namespace   nn::tm
 * @brief       Target Manager API (TMAPI) public interfaces.
 * @details     Interface for adding, removing, and controlling targets in Nintendo
 *              Target Manager.
 *
 * Connections to Target Manager do not require a connect or attach call.
 * Any function call through this API will automatically establish a connection
 * to Target Manager. If a connection could not be made, an error will be
 * returned.
 *
 * Regarding functions which retrieve variable length arrays of data into
 * buffers provided by the caller...
 *
 * Such functions allow the caller to provide a buffer into which results will
 * be placed and a value which indicates the number of elements in the buffer.
 *
 * If the number of results will fit within the given buffer, then the function
 * will return a successful error code (assuming nothing else goes wrong) and
 * the parameter indicating the size of the buffer will be changed to the
 * number of elements which are returned.
 *
 * If the number of results will NOT fit within the given buffer, the buffer
 * will be filled up to the specified buffer size.
 *
 * For example:
 *
 *    // Get the number of targets.
 *    int32_t targetCount = 0;
 *    nn::tm::Error error = nn::tm::GetTargetCount( &targetCount );
 *
 *    if( error == nn::tm::Error_Ok )
 *    {
 *        nn::tm::TargetHandle* pTargetHandles = new nn::tm::TargetHandle[targetCount];
 *        error = nn::tm::GetTargets( pTargetHandles, &targetCount, targetCount );
 *
 *        // Get the target names.
 *        if( error == nn::tm::Error_Ok )
 *        {
 *            for( int32_t iTarget = 0; iTarget < targetCount; iTarget++ )
 *            {
 *                TargetName nameBuffer = {0};
 *                error = nn::tm::GetTargetName( &nameBuffer, pTargetHandles[iTarget] );
 *
 *                // Do something with nameBuffer.
 *            }
 *        }
 *
 *        delete[] pTargetHandles;
 *    }
 *    else
 *    {
 *        // Handle error.
 *    }
 *
 * Why did we structure this interface this way?  Some of these functions can
 * be transactionally time consuming.  If you already have a sufficiently large
 * buffer, then we are able to get results in a single transaction.  This is
 * more efficient than a traditional GetCount(N) then GetStuff(N) pairing which
 * always requires two transactions.
 */

#pragma once

//==============================================================================
//  INCLUDES
//==============================================================================

#include <stdint.h>

// Typedef for Windows event handles.
typedef void* HANDLE;

//==============================================================================

namespace nn {

namespace tm {

//------------------------------------------------------------------------------

/** @brief Default value for invalid target and discovery handles.*/
const uint64_t InvalidHandle = (uint64_t) - 1;

/** @brief The maximum IP address length, including the null terminator. Only IPv4 addresses and port numbers are supported.*/
const int32_t HtcsIpAndPortAddressBufferLength = 128;

/** @brief Maximum length of a HTCS peer name, including the null terminator*/
const int32_t HtcsPeerNameBufferLength = 32;

/** @brief Maximum length of a HTCS port name, including the null terminator.*/
const int32_t HtcsPortNameBufferLength = 32;

/** @brief Maximum length of a target name, including the null terminator.*/
const int32_t TargetNameBufferLength = 64;

/** @brief Maximum length of a target ip address, including the null terminator.*/
const int32_t TargetIpAddressBufferLength = 128;

/** @brief Maximum length of a target serial number, including the null terminator.*/
const int32_t TargetSerialNumberBufferLength = 24;

/** @brief Maximum length of a target mac address, including the null terminator.*/
const int32_t TargetMacAddressBufferLength = 32;

/** @brief Maximum length of a program name, including the null terminator.*/
const int32_t TargetProgramNameBufferLength = 256;

/** @brief Maximum length of a working directory, including the null terminator.*/
const int32_t TargetWorkingDirectoryBufferLength = 512;

/** @brief Maximum length of a nn::tm::Error text string.*/
const int32_t ErrorTextBufferLength = 1024;

/** @brief Discovery scan states in Target Manager. Only one discovery scan may be running at a given time.*/
enum DiscoveryState : int32_t
{
    DiscoveryState_Uninitialized = 0, ///< A discovery scan has not been executed. No targets have been discovered.
    DiscoveryState_Started       = 1, ///< A discovery scan is currently in progress.
    DiscoveryState_Complete      = 2, ///< A discovery scan is not in progress. The previous discovery scan was completed or aborted.
};

/** @brief Hardware power states for targets.*/
enum TargetPowerState : int32_t
{
    TargetPowerState_Unknown = 0, ///< The power state of the target is unknown.
    TargetPowerState_Off     = 1, ///< The target has been powered off.
    TargetPowerState_On      = 2, ///< The target is powered on.
    TargetPowerState_Asleep  = 3, ///< The target is asleep.
};

/** @brief Program states on a target.*/
enum TargetProgramState : int32_t
{
    TargetProgramState_NoProgram = -1, ///< No program is currently loaded onto the target.
    TargetProgramState_Loading   =  0, ///< The program is being loaded onto the target.
    TargetProgramState_Loaded    =  1, ///< The program has been loaded onto the target. This program has not been started.
    TargetProgramState_Running   =  2, ///< The program has been loaded and is running.
    TargetProgramState_Halted    =  3, ///< The program has been halted.
};

/** @brief Used to specify how the target is connected to the host.*/
enum TargetConnectionType : int32_t
{
    TargetConnectionType_Unknown  = 0, ///< The target connection type is unknown.
    TargetConnectionType_Usb      = 1, ///< The target is connected via a USB connection.
    TargetConnectionType_Ethernet = 2, ///< The target is connected via an ethernet connection.
    TargetConnectionType_Future   = 3, ///< Reserved for future use.
};

/** @brief Used to specify the hardware type of the target.*/
enum TargetHardwareType : int32_t
{
    TargetHardwareType_Unknown  = 0, ///< The target type is unknown.
    TargetHardwareType_Edev     = 1, ///< The target type is an EDEV.
    TargetHardwareType_Sdev     = 2, ///< The target type is an SDEV.
    TargetHardwareType_Sim      = 3, ///< The target type is a simulated target.
    TargetHardwareType_Future   = 4, ///< Reserved for future use.
};

/** @brief Used to specify the connection status of the target.*/
enum TargetStatus : int32_t
{
    TargetStatus_Unknown       = 0, ///< The target status is unknown.
    TargetStatus_Disconnected  = 1, ///< The target is disconnected from Target Manager.
    TargetStatus_Disconnecting = 2, ///< The target is being disconnected from Target Manager.
    TargetStatus_Connected     = 3, ///< The target is connected to Target Manager.
    TargetStatus_Connecting    = 4, ///< The target is being connected to Target Manager.
    TargetStatus_Asleep        = 5, ///< The target is asleep.
    TargetStatus_PoweringOn    = 6, ///< The target is powering on.
    TargetStatus_PoweringOff   = 7, ///< The target is powering off.
};

/** @brief Target Manager API error codes.*/
enum Error : int32_t
{
    Error_Ok                                    = 0,  ///< No error.
    Error_Unknown                               = 2,  ///< Unknown error.
    Error_Timeout                               = 3,  ///< Timeout.
    Error_NoConnection                          = 4,  ///< No connection to Target Manager service.
    Error_IncompatibleProtocol                  = 5,  ///< Target Manager protocol version incompatible.
    Error_InvalidOpcode                         = 6,  ///< Invalid protocol opcode.
    Error_TargetBusy                            = 7,  ///< The requested target is busy.
    Error_TargetUnavailable                     = 8,  ///< Active target no longer available.
    Error_InvalidAddress                        = 9,  ///< Invalid memory address.
    Error_ExecutableNotFound                    = 10, ///< Requested executable file was not found.
    Error_InvalidTarget                         = 11, ///< Invalid target specified.
    Error_InvalidResponse                       = 13, ///< Invalid value returned.
    Error_TargetDuplicated                      = 14, ///< Target already exists.
    Error_InvalidParameter                      = 15, ///< One of the parameter(s) is invalid.
    Error_InvalidDeviceHandle                   = 16, ///< The HID handle is invalid.
    Error_InvalidController                     = 17, ///< The controller index is invalid.
    Error_ExecutableNotCompatible               = 18, ///< The executable file is not compatible with the hardware.
    Error_BufferTooSmall                        = 19, ///< A provided buffer is too small.
    Error_InternalError                         = 20, ///< An "internal error" occurred.
    Error_Cancelled                             = 21, ///< Operation cancelled.
    Error_ConnectVersionError                   = 22, ///< Incorrect version connection problem
    Error_ConnectFailed                         = 23, ///< Connect failure due to unknown reason.
    Error_NotImplemented                        = 24, ///< An API or feature is not implemented.
    Error_IncompatibleTmApiVersion              = 25, ///< The client and Target Manager API versions are not compatible.
    Error_DebuggingUnavailable                  = 26, ///< Debugging service isn't available.
    Error_NotBackwardsCompatible                = 27, ///< The packet failed to send because the specified task is not backwards compatible an older target  i.e. the target is not up to date with this version of Target Manager.
    Error_ServiceUnavailable                    = 28, ///< The service that handles this isn't available.
    Error_TargetDiscoveryAlreadyStarted         = 29, ///< Target discovery has already started.
    Error_ConnectFailedLogServer                = 30, ///< The connect failed because log service is unavailable
    Error_TargetIoSearchComplete                = 31, ///< The target Io search is finished  we have found all there is to find.
    Error_TargetIoInsufficientSpace             = 32, ///< Not enough space on the memory card.
    Error_TargetIoNoMemoryCard                  = 33, ///< No memory card inserted in the device.
    Error_TargetIoPathAlreadyExists             = 34, ///< Path already exists on the memory card.
    Error_TargetIoMemoryCardError               = 35, ///< Problem reading the memory card.
    Error_TargetAsleep                          = 36, ///< Target is asleep.
    Error_NoProgramLoaded                       = 37, ///< No program is loaded to the target.
    Error_IncompatibleTargetManager             = 38, ///< This function is not implemented in the detected version of Target Manager. This can occur if client code is attempting to call a new function on an older version of Targer Manager.
    Error_TargetNotConnected                    = 39, ///< Requested action requires a connection to the target.
    Error_UnableToCreateCoredump                = 40, ///< Unable to create coredump file.
    Error_InvalidHtcsAddress                    = 44, ///< An invalid HTCS address was requested. HTCS addresses must be of the form "xxx.xxx.xxx.xxx:yyyyy".
    Error_DuplicateHtcsAddress                  = 45, ///< An HTCS reservation was requested that conflicts with an existing HTCS reservation with the same ip address and port number or the same server and port names.
    Error_HtcsAddressNotFound                   = 46, ///< No HTCS entry exists with the specified address.
    Error_OtherUser                             = 47, ///< The target is in use by another user.
    Error_InvalidScreenshotFolderSetting        = 48, ///< Invalid screenshot folder setting. Check your screenshot settings in Options.
    Error_InvalidScreenshotFilenameSetting      = 49, ///< Invalid screenshot filename setting. Check your screenshot settings in Options.
    Error_HtcsEntryNotFound                     = 50, ///< No HTCS entry exists with the specified peer and port name.
    Error_HtcsEventHandleAlreadyRegistered      = 51, ///< The specified event handle for HTCS updates has already been registered.
    Error_HtcsEventHandleNotFound               = 52, ///< The specified event handle for HTCS updates has not been registered.
    Error_InvalidTargetName                     = 53, ///< Invalid target name.
    Error_ProgramEventHandleAlreadyRegistered   = 54, ///< Program event handle already registered.
    Error_InvalidWorkingDirectory               = 55, ///< Invalid working directory specified.
    Error_ProgramStartFailed                    = 101, ///< Failed to StartProgram.
    Error_ProgramNotLoaded                      = 102, ///< Data has not finished loading.
    Error_ProgramInvalidArguments               = 103, ///< Invalid program arguments.
    Error_ProgramCreateFailed                   = 104, ///< Failed to create a program with path.
    Error_ProgramIsRunning                      = 105, ///< Program already running.
    Error_ProgramNotRunning                     = 106, ///< Desired program is not running.
    Error_ProgramNotHalted                      = 107, ///< Program not halted.
    Error_ProgramAlreadyStarted                 = 108, ///< Program already started.
    Error_ProgramHalted                         = 109, ///< Program halted.
    Error_ProgramUnknownRegister                = 110, ///< Unknown register.
    Error_ProgramAttachFailed                   = 111, ///< Failed to attach to a program by program Id.
    Error_ProgramNotStarted                     = 112, ///< Program hasn't been started yet.
    Error_ProgramDebuggingDisabled              = 113, ///< Debugging is disabled for this program.
    Error_ProgramPackageIncomplete              = 114, ///< A file is missing from the EXE.
};

/** @brief Type of dump file to create.*/
enum DumpType : int32_t
{
    DumpType_Quick = 0,  ///< Quick dump.
    DumpType_Full  = 1,  ///< Full dump.
};

/** @brief A unique handle for interacting with targets.*/
struct TargetHandle
{
    uint64_t handle;

    bool operator==( const TargetHandle& rhs ) const
    {
        return (handle == rhs.handle);
    }
};

/** @brief A unique handle for interacting with discovered targets.*/
struct DiscoveryHandle
{
    uint64_t handle;

    bool operator==( const DiscoveryHandle& rhs ) const
    {
        return (handle == rhs.handle);
    }
};

/** @brief The null-terminated ip address and port number for HTCS. The expected
 *  format is "xxx.xxx.xxx.xxx:yyyyy". For example, if the desired ip address is
 *  "127.0.0.1" and the port number is "12345", then the HtcsAddress would be
 *  "127.0.0.1:12345".*/
struct HtcsAddress
{
    char ipAndPortAddress[HtcsIpAndPortAddressBufferLength];
};

/** @brief The null-terminated peer name of the HTCS address.*/
struct HtcsPeerName
{
    char name[HtcsPeerNameBufferLength];
};

/** @brief The null-terminated port name of the HTCS address.*/
struct HtcsPortName
{
    char name[HtcsPortNameBufferLength];
};

/** @brief An HtcsPeerName that is used to specify the host PC name.*/
const HtcsPeerName PeerNameHostPc = HtcsPeerName{ "" };

/** @brief An HtcsPeerName that is used to specify an arbitrary peer name.*/
const HtcsPeerName PeerNameAny = HtcsPeerName{ "" };

/** @brief An HtcsPortName that is used to specify an arbitrary port name.*/
const HtcsPortName PortNameAny = HtcsPortName{ "" };

/** @brief The null-terminated target name*/
struct TargetName
{
    char name[TargetNameBufferLength];
};

/** @brief The null-terminated user connected to the target. The user will be ""
 *  if no user is connected. The user will be "Unknown" the user status has not
 *  been queried for the target.
 */
struct TargetUser
{
    char user[TargetNameBufferLength];
};

/** @brief The null-terminated target ip address.*/
struct TargetIpAddress
{
    char ipAddress[TargetIpAddressBufferLength];
};

/** @brief The null-terminated target serial number.*/
struct TargetSerialNumber
{
    char serialNumber[TargetSerialNumberBufferLength];
};

/** @brief The null-terminated target mac address.*/
struct TargetMacAddress
{
    char macAddress[TargetMacAddressBufferLength];
};

/** @brief The null-terminated target serial number.*/
struct TargetProgramName
{
    char name[TargetProgramNameBufferLength];
};

/** @brief The struct for containing the target agent version.*/
struct TargetAgentVersion
{
    int32_t major;
    int32_t minor;
    int32_t micro;
    int32_t revision;
};

/** @brief The struct for containing the target working directory.*/
struct TargetWorkingDirectory
{
    char directory[TargetWorkingDirectoryBufferLength];
};

/** @brief The null-terminated nn::tm::Error text buffer.*/
struct ErrorText
{
    char text[ErrorTextBufferLength];
};

//------------------------------------------------------------------------------
// Target inventory.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Returns the number of targets in Target Manager. All targets
 *              that have been added to Target Manager will be counted. This
 *              includes targets that are asleep, powered off, or have an
 *              unknown status.
 *
 * @param[out]  pOutTargetCount The number of targets.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     The returned number of targets can be used to determine the
 *              proper buffer size for GetTargets().
 */
Error           GetTargetCount( int32_t* pOutTargetCount );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the array of targets in Target Manager. All targets that
 *              have been added to Target Manager will be returned. This
 *              includes targets that are asleep, powered off, or have an
 *              unknown status.
 *
 * @param[out]  pOutTargetsBuffer A client preallocated buffer for storing
 *              TargetHandles.
 * @param[out]  pOutBufferSize The size, in number of elements, of the returned
 *              preallocated buffer.
 * @param[in]   bufferSize The size, in number of elements, of the client
 *              preallocated buffer.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     The returned TargetHandles can be used to acquire additional
 *              Target information, such as GetTargetName( targetHandle ).
 */
Error           GetTargets( TargetHandle* pOutTargetsBuffer,
                            int32_t*      pOutBufferSize,
                            int32_t       bufferSize );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the TargetHandle of the default target in Target
 *              Manager.
 *
 * @param[out]  pOutTargetHandle The TargetHandle of the current default target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     The default target will be the target used when launching
 *              applications from Visual Studio.
 */
Error           GetDefaultTarget( TargetHandle* pOutTargetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Sets the default target in Target Manager to the target
 *              specified by the TargetHandle.
 *
 * @param[in]   targetHandle The TargetHandle of the desired default target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     The default target will be the target used when launching
 *              applications from Visual Studio.
 */
Error           SetDefaultTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
// Discovery.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Starts a discovery scan.
 *
 * @param[in]   msTimeout The amount of time in milliseconds before the
 *              discovery scan is completed.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           StartDiscovery( int32_t msTimeout );

//------------------------------------------------------------------------------
/**
 * @brief       Aborts the current discovery scan.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     If a discovery scan is not in progress, nn::tm::Error_Ok is
 *              returned.
 */
Error           AbortDiscovery( void );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the current state of the discovery scan.
 *
 * @param[out]  pOutDiscoveryState The current state of the discovery scan.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     If a scan is aborted with AbortDiscovery(), the returned
 *              state will be DiscoveryState_Complete.
 */
Error           GetDiscoveryState( DiscoveryState* pOutDiscoveryState );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the number of discovered targets from the last executed
 *              discovery scan.
 *
 * @param[out]  pOutDiscoveryTargetCount The number of discovered targets.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetCount( int32_t* pOutDiscoveryTargetCount );

//------------------------------------------------------------------------------
/**
 * @brief       Returns a buffer containing DiscoveryHandles.
 *
 * @param[out]  pOutDiscoveredTargetsBuffer The client preallocated buffer for
 *              storing DiscoveryHandles.
 * @param[out]  pOutBufferSize The size, in number of elements, of the returned
 *              preallocated buffer.
 * @param[in]   bufferSize The size, in number of elements, of the client
 *              preallocated buffer.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargets( DiscoveryHandle*       pOutDiscoveredTargetsBuffer,
                                     int32_t*               pOutBufferSize,
                                     int32_t                bufferSize );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the name of the discovered target specified by the
 *              DiscoveryHandle.
 *
 * @param[out]  pOutTargetName The client preallocated buffer to store the
 *              target name. The returned buffer is null-terminated.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetName( TargetName*     pOutTargetName,
                                        DiscoveryHandle discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the ip address of the discovered target specified by the
 *              DiscoveryHandle.
 *
 * @param[out]  pOutTargetIpAddress The client preallocated buffer to store the
 *              ip address. The returned buffer is null-terminated.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetIpAddress( TargetIpAddress* pOutTargetIpAddress,
                                             DiscoveryHandle  discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the serial number of the discovered target specified by the
 *              DiscoveryHandle.
 *
 * @param[out]  pOutTargetSerialNumber The client preallocated buffer to store
 *              the serial number. The returned buffer is null-terminated.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetSerialNumber( TargetSerialNumber* pOutTargetSerialNumber,
                                                DiscoveryHandle     discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the mac address of the discovered target specified by the
 *              DiscoveryHandle.
 *
 * @param[out]  pOutTargetMacAddress The client preallocated buffer to store
 *              the mac address. The returned buffer is null-terminated.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetMacAddress( TargetMacAddress*    pOutTargetMacAddress,
                                              DiscoveryHandle      discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the hardware type of the discovered target specified by the
 *              DiscoveryHandle.
 *
 * @param[out]  pOutTargetHardwareType The hardware type of the target.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetHardwareType( TargetHardwareType* pOutTargetHardwareType,
                                                DiscoveryHandle     discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the connection type of the discovered target specified by
 *              the DiscoveryHandle.
 *
 * @param[out]  pOutTargetConnectionType The connection type of the hardware,
 *              such as USB or ethernet.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetDiscoveryTargetConnectionType( TargetConnectionType* pOutTargetConnectionType,
                                                  DiscoveryHandle       discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the agent version of the discovered target specified by
 *              the DiscoveryHandle. Only valid for EDEV targets. Attempting to
 *              issue this command to a discovered SDEV target will return the
 *              Error_TargetNotConnected error.
 *
 * @param[out]  pOutTargetAgentVersion The target major, minor, micro, and
 *              revision versions.
 * @param[in]   discoveryHandle The handle of the discovered target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code. If called on
 *              a discovered SDEV target, the Error_TargetNotConnected error will
 *              be returned. To determine the SDEV target agent version, connect
 *              to the target and use GetTargetAgentVersion().
 */
Error           GetDiscoveryTargetAgentVersion( TargetAgentVersion* pOutTargetAgentVersion,
                                                DiscoveryHandle     discoveryHandle );

//------------------------------------------------------------------------------
// Target additions and removals.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Adds a target specified by the ip address to Target Manager.
 *
 * @param[out]  pOutTargetHandle The TargetHandle of the newly added target.
 * @param[in]   pIpAddress The null-terminated ip address of the target to be
 *              added.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     The target specified by the ip address is not required to be
 *              present or powered on to be added to Target Manager.
 */
Error           AddTarget(       TargetHandle*    pOutTargetHandle,
                           const TargetIpAddress* pIpAddress );

//------------------------------------------------------------------------------
/**
 * @brief       Adds a target specified by the DiscoveryHandle to Target
 *              Manager.
 *
 * @param[out]  pOutTargetHandle The TargetHandle of the newly added target.
 * @param[in]   discoveryHandle The discovery handle of the target to be added.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     To determine the discoveryHandle of the target to be added,
 *              use TmDiscoveryGetTargets(). Additional discovered target
 *              information can be acquired with the GetDiscoveryTarget
 *              methods.
 */
Error           AddTarget( TargetHandle*   pOutTargetHandle,
                           DiscoveryHandle discoveryHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Removes the target specified by the TargetHandle from Target
 *              Manager.
 *
 * @param[in]   targetHandle The handle of the target to be removed.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           RemoveTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Removes all targets from Target Manager.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           RemoveAllTargets( void );

//------------------------------------------------------------------------------
// Target control.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Connects to the target specified by the TargetHandle.
 *
 * @param[in]   targetHandle The handle of the target to be connected.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           ConnectTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Disconnects from the target specified by the TargetHandle.
 *
 * @param[in]   targetHandle The handle of the target to be disconnected.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           DisconnectTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Powers on the target specified by the TargetHandle.
 *
 * @param[in]   targetHandle The handle of the target to be powered on.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 *
 * @details     This control is only permitted for SDEV targets.
 */
Error           PowerOnTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Powers off the target specified by the TargetHandle.
 *
 * @param[in]   targetHandle The handle of the target to be powered off.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           PowerOffTarget( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Resets the target specified by the TargetHandle. Target
 *              Manager must be connected to the target to issue this command.
 *
 * @param[in]   targetHandle The handle of the target to be reset.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           ResetTargetSoft( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Resets the target specified by the TargetHandle. Hard resets on
 *              SDEV targets do not require a connection to the target. Any
 *              existing connection to the SDEV target will be severed. Hard
 *              resets on EDEV targets require a connection to the target.
 *
 * @param[in]   targetHandle The handle of the target to be reset.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           ResetTargetHard( TargetHandle targetHandle );

//------------------------------------------------------------------------------
// Target attributes.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Returns the target name of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetName A client preallocated buffer to contain the name
 *              of the requested target. The returned buffer is null-terminated.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetName( TargetName*  pOutTargetName,
                               TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the user connected to the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetUser A client preallocated buffer to contain the user
 *              of the requested target. The user will be "" if no user is
 *              connected. The user will be "Unknown" the user status has not
 *              been queried for the target.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetUser( TargetUser*  pOutTargetUser,
                               TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the target ip address of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetIpAddress A client preallocated buffer to contain the
 *              ip address of the requested target. The returned buffer is
 *              null-terminated.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetIpAddress( TargetIpAddress* pOutTargetIpAddress,
                                    TargetHandle     targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the target serial number of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetSerialNumber A client preallocated buffer to contain
 *              the serial number of the requested target. The returned buffer
 *              is null-terminated.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetSerialNumber( TargetSerialNumber* pOutTargetSerialNumber,
                                       TargetHandle        targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the mac address of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetMacAddress A client preallocated buffer to contain
 *              the mac address of the requested target. The returned buffer
 *              is null-terminated.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetMacAddress( TargetMacAddress* pOutTargetMacAddress,
                                     TargetHandle      targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the target connection status of the target specified by
 *              the TargetHandle.
 *
 * @param[out]  pOutTargetStatus The target status, such as connected,
 *              disconnected, or asleep.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetStatus( TargetStatus* pOutTargetStatus,
                                 TargetHandle  targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the target connection type of the target specified by
 *              the TargetHandle.
 *
 * @param[out]  pOutTargetConnectionType The connection type of the hardware,
 *              such as USB or ethernet.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetConnectionType( TargetConnectionType* pOutTargetConnectionType,
                                         TargetHandle          targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the hardware type of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetHardwareType The hardware type of the target.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetHardwareType( TargetHardwareType* pOutTargetHardwareType,
                                       TargetHandle        targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the agent version of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutTargetAgentVersion The target major, minor, micro, and
 *              revision versions.
 * @param[in]   targetHandle The handle of the requested target.
 * @details     SDEV targets require a one-time connection to Target Manager to
 *              acquire the agent version. If an SDEV target has not been
 *              connected to Target Manager, GetTargetAgentVersion will return
 *              nn::tm::Error_TargetNotConnected.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetAgentVersion( TargetAgentVersion* pOutTargetAgentVersion,
                                       TargetHandle        targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns target sleep status of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutIsSleeping true if the target is asleep, false otherwise,
 *              for the requested target.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetIsSleeping( bool*        pOutIsSleeping,
                                     TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns target connection status of the target specified by the
 *              TargetHandle.
 *
 * @param[out]  pOutIsConnected Returns true if the target is connected for the
 *              requested target.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetIsConnected( bool*        pOutIsConnected,
                                      TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Sets the name of the target specified by the TargetHandle.
 *
 * @param[in]   targetHandle The handle of the requested target.
 * @param[in]   pTargetName The desired name of the target. pTargetName is a
 *              UTF-8 null-terminated string.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           SetTargetName(       TargetHandle targetHandle,
                               const TargetName*  pTargetName );

//------------------------------------------------------------------------------
/**
 * @brief       Sets the dock state of the target specified by the
 *              TargetHandle. This function is only supported by SDEV targets.
 *
 * @param[in]   targetHandle The handle of the requested target.
 * @param[in]   bDocked The requested dock state of the target: true for
 *              docked, false for undocked.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           SetTargetDockState( TargetHandle targetHandle,
                                    bool         bDocked );

//------------------------------------------------------------------------------
/**
* @brief        Gets the working directory of the target specified by the
*               TargetHandle.
*
* @param[out]   pOutWorkingDirectory The working directory for the target.
* @param[in]    targetHandle The handle of the requested target.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           GetTargetWorkingDirectory( TargetWorkingDirectory*   pOutWorkingDirectory,
                                           TargetHandle              targetHandle );

//------------------------------------------------------------------------------
/**
* @brief        Sets the working directory of the target specified by the
*               TargetHandle.
*
* @param[in]    targetHandle The handle of the requested target.
* @param[in]    pWorkingDirectory The working directory for the target.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           SetTargetWorkingDirectory( TargetHandle              targetHandle,
                                           TargetWorkingDirectory*   pWorkingDirectory );

//------------------------------------------------------------------------------
// Program control and attributes.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Launches an executable on the target specified by the
 *              TargetHandle.
 *
 * @param[in]   targetHandle The handle of the requested target.
 * @param[in]   pFileName The absolute null-terminated path to the executable.
 * @param[in]   pArguments Optional. The null-terminated arguments to be passed
 *              to the executable.
 * @param[in]   pWorkingDirectory Optional. The null-terminated working
 *              directory of the executable. If not specified, the target
 *              working directory will be used.
 * @param[in]   pTDF Optional. The absolute null-terminated path to the target
 *              environment variables definitions file.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           LaunchTargetProgram(       TargetHandle targetHandle,
                                     const char*        pFileName,
                                     const char*        pArguments,
                                     const char*        pWorkingDirectory,
                                     const char*        pTDF );

//------------------------------------------------------------------------------
/**
 * @brief       Kill the user program on the target specified by the
 *              TargetHandle.
 *
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           KillTargetProgram( TargetHandle targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the name of the program on the target specified by
 *              the TargetHandle.
 *
 * @param[out]  pOutTargetProgramName The client preallocated buffer to store
 *              the name of the currently active program. The returned buffer is
 *              null-terminated. Null is returned if no programs are currently
 *              running.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetProgramName( TargetProgramName* pOutTargetProgramName,
                                      TargetHandle       targetHandle );

//------------------------------------------------------------------------------
/**
 * @brief       Returns the state of the program on the target specified by
 *              the TargetHandle.
 *
 * @param[out]  pOutTargetProgramState The state of the currently active
 *              program.
 * @param[in]   targetHandle The handle of the requested target.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetTargetProgramState( TargetProgramState* pOutTargetProgramState,
                                       TargetHandle        targetHandle );

//------------------------------------------------------------------------------
/**
* @brief        Creates a nxdmp file for the application on the target specified
*               by the TargetHandle.
*
* @param[in]    targetHandle The handle of the requested target.
* @param[in]    pFilePath The requested output path for the nxdmp file.
* @param[in]    dumpType The type of dump file to create.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           CreateTargetProgramDump( TargetHandle   targetHandle,
                                         const char*    pFilePath,
                                         DumpType       dumpType );

//------------------------------------------------------------------------------
/**
 * @brief       Registers an EVENT which will be signaled when running program exits.
 *
 * @param[out]  eventHandle A handle that is created by CreateEvent.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           RegisterProgramExitEvent( HANDLE eventHandle );

//------------------------------------------------------------------------------
// HTCS.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Registers an HTCS entry.
 *
 * @param[in]   pPeerName The peer name on the host. To specify the host PC
 *              name, use PeerNameHostPc.
 * @param[in]   pPortName The port name on the host.
 * @param[in]   pSocketAddress The HTCS address that the host application is
 *              listening on.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           RegisterHtcsEntry( const HtcsPeerName* pPeerName,
                                   const HtcsPortName* pPortName,
                                   const HtcsAddress*  pSocketAddress );

//------------------------------------------------------------------------------
/**
 * @brief       Unregisters an HTCS entry.
 *
 * @param[in]   pPeerName The peer name on the host. To specify the host PC
 *              name, use PeerNameHostPc.
 * @param[in]   pPortName The port name on the host.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           UnregisterHtcsEntry( const HtcsPeerName* pPeerName,
                                     const HtcsPortName* pPortName );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the number of the HTCS addresses reserved by Target Manager.
 *
 * @param[out]  pOutHtcsAddressesCount the number of the HTCS ports reserved by
 *              Target Manager.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetHtcsAddressesCount( int32_t* pOutHtcsAddressesCount );

//------------------------------------------------------------------------------
/**
 * @brief       Gets all of the HTCS addresses reserved by Target Manager.
 *
 * @param[out]  pOutHtcsAddresses The client preallocated buffer for storing
 *              the Target Manager HTCS reserved addresses.
 * @param[out]  pOutBufferSize The size, in number of elements, of the returned
 *              preallocated buffer.
 * @param[in]   bufferSize The size, in number of elements, of the client
 *              preallocated buffer.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetHtcsAddresses( HtcsAddress* pOutHtcsAddresses,
                                  int32_t*     pOutBufferSize,
                                  int32_t      bufferSize );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the peer name of the HTCS address.
 *
 * @param[out]  pOutPeerName The client preallocated buffer for storing the
 *              peer name of the HTCS reserved port.
 * @param[in]   pSocketAddress The HtcsAddress of the HTCS entry.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetHtcsPeerName(       HtcsPeerName* pOutPeerName,
                                 const HtcsAddress*  pSocketAddress );

//------------------------------------------------------------------------------
/**
 * @brief       Gets the port name of the HTCS address.
 *
 * @param[out]  pOutPortName The client preallocated buffer for storing the
 *              port name of the HTCS reserved port.
 * @param[in]   pSocketAddress The HtcsAddress of the HTCS entry.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetHtcsPortName(       HtcsPortName* pOutPortName,
                                 const HtcsAddress*  pSocketAddress );

//------------------------------------------------------------------------------
/**
* @brief        Gets the control port number used for HTCS.
*
* @param[out]   pOutControlPort Control port number used for HTCS.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           GetHtcsControlPort( int32_t* pOutControlPort );

//------------------------------------------------------------------------------
/**
* @brief        Gets a filtered list of HtcsAddresses as specified by the peer
*               and port names.
*
* @param[out]   pOutHtcsAddresses A client preallocated buffer for storing
*               IP addresses and ports.
* @param[out]   pOutBufferSize The size, in number of elements, of the returned
*               preallocated buffer.
* @param[in]    bufferSize The size, in number of elements, of the client
*               preallocated buffer.
* @param[in]    pPeerName The peer name used to filter the returned list of
*               HTCS addresses. Use a nullptr or PeerNameAny to not filter on a
*               peer name.
* @param[in]    pPortName The port name used to filter the returned list of
*               HTCS addresses. Use a nullptr or PortNameAny to not filter on a
*               port name.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           GetHtcsEntryEndPoints( HtcsAddress* pOutHtcsAddresses, int32_t* pOutBufferSize, int32_t bufferSize, const HtcsPeerName* pPeerName, const HtcsPortName* pPortName );

//------------------------------------------------------------------------------
/**
* @brief        Specifies an event handle to be signaled when HTCS port mapping
*               is updated.
*
* @param[in]    eventHandle A handle that was created by CreateEvent.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           RegisterHtcsPortMappingUpdateEvent( HANDLE eventHandle );

//------------------------------------------------------------------------------
/**
* @brief        Unregisters an event handle. The event will no longer be
*               signaled when HTCS port mapping is updated.
*
* @param[in]    eventHandle A handle that was created by the CreateEvent.
*
* @result       nn::tm::Error_Ok if successful, else an error code.
*/
Error           UnregisterHtcsPortMappingUpdateEvent( HANDLE eventHandle );

//------------------------------------------------------------------------------
// Error-to-text translation.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/**
 * @brief       Converts the nn::tm::Error code into an text string.  The text
 *              will attempt to use the system language or default to English.
 *
 * @param[out]  pOutTextBuffer The client preallocated buffer for storing the
 *              text string. The returned buffer is null-terminated.
 * @param[in]   error The nn::tm::Error code to be converted into text.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetErrorText( ErrorText* pOutTextBuffer,
                              Error      error );

//------------------------------------------------------------------------------
/**
 * @brief       Converts the nn::tm::Error code into an English text string.
 *
 * @param[out]  pOutTextBuffer The client preallocated buffer for storing the
 *              text string. The returned buffer is null-terminated.
 * @param[in]   error The nn::tm::Error code to be converted into text.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetErrorTextEn( ErrorText* pOutTextBuffer,
                                Error      error );

//------------------------------------------------------------------------------
/**
 * @brief       Converts the nn::tm::Error code into an Japanese text string.
 *
 * @param[out]  pOutTextBuffer The client preallocated buffer for storing the
 *              text string. The returned buffer is null-terminated.
 * @param[in]   error The nn::tm::Error code to be converted into text.
 *
 * @result      nn::tm::Error_Ok if successful, else an error code.
 */
Error           GetErrorTextJa( ErrorText* pOutTextBuffer,
                                Error      error );

//------------------------------------------------------------------------------
// Utility
//------------------------------------------------------------------------------

/**
* @brief        Takes a screenshot on the target and returns the file path to
*               screenshot.
*
* @param[out]   pOutFilePath The client preallocated buffer to hold the path to
*               the screenshot. If pOutFilePath is null, the full file path will
*               not be returned.
* @param[in]    bufferSize Size of pOutFilePath in bytes.
* @param[in]    targetHandle The handle of the requested target.
* @param[in]    pDirectory The desired directory path to the screenshot. If
*               pDirectory is null or an empty string, the folder specified in
*               Options -> Screenshots will be used.
* @param[in]    pFileName The desired file name of the screenshot. If pFileName
*               is null or an empty string, the filename specified in
*               Options -> Screenshots will be used. pFileName can include the
*               following variables: $(Target), $(Serial), $(Date), $(Time)
*               $(Index).
*
* @result       nn::tm::Error_Ok if successful, else an error code. If
*               Error_InvalidParameter is returned, the specified pDirectory or
*               pFileName is invalid. If Error_InvalidScreenshotFolderSetting or
*               Error_InvalidScreenshotFilenameSetting is returned, the Target
*               Manager screenshot configuration is invalid. Verify settings in
*               Target Manager Options.
*/
Error           TakeScreenshot( char*           pOutFilePath,
                                size_t          bufferSize,
                                TargetHandle    targetHandle,
                                const char*     pDirectory,
                                const char*     pFileName );

//------------------------------------------------------------------------------
} // namespace tm

} // namespace nn
//==============================================================================
