﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TutorialBaseClass.Nx.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This class is shared among all tutorials in order to abstract away per-operating system
 *  details related to NVN initialization and operating system integration. Each tutorial
 *  derives from the TutorialBaseClass, and provides an implementation of the t()
 *  global singleton accessor function.
 */

/**
 * @page PageSampleNvnTutorialLibrary NVN Tutorial Library
 * @tableofcontents
 *
 * @brief
 *   The NVN Tutorial Library contains common code shared by NVN Tutorial 01 and 03-09.
 *
 * @section PageSampleNvnTutorialLibrary_SectionBrief Overview
 *   The NVN Tutorial Library contains common code shared by NVN Tutorial 01 and 03-09.
 *   The source files that make up the library are listed and linked below along with a short description
 *   of what it implements. More implementation details can be found in the inline comments in each file.
 *   The project file for the library is included each of the tutorials' visual studio solution
 *   files and is built before the tutorial runs.
 *
 * @section PageSampleNvnTutorialLibrary_SectionFileStructure File Structure
 *
 * <table>
 * <tr><td> @link AlignedAllocator.h AlignedAllocator.h@endlink, @link AlignedAllocator.cpp AlignedAllocator.cpp@endlink </td><td> @copybrief AlignedAllocator.h </td></tr>
 * <tr><td> @link AssetFileDataHolder.h AssetFileDataHolder.h@endlink, @link AssetFileDataHolder.cpp AssetFileDataHolder.cpp@endlink </td><td> @copybrief AssetFileDataHolder.h </td></tr>
 * <tr><td> @link AssetFileLoadingHelper.h AssetFileLoadingHelper.h@endlink, @link AssetFileLoadingHelper.cpp AssetFileLoadingHelper.cpp@endlink </td><td> @copybrief AssetFileLoadingHelper.h </td></tr>
 * <tr><td> @link DebugTextData.h DebugTextData.h@endlink </td><td> @copybrief DebugTextData.h </td></tr>
 * <tr><td> @link DebugTextRenderer.h DebugTextRenderer.h@endlink, @link DebugTextRenderer.cpp DebugTextRenderer.cpp@endlink </td><td> @copybrief DebugTextRenderer.h </td></tr>
 * <tr><td> @link FrameBufferManager.h FrameBufferManager.h@endlink, @link FrameBufferManager.cpp FrameBufferManager.cpp@endlink </td><td> @copybrief FrameBufferManager.h </td></tr>
 * <tr><td> @link GraphicsObject.h GraphicsObject.h@endlink, @link GraphicsObject.cpp GraphicsObject.cpp@endlink </td><td> @copybrief GraphicsObject.h </td></tr>
 * <tr><td> @link GraphicsObjectCube.h GraphicsObjectCubes.h@endlink, @link GraphicsObjectCube.cpp GraphicsObjectCube.cpp@endlink </td><td> @copybrief GraphicsObjectCube.h </td></tr>
 * <tr><td> @link GraphicsObjectSpecializedCube.h GraphicsObjectSpecializedCube.h@endlink, @link GraphicsObjectSpecializedCube.cpp GraphicsObjectSpecializedCube.cpp@endlink </td><td> @copybrief GraphicsObjectSpecializedCube.h </td></tr>
 * <tr><td> @link ManagedCommandBuffer.h ManagedCommandBuffer.h@endlink, @link ManagedCommandBuffer.cpp ManagedCommandBuffer.cpp@endlink </td><td> @copybrief ManagedCommandBuffer.h </td></tr>
 * <tr><td> @link ManagedUniformBuffer.h ManagedUniformBuffer.h@endlink, @link ManagedUniformBuffer.cpp ManagedUniformBuffer.cpp@endlink </td><td> @copybrief ManagedUniformBuffer.h </td></tr>
 * <tr><td> @link MemoryPool.h MemoryPool.h@endlink, @link MemoryPool.cpp MemoryPool.cpp@endlink </td><td> @copybrief MemoryPool.h </td></tr>
 * <tr><td> @link Model.h Model.h@endlink </td><td> @copybrief Model.h </td></tr>
 * <tr><td> @link MultiThreadedAssetFileLoadingHelper.h MultiThreadedAssetFileLoadingHelper.h@endlink, @link MultiThreadedAssetFileLoadingHelper.cpp MultiThreadedAssetFileLoadingHelper.cpp@endlink </td><td> @copybrief MultiThreadedAssetFileLoadingHelper.h </td></tr>
 * <tr><td> @link OutputFileHeaders.h OutputFileHeaders.h@endlink </td><td> @copybrief OutputFileHeaders.h </td></tr>
 * <tr><td> @link RenderThreadPool.h RenderThreadPool.h@endlink, @link RenderThreadPool.cpp RenderThreadPool.cpp@endlink </td><td> @copybrief RenderThreadPool.h </td></tr>
 * <tr><td> @link ShaderTypeEnum.h ShaderTypeEnum.h@endlink, @link ShaderTypeEnum.cpp ShaderTypeEnum.cpp@endlink </td><td> @copybrief ShaderTypeEnum.h </td></tr>
 * <tr><td> @link StatTracker.h StatTracker.h@endlink, @link StatTracker.cpp StatTracker.cpp@endlink </td><td> @copybrief StatTracker.h </td></tr>
 * <tr><td> @link TextureIDManager.h TextureIDManager.h@endlink, @link TextureIDManager.cpp TextureIDManager.cpp@endlink </td><td> @copybrief TextureIDManager.h </td></tr>
 * <tr><td> @link TutorialBaseClass.h TutorialBaseClass.h@endlink, @link TutorialBaseClass.Windows.cpp TutorialBaseClass.Windows.cpp@endlink, @link TutorialBaseClass.Nx.cpp TutorialBaseClass.Nx.cpp@endlink </td><td> @copybrief TutorialBaseClass.h </td></tr>
 * <tr><td> @link TutorialUtil.h TutorialUtil.h@endlink, @link TutorialUtil.cpp TutorialUtil.cpp@endlink </td><td> @copybrief TutorialUtil.h </td></tr>
 * <tr><td> @link UniformBuffer.h UniformBuffer.h@endlink, @link UniformBuffer.cpp UniformBuffer.cpp@endlink </td><td> @copybrief UniformBuffer.h </td></tr>
 * <tr><td> @link UniformBufferManager.h UniformBufferManager.h@endlink, @link UniformBufferManager.cpp UniformBufferManager.cpp@endlink </td><td> @copybrief UniformBufferManager.h </td></tr>
 * </table>
 *
 */

#include <nn/nn_SdkAssert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/vi.h>
#include <nn/hid.h>

#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nv/nv_MemoryManagement.h>
#include <nvntutorial/TutorialUtil.h>
#include <nvntutorial/TutorialBaseClass.h>

extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char* name);
}

namespace {
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }

    void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void NvFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void* NvDevtoolsAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void NvDevtoolsFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvDevtoolsReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void GlslcFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
} // namespace

class Tutorial
{
public:
    explicit Tutorial();
    ~Tutorial();

    int Run();

private:
    static char const* const m_pWindowClassName;
    static char const* const m_pWindowTitle;

    static int const m_TotalFrameTime = 16; // 60fps.
    // If there is less than this many milliseconds in a frame, just proceed to the next.
    static int const m_PadFrameTime = 2;

    void InitWindow();
    void InitNVN();

    void Draw(unsigned long long millisec);

    nn::vi::NativeWindowHandle  m_NativeWindow;
    nn::vi::Display*            m_pDisplay;
    nn::vi::Layer*              m_pLayer;
    int                         m_WindowWidth;
    int                         m_WindowHeight;
};

// static
char const* const Tutorial::m_pWindowClassName = "NVNTutorialWindowClass";
// static
char const* const Tutorial::m_pWindowTitle = "NVN Tutorial";

Tutorial::Tutorial() : m_NativeWindow(NULL), m_pDisplay(NULL), m_pLayer(NULL), m_WindowWidth(1280), m_WindowHeight(720)
{
    InitWindow();
    InitNVN();
}

Tutorial::~Tutorial()
{
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);

    nn::vi::Finalize();
}

int Tutorial::Run()
{
    nn::os::Tick frameTime(nn::TimeSpan::FromMilliSeconds(0));
    nn::os::Tick frameTimePrev = nn::os::GetSystemTick();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

    t()->Resize(m_WindowWidth, m_WindowHeight);

        /* Initialize touch screen for user exit */
    nn::hid::InitializeTouchScreen();
    nn::hid::TouchScreenState<1> touchScreenState;
    int touchCount = 0;

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Hitting Esc on Windows or touching the Touch Screen on NX will exit the program.\n");
    NN_LOG("--------------------------------------------------------------------------------\n");

    while (runLoop)
    {
        if (doFrame)
        {
            frameTime = nn::os::GetSystemTick();

            NN_ASSERT(frameTime >= frameTimePrev, "test");
            Draw(frameTime.ToTimeSpan().GetMilliSeconds() - frameTimePrev.ToTimeSpan().GetMilliSeconds());
            frameTimePrev = frameTime;

            doFrame = false;
        }

        doSleep = runLoop;
        if (doSleep)
        {
            // Figure out how much time to sleep off, if any
            sleepTime = m_TotalFrameTime - static_cast<int>((nn::os::GetSystemTick().ToTimeSpan() - frameTime.ToTimeSpan()).GetMilliSeconds());
            if (sleepTime <= m_PadFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

            /*
             * If the touch screen has been hit, break out of the run loop
             * and clean up.
             */
        touchCount = nn::hid::GetTouchScreenStates(&touchScreenState, 1);
        if (touchScreenState.count > 0)
        {
            break;
        }
        doFrame = true;
    }

    return 0;
}

void Tutorial::InitWindow()
{
        /* Initialize Visual Interface (VI) system to display
         * to the target's screen
         */
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);

    result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
}

void Tutorial::InitNVN()
{
    // The NVN loader is exposed as a obfuscated OpenGL extension in Windows
    t()->Init(nvnBootstrapLoader, m_NativeWindow);
}

void Tutorial::Draw(unsigned long long millisec)
{
    t()->Draw(millisec);
}

void TutorialRun()
{
        /* Set functions that the file system uses for allocation/deallocation */
    nn::fs::SetAllocator(Allocate, Deallocate);

        /*
        * Set memory allocator for graphics subsystem.
        * This function must be called before using any graphics API's.
        */
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);

        /*
         * Set memory allocator for graphics developer tools and NVN debug layer.
         * This function must be called before using any graphics developer features.
         */
    nv::SetGraphicsDevtoolsAllocator(NvDevtoolsAllocateFunction, NvDevtoolsFreeFunction, NvDevtoolsReallocateFunction, NULL);

        /*
        * Donate memory for graphics driver to work in.
        * This function must be called before using any graphics API's.
        */
    size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    void* graphicsHeap = malloc(graphicsSystemMemorySize);
    nv::InitializeGraphics(graphicsHeap, graphicsSystemMemorySize);

        /*
         * If using glslc online, it is required that this function is called
         * before you use any GLSLC functions bar glslcInitialize.
         */
    glslcSetAllocator(GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL);

    Tutorial tutorial;
    tutorial.Run();

    t()->Shutdown();

    free(graphicsHeap);
}
