﻿// ==========================================================================
//  Copyright (C) Deja Tools, LLC.  All rights reserved.
// ==========================================================================

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <cstdarg>
#include <cstdlib>
#include <string.h>
#include <random>

#ifdef NN_NINTENDO_SDK
#include <nn.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/tma/tma.h>
#include <nn/htcs/htcs_Socket.h>
#include <nn/tma/tma_htcs.h>
#define REPORT( ... ) NN_LOG( __VA_ARGS__ )
#ifdef _WIN32
#include <Windows.h> //For Sleep()
#endif
#else
#include <tchar.h>
#include <WinSock2.h>
#include <ws2tcpip.h>
#include <Windows.h>
#include <stdint.h>
#include "../../HTCSDirectory/HTCSDirectory.h"
#endif
#include <time.h>

#include "../StressTestsCommon.h"
#include "../Utils/Utils.h"

// --------------------------------------------------------------------------

#ifdef NN_NINTENDO_SDK
    using Socket = int;
    using ThreadReturn = int32_t;
    const Socket INVALID_SOCKET{ -1 };
    #define SPRINTF_S( a, size, ... ) sprintf( (a), __VA_ARGS__ )
#else
    using Socket = SOCKET;
    using ThreadReturn = DWORD;
    #define SPRINTF_S( a, size, ... ) sprintf_s( (a), (size), __VA_ARGS__ )
#endif

using TestFunction = int32_t (*)( Socket );

// Forward Test function declarations.
int32_t IncrementalSendTest( Socket Socket );
int32_t IncrementalReceiveTest( Socket Socket );
int32_t LargeSendTest( Socket Socket );
int32_t LargeReceiveTest( Socket Socket );
int32_t SmallSendTest( Socket Socket );
int32_t SmallReceiveTest( Socket Socket );
int32_t RandomSendTest( Socket Socket );
int32_t RandomReceiveTest( Socket Socket );
int32_t SmallRandomSendReceiveTest( Socket Socket );
int32_t LargeRandomSendReceiveTest( Socket Socket );
int32_t SmallThreadedTest( Socket Socket );
int32_t LargeThreadedTest( Socket Socket );

namespace
{
    const int32_t IncrementalSendTest_ExpectedNumberOfSendCommands{ 1024 * 128 };
    const int32_t IncrementalReceiveTest_ExpectedNumberOfReceiveCommands{ 1024 * 128 };
    const int32_t LargeSendTest_ExpectedNumberOfSendCommands{ 30 };
    const int32_t LargeSendTest_DataLength{ StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t LargeReceiveTest_ExpectedNumberOfReceiveCommands{ 30 };
    const int32_t LargeReceiveTest_DataLength{ StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallSendTest_ExpectedNumberOfSendCommands{ 1000 * 30 };
    const int32_t SmallSendTest_RandomNumber_Minimum{ StressTestsCommon::SMALL_MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallSendTest_RandomNumber_Maximum{ StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallReceiveTest_ExpectedNumberOfReceiveCommands{ 1000 * 30 };
    const int32_t SmallReceiveTest_RandomNumber_Minimum{ StressTestsCommon::SMALL_MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallReceiveTest_RandomNumber_Maximum{ StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t RandomSendTest_ExpectedNumberOfSendCommands{ 30 };
    const int32_t RandomSendTest_RandomNumber_Minimum{ StressTestsCommon::MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t RandomSendTest_RandomNumber_Maximum{ StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t RandomReceiveTest_ExpectedNumberOfReceiveCommands{ 30 };
    const int32_t RandomReceiveTest_RandomNumber_Minimum{ StressTestsCommon::MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t RandomReceiveTest_RandomNumber_Maximum{ StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallRandomSendReceiveTest_ExpectedNumberOfCommands{ 1000 * 10 };
    const int32_t SmallRandomSendReceiveTest_RandomNumber_Minimum{ StressTestsCommon::SMALL_MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallRandomSendReceiveTest_RandomNumber_Maximum{ StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t LargeRandomSendReceiveTest_ExpectedNumberOfCommands{ 30 };
    const int32_t LargeRandomSendReceiveTest_RandomNumber_Minimum{ StressTestsCommon::MINIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t LargeRandomSendReceiveTest_RandomNumber_Maximum{ StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND };
    const int32_t SmallThreadedTest_NumberOfSockets{ 8 };
    const int32_t LargeThreadedTest_NumberOfSockets{ 8 };


    struct Test
    {
        const char*     m_pName{ nullptr };
        const char*     m_pShortCommandLine{ nullptr };
        const char*     m_pCommandLine{ nullptr };
        const char*     m_pDescriptionFormat{ nullptr };
        const int32_t   m_DescriptionValue1{ 0 };
        const int32_t   m_DescriptionValue2{ 0 };
        const int32_t   m_DescriptionValue3{ 0 };
        TestFunction    m_pfTestFunction{ nullptr };
        Test( const char* pName, const char* pShortCommandLine, const char *pCommandLine, const char* pDescriptionFormat, int32_t Value1, int32_t Value2, int32_t Value3, TestFunction pfTestFunction )
            :   m_pName             ( pName ),
                m_pShortCommandLine ( pShortCommandLine ),
                m_pCommandLine      ( pCommandLine ),
                m_pDescriptionFormat( pDescriptionFormat ),
                m_DescriptionValue1 ( Value1 ),
                m_DescriptionValue2 ( Value2 ),
                m_DescriptionValue3 ( Value3 ),
                m_pfTestFunction    ( pfTestFunction )
        {
        }
    };

    const Test s_IncrementalSendTest
    {
        "Incremental Send",
        "-is",
        "--incremental_send",
        "Send %d packets sized from 0 to %d bytes.",
        IncrementalSendTest_ExpectedNumberOfSendCommands,
        IncrementalSendTest_ExpectedNumberOfSendCommands,
        0,
        IncrementalSendTest
    };
    const Test s_IncrementalReceiveTest
    {
        "Incremental Receive",
        "-ir",
        "--incremental_receive",
        "Receive %d packets sized from 0 to %d bytes.",
        IncrementalReceiveTest_ExpectedNumberOfReceiveCommands,
        IncrementalReceiveTest_ExpectedNumberOfReceiveCommands,
        0,
        IncrementalReceiveTest
    };
    const Test s_LargeSendTest
    {
        "Large Send",
        "-ls",
        "--large_send",
        "Send %d packets sized at %d bytes.",
        LargeSendTest_ExpectedNumberOfSendCommands,
        LargeSendTest_DataLength,
        0,
        LargeSendTest
    };
    const Test s_LargeReceiveTest
    {
        "Large Receive",
        "-lr",
        "--large_receive",
        "Receive %d packets sized at %d bytes.",
        LargeReceiveTest_ExpectedNumberOfReceiveCommands,
        LargeReceiveTest_DataLength,
        0,
        LargeReceiveTest
    };
    const Test s_SmallSendTest
    {
        "Small Send",
        "-ss",
        "--small_send",
        "Send %d packets randomly sized from %d to %d bytes.",
        SmallSendTest_ExpectedNumberOfSendCommands,
        SmallSendTest_RandomNumber_Minimum,
        SmallSendTest_RandomNumber_Maximum,
        SmallSendTest
    };
    const Test s_SmallReceiveTest
    {
        "Small Receive",
        "-sr",
        "--small_receive",
        "Receive %d packets randomly sized from %d to %d bytes.",
        SmallReceiveTest_ExpectedNumberOfReceiveCommands,
        SmallReceiveTest_RandomNumber_Minimum,
        SmallReceiveTest_RandomNumber_Maximum,
        SmallReceiveTest
    };
    const Test s_RandomSendTest
    {
        "Random Send",
        "-rs",
        "--random_send",
        "Send %d packet randomly sized from %d to %d bytes.",
        RandomSendTest_ExpectedNumberOfSendCommands,
        RandomSendTest_RandomNumber_Minimum,
        RandomSendTest_RandomNumber_Maximum,
        RandomSendTest
    };
    const Test s_RandomReceiveTest
    {
        "Random Receive",
        "-rr",
        "--random_receive",
        "Receive %d packet randomly sized from %d to %d bytes.",
        RandomReceiveTest_ExpectedNumberOfReceiveCommands,
        RandomReceiveTest_RandomNumber_Minimum,
        RandomReceiveTest_RandomNumber_Maximum,
        RandomReceiveTest
    };
    const Test s_SmallRandomSendReceiveTest
    {
        "Small Random Send/Receive",
        "-sra",
        "--small_random",
        "Randomly Send/Receive %d packets random(%d, %d) bytes.",
        SmallRandomSendReceiveTest_ExpectedNumberOfCommands,
        SmallRandomSendReceiveTest_RandomNumber_Minimum,
        SmallRandomSendReceiveTest_RandomNumber_Maximum,
        SmallRandomSendReceiveTest
    };
    const Test s_LargeRandomSendReceiveTest
    {
        "Large Random Send/Receive",
        "-lra",
        "--large_random",
        "Randomly Send/Receive %d packets random(%d, %d) bytes.",
        LargeRandomSendReceiveTest_ExpectedNumberOfCommands,
        LargeRandomSendReceiveTest_RandomNumber_Minimum,
        LargeRandomSendReceiveTest_RandomNumber_Maximum,
        LargeRandomSendReceiveTest
    };
    const Test s_SmallThreadedTest
    {
        "Small Threaded",
        "-st",
        "--small_threaded",
        "Create %d threads, each choosing a random 'small test'.",
        SmallThreadedTest_NumberOfSockets,
        0,
        0,
        SmallThreadedTest
    };
    const Test s_LargeThreadedTest
    {
        "Large Threaded",
        "-lt",
        "--large_threaded",
        "Create %d threads, each choosing a random 'non-threaded test'.",
        LargeThreadedTest_NumberOfSockets,
        0,
        0,
        LargeThreadedTest
    };

    const Test *Tests[] =
    {
        &s_IncrementalSendTest,
        &s_IncrementalReceiveTest,
        &s_LargeSendTest,
        &s_LargeReceiveTest,
        &s_SmallSendTest,
        &s_SmallReceiveTest,
        &s_RandomSendTest,
        &s_RandomReceiveTest,
        &s_SmallRandomSendReceiveTest,
        &s_LargeRandomSendReceiveTest,
        &s_SmallThreadedTest,
        &s_LargeThreadedTest,
    };
    const int32_t NumberOfTests{ sizeof(Tests) / sizeof(Tests[0]) };
    bool bDisplayHelp{ false };

    const char ApplicationName[]{ "SimpleCommandClient" };
    std::uniform_int_distribution<> s_RandomNumberDistribution( StressTestsCommon::MINIMUM_NUMBER_OF_BYTES_TO_SEND, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );
    std::uniform_int_distribution<> s_SmallRandomNumberDistribution( SmallSendTest_RandomNumber_Minimum, SmallSendTest_RandomNumber_Maximum );
    std::mt19937                    s_GeneratorEngine;

    struct TestResult
    {
        int32_t Result{ 0 };
        time_t  StartTime{ 0 };
        time_t  EndTime{ 0 };
        bool    bEnabled{ false };
    };
    TestResult TestResults[NumberOfTests];

    bool bVerboseLog = false;
}

// --------------------------------------------------------------------------
// Forward declarations.
int32_t ReceiveHeaderAndData( Socket Socket, StressTestsCommon::Header& Header, void* pDataBuffer, int32_t* pActualByteCount = nullptr );
bool SendHeader( Socket Socket, const StressTestsCommon::Header& Header );

// --------------------------------------------------------------------------

int32_t GetRandomNumber( void )
{
    const int32_t Number{ s_RandomNumberDistribution( s_GeneratorEngine ) };
    return Number;
}

// --------------------------------------------------------------------------

int32_t GetSmallRandomNumber( void )
{
    const int32_t Number{ s_SmallRandomNumberDistribution(s_GeneratorEngine) };
    return Number;
}

// --------------------------------------------------------------------------

int32_t HTCSGeneratePortName( char* PortName, int32_t PortNameSize, int32_t PortSuffix )
{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
#if 1
        SPRINTF_S( PortName, PortNameSize, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#else
#if defined(WIN32)
        sprintf_s( PortName, PortNameSize, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#else
        (void)PortNameSize;
        sprintf( PortName, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#endif
#endif
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t HTCSCreateConnection( Socket& Socket, int32_t Port )
{
    int32_t Result{ 0 };

    // Generate the Port Name string.
    const int32_t BufferSize{ 64 };
    char LocalPortName[BufferSize];
    if( Result == 0 )
    {
        Result = HTCSGeneratePortName( LocalPortName, BufferSize, Port );
        if( Result != 0 )
        {
            REPORT( "[%s]  Failed to generate HTCS port name for Port# %d.\n", ApplicationName, Port );
        }
    }

    // Create the HTCS Connection.
#if defined(NN_NINTENDO_SDK)
    Socket = CreateConnectSocket( StressTestsCommon::ApplicationName, LocalPortName );
#else
#if WIN32
    htcs_directory HTCSDirectory{};
    if( Result == 0 )
    {
        if( HTCSDirectory.HasError() )
        {
            HTCSDirectory.ReportError();
            Result = EXIT_FAILURE;
        }
    }
#endif

    // Resolve the IP Address and Port number.
    char ResolvedIPAddress[BufferSize];
    int32_t ResolvedPort{ 0 };
    if( Result == 0 )
    {
        if( HTCSDirectory.ResolveTcpAddress( StressTestsCommon::ApplicationName, LocalPortName, ResolvedIPAddress, &ResolvedPort ) == FALSE )
        {
            REPORT( "[%s]  Failed to find command server %s:%s\n", ApplicationName, StressTestsCommon::ApplicationName, LocalPortName );
            Result = EXIT_FAILURE;
        }
    }

    // Create the HTCS connection.
    if( Result == 0 )
    {
        Socket = CreateConnectSocket( ResolvedIPAddress, ResolvedPort );
    }
#endif

    if( Socket == INVALID_SOCKET )
    {
        REPORT( "[%s] Failed to create HTCS Socket.  Port: %d.\n", ApplicationName, Port );
        Result = EXIT_FAILURE;
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t HTCSCloseConnection( Socket& Socket )
{
    int32_t Result{ 0 };

    // Send the CloseServer command to the Server.
    if( Socket != INVALID_SOCKET )
    {
        const StressTestsCommon::Header Header
        {
            StressTestsCommon::Command::CloseServer,
            0,
            0
        };
        if( !SendHeader( Socket, Header ) )
        {
            REPORT( "[%s] Failed to send the CloseServer command.\n", ApplicationName );
            Result = EXIT_FAILURE;
        }
    }

    // Wait for the CloseServer response.
    if( Socket != INVALID_SOCKET )
    {
        StressTestsCommon::Header Header{};
        const int32_t CloseServerResult{ ReceiveHeaderAndData( Socket, Header, nullptr ) };
        if( Result == 0 )
        {
            if( Header.m_Command == StressTestsCommon::Command::CloseServer )
            {
                Result = CloseServerResult;
            }
            else
            {
                Result = EXIT_FAILURE;
            }
        }
    }

    // Close the HTCS Socket.
    CloseSocket( Socket );
    Socket = INVALID_SOCKET;

    return Result;
}
// --------------------------------------------------------------------------
// Reads from the passed socket and puts it into the passed buffer.  Returns the amount
// read, or the error code if there was a problem.

int32_t Receive( Socket Socket, void* pDataBuffer, int32_t ReadAmount )
{
    int32_t AmountRead{ 0 };
    int32_t AmountToRead{ ReadAmount };
    char*   pBuffer{ reinterpret_cast<char *>(pDataBuffer) };
    while( AmountRead < (int32_t)ReadAmount )
    {
#ifdef NN_NINTENDO_SDK
        int32_t Res = nn::htcs::Recv( Socket, pBuffer, AmountToRead, HTCS_MSG_WAITALL );
#else
        int32_t Res = recv( Socket, pBuffer, AmountToRead, 0 );
#endif

        if ( Res > 0  )
        {
            AmountRead += Res;
            AmountToRead -= Res;
            pBuffer += Res;
        }
        else
        {
            return Res;
        }
    }

    return AmountRead;
}

// --------------------------------------------------------------------------
// Sends the passed data to the passed socket.  Returns the amount sent, or the error code
// if there was a problem.
int32_t Send( Socket Socket, const void* pData, uint32_t SizeOfData )
{
    int32_t TotalWritten = 0;
    char* pBuffer = (char*)pData;
    size_t AmountLeft  = SizeOfData;

    while( AmountLeft > 0 )
    {
#ifdef NN_NINTENDO_SDK
        int32_t WrittenThisPass = nn::htcs::Send( Socket, pBuffer, AmountLeft, HTCS_MSG_WAITALL );
#else
        int32_t WrittenThisPass = send( Socket, pBuffer, AmountLeft, 0 );
#endif
        if( WrittenThisPass  <= 0 )
        {
            return WrittenThisPass;
        }

        TotalWritten += WrittenThisPass;
        AmountLeft -= WrittenThisPass;
        pBuffer += WrittenThisPass;
    }

    return TotalWritten;
}

// --------------------------------------------------------------------------

bool SendValue( Socket Socket, int32_t Command )
{
    int32_t Sent = Send( Socket, (char*)&Command, sizeof(int32_t));
    if( Sent != sizeof(int32_t) )
    {
        REPORT( "[%s] Error 0x%x calling Send.\n", ApplicationName, Sent );
        return false;
    }
    return true;
}

// --------------------------------------------------------------------------

bool SendHeader( Socket Socket, const StressTestsCommon::Header& Header )
{
    int32_t Sent = Send( Socket, &Header, sizeof(Header));
    const bool bSuccess{ Sent == sizeof(Header) };
    if( !bSuccess )
    {
        REPORT( "[%s] Error 0x%x calling Send.\n", ApplicationName, Sent );
    }
    return bSuccess;
}

// --------------------------------------------------------------------------

int32_t SendHeaderAndData( int Socket, const StressTestsCommon::Header& Header, const void* pDataBuffer, int32_t* pActualByteCount = nullptr )

{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
        int32_t Sent = Send( Socket, &Header, sizeof(Header) );
        if( Sent != sizeof(Header) )
        {
            REPORT( "[%s] SendHeaderAndData: Error 0x%x calling Send.\n", ApplicationName, Sent );
            Result = EXIT_FAILURE;
        }
    }

    int32_t ActualByteCount{ 0 };
    if( Result == 0 )
    {
        ActualByteCount = Send( Socket, pDataBuffer, Header.m_DataLength );
        if( ActualByteCount != Header.m_DataLength )
        {
            REPORT( "[%s] SendHeaderAndData: Failed to Send: %d != %d bytes.\n", ApplicationName, ActualByteCount, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    // Return the Actual Byte Count, if requested.
    if( pActualByteCount != nullptr )
    {
        *pActualByteCount = ActualByteCount;
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t ReceiveHeaderAndData( Socket Socket, StressTestsCommon::Header& Header, void* pDataBuffer, int32_t* pActualByteCount )
{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
        int32_t Sent = Receive( Socket, &Header, sizeof(Header) );
        if( Sent != sizeof(Header) )
        {
            REPORT( "[%s] ReceiveHeaderAndData: Error 0x%x calling Receive.\n", ApplicationName, Sent );
            Result = EXIT_FAILURE;
        }
    }

    int32_t ActualByteCount{ 0 };
    if( Result == 0 )
    {
        ActualByteCount = Receive( Socket, pDataBuffer, Header.m_DataLength );
        if( ActualByteCount != Header.m_DataLength )
        {
            REPORT( "[%s] ReceiveHeaderAndData: Failed to Receive: %d != %d bytes.\n", ApplicationName, ActualByteCount, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    // Return the Actual Byte Count, if requested.
    if( pActualByteCount != nullptr )
    {
        *pActualByteCount = ActualByteCount;
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t DoSendCommand( Socket Socket, const StressTestsCommon::Header& Header, void* pDataBuffer, int32_t& ActualByteCount )
{
    int32_t Result{ 0 };

    // Default the out parameter.
    ActualByteCount = 0;

    // Send the 'Send Command' packet.
    if( Result == 0 )
    {
        if( !SendHeader( Socket, Header ) )
        {
            REPORT( "[%s] Socket: %d. DoSendCommand: Failed to send 'Send Command Header' to server. (Command.)\n", ApplicationName, Socket );
            Result = EXIT_FAILURE;
        }
    }

    // Send the data.
    if( Result == 0 )
    {
        ActualByteCount = Receive( Socket, pDataBuffer, Header.m_DataLength );
        if( ActualByteCount != Header.m_DataLength )
        {
            REPORT( "[%s] Socket: %d. DoSendCommand: Failed to Receive: 0x%08X != 0x%08X bytes. (Data)\n", ApplicationName, Socket, ActualByteCount, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t DoReceiveCommand( int32_t Socket, const StressTestsCommon::Header& Header, const uint8_t* pDataBuffer, int32_t &ActualByteCount )
{
    int32_t Result{ 0 };

    // Ensure that the ExpectedByteCount does not exceed the send buffer.
    if( Result == 0 )
    {
        if( Header.m_DataLength > StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND )
        {
            REPORT( "[%s] Socket: %d. DoReceiveCommand: Receive size is too large: %d <= %d. (ExpectedByteCount <= %d.)\n", ApplicationName, Socket, Header.m_DataLength, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );
            Result = EXIT_FAILURE;
        }
    }

    // Send the Header.
    if( Result == 0 )
    {
        if( !SendHeader( Socket, Header ) )
        {
            REPORT( "[%s] Socket %d. DoReceiveCommand: Failed to send 'Receive Command' to server. (Command.)\n", ApplicationName, Socket );
            Result = EXIT_FAILURE;
        }
    }

    // Send the Data.
    if( Result == 0 )
    {
        ActualByteCount = Send( Socket, pDataBuffer, Header.m_DataLength );
        if( ActualByteCount != Header.m_DataLength )
        {
            REPORT( "[%s] Socket %d. DoReceiveCommand: Failed to Send: %d != %d bytes.\n", ApplicationName, Socket, ActualByteCount, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    // Receive the ReceiveResult from the Server.
    if( Result == 0 )
    {
        StressTestsCommon::Header ReceiveResultHeader{};
        StressTestsCommon::ReceiveResult ReceiveResult{};
        int32_t ActualByteCount{ 0 };
        Result = ReceiveHeaderAndData( Socket, ReceiveResultHeader, &ReceiveResult, &ActualByteCount );
        if( Result == 0 )
        {
            if( ReceiveResult.m_PatternMatched == 0 )
            {
                REPORT( "[%s] Socket %d. DoReceiveCommand: Failed to Pattern match.  Discrepancy Index: %d/%d.  0x%02X != 0x%02X.\n", ApplicationName, Socket, ReceiveResult.m_DiscrepancyIndex, Header.m_DataLength, ReceiveResult.m_ActualValue, ReceiveResult.m_ExpectedValue );
                Result = EXIT_FAILURE;
            }
        }
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t SendInitializeServer( Socket Socket, int32_t NumberOfCommands, int32_t NumberOfSendCommands, int32_t NumberOfReceiveCommands, int32_t NumberOfSockets = 1 )
{
    int32_t Result{ 0 };

    // Send the "initialize test" to the Server.
    StressTestsCommon::Header Header;
    if( Result == 0 )
    {
        // Send the InitializeTest data.
        const StressTestsCommon::InitializeTest InitializeTest
        {
            NumberOfCommands,               // m_ExpectedNumberOfCommands.
            NumberOfSendCommands,           // m_ExpectedNumberOfSendCommands.
            NumberOfReceiveCommands,        // m_ExpectedNumberOfReceiveCommands.
        };
        Header.m_Command        = StressTestsCommon::Command::InitializeServerTest;
        Header.m_DataLength     = sizeof(InitializeTest);
        Header.m_SequenceNumber = 0;
        Result = SendHeaderAndData( Socket, Header, &InitializeTest );
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t RequestServerTestResults( Socket Socket, StressTestsCommon::FinalizeTest& FinalizeTest )
{
    int32_t Result{ 0 };

    // Send the "request server results" to the Server.
    StressTestsCommon::Header Header;
    if( Result == 0 )
    {
        Header.m_Command        = StressTestsCommon::Command::RequestServerTestResults;
        Header.m_DataLength     = 0;
        Header.m_SequenceNumber = 0;
        Result = SendHeaderAndData( Socket, Header, nullptr );
    }

    // Receive the FinalizeTest results.
    if( Result == 0 )
    {
        Result = ReceiveHeaderAndData( Socket, Header, &FinalizeTest );
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t CompareFinalizeTestResults( const StressTestsCommon::FinalizeTest& Client, const StressTestsCommon::FinalizeTest& Server )
{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
        const bool bCompareTotalCalls     { Client.m_ExpectedNumberOfCommands != 0 };
        const bool bCompareSendCalls      { Client.m_ExpectedNumberOfSendCommands != 0 };
        const bool bCompareReceiveCalls   { Client.m_ExpectedNumberOfReceiveCommands != 0 };
        const bool bTotalCallsMatch       { !bCompareTotalCalls || (Client.m_ExpectedNumberOfCommands == Server.m_ActualNumberOfCommands) };
        const bool bSendCallsMatch        { !bCompareSendCalls || (Client.m_ExpectedNumberOfSendCommands == Server.m_ActualNumberOfSendCommands) };
        const bool bReceiveCallsMatch     { !bCompareReceiveCalls || (Client.m_ExpectedNumberOfReceiveCommands == Server.m_ActualNumberOfReceiveCommands) };
        const bool bByteCountsMatch       { Client.m_NumberOfBytesTransferred == Server.m_NumberOfBytesTransferred };
        const bool bSendByteCountsMatch   { Client.m_NumberOfSendBytes == Server.m_NumberOfSendBytes };
        const bool bReceiveByteCountsMatch{ Client.m_NumberOfReceiveBytes == Server.m_NumberOfReceiveBytes };
        if( bCompareTotalCalls )
        {
            REPORT( "[%s]   Total of send/receive calls: Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_ExpectedNumberOfCommands,        Server.m_ActualNumberOfCommands,        bTotalCallsMatch ? "Passed" : "Failed" );
        }
        if( bCompareSendCalls )
        {
            REPORT( "[%s]   Number of send calls:        Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_ExpectedNumberOfSendCommands,    Server.m_ActualNumberOfSendCommands,    bSendCallsMatch ? "Passed" : "Failed" );
        }
        if( bCompareReceiveCalls )
        {
            REPORT( "[%s]   Number of receive calls:     Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_ExpectedNumberOfReceiveCommands, Server.m_ActualNumberOfReceiveCommands, bReceiveCallsMatch ? "Passed" : "Failed" );
        }
        REPORT( "[%s]   Total Number of bytes:       Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_NumberOfBytesTransferred,        Server.m_NumberOfBytesTransferred,      bByteCountsMatch ? "Passed" : "Failed" );
        REPORT( "[%s]   Number of send bytes:        Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_NumberOfSendBytes,               Server.m_NumberOfSendBytes,             bSendByteCountsMatch ? "Passed" : "Failed" );
        REPORT( "[%s]   Number of receive bytes:     Client: %d; Server: %d; Result: %s.\n", ApplicationName, Client.m_NumberOfReceiveBytes,            Server.m_NumberOfReceiveBytes,          bReceiveByteCountsMatch ? "Passed" : "Failed" );
        if( !bTotalCallsMatch || !bSendCallsMatch || !bReceiveCallsMatch || !bByteCountsMatch || !bSendByteCountsMatch || !bReceiveByteCountsMatch )
        {
            Result = EXIT_FAILURE;
        }
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t IncrementalSendTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------------\n", ApplicationName );
        REPORT( "[%s] IncrementalSendTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ IncrementalSendTest_ExpectedNumberOfSendCommands };
//const int32_t ExpectedNumberOfSendCommands{ 1024 * 32 };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    const int32_t ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
//        memset( pDataBuffer, 0xBB, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );
        memset( pDataBuffer, 0xBB, index );

        // Request the data to be sent from the Server to Client.
        const char *pSendResult{ "Untested" };
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Send;
            Header.m_DataLength     = index;
            Header.m_SequenceNumber = index;
            Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pSendResult = (Result == 0) ? "Yes" : "No";
        }

        // Validate the sent data.
        int32_t DiscrepancyIndex{ -1 };
        uint8_t DataBufferValue{ 0 };
        uint8_t ExpectedBufferValue{ 0 };
        const char *pPatternMatches{ "Untested" };
        if (Result == 0)
        {
            Result = StressTestsCommon::ValidatePatternFill( pDataBuffer, Header.m_DataLength, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
            pPatternMatches = (Result == 0) ? "Yes" : "No";
        }

        if( bVerboseLog )
        {
            REPORT( "  Sent %d == %d '%s'.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pSendResult, pPatternMatches, Result );
        }
        if( Result != 0 )
        {
            if( DiscrepancyIndex != -1 )
            {
                if( bVerboseLog )
                {
                    REPORT( "[%s] Socket: %d. Pattern discrepancy at index: %d.  0x%02X != 0x%02X.\n", ApplicationName, Socket, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
                }
            }
        }

        ActualNumberOfBytesSent += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands  += (Result == 0) ? 1 : 0;
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket %d. IncrementalSendTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Incremental Send Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t IncrementalReceiveTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------------\n", ApplicationName );
        REPORT( "[%s] IncrementalReceiveTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ IncrementalReceiveTest_ExpectedNumberOfReceiveCommands };
//    const int32_t ExpectedNumberOfReceiveCommands{ 1024 * 32 };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    const int32_t ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption on the "Remote host".
//        memset( pDataBuffer, 0xCC, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );
        memset( pDataBuffer, 0xCC, index );

        // Request the data to be sent from the Client to the Server.
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Receive;
            Header.m_DataLength     = index;
            Header.m_SequenceNumber = index;
        }

        // Prefill the data to send with a pattern.
        if( Result == 0 )
        {
            Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
        }

        // Send the data from the Client to the Server.
        const char *pReceiveResult{ "Untested" };
        if (Result == 0)
        {
            Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pReceiveResult = (Result == 0) ? "Yes" : "No";
        }

        // Ask if the Server if the pattern matched or not.

        ActualNumberOfBytesReceived += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands      += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "  Received %d == %d.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pReceiveResult, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. Incremental ReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Incremental Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t LargeSendTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
        REPORT( "[%s] LargeSendTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ LargeSendTest_ExpectedNumberOfSendCommands };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    const int32_t ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
        memset( pDataBuffer, 0xDD, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Server to Client.
        const char *pSendResult{ "Untested" };
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Send;
            Header.m_DataLength     = LargeSendTest_DataLength;
            Header.m_SequenceNumber = index;
            Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pSendResult = (Result == 0) ? "Yes" : "No";
        }

        // Validate the sent data.
        int32_t DiscrepancyIndex{ -1 };
        uint8_t DataBufferValue{ 0 };
        uint8_t ExpectedBufferValue{ 0 };
        const char *pPatternMatches{ "Untested" };
        if (Result == 0)
        {
            Result = StressTestsCommon::ValidatePatternFill( pDataBuffer, Header.m_DataLength, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
            pPatternMatches = (Result == 0) ? "Yes" : "No";
        }

        if( bVerboseLog )
        {
            REPORT( "  Sent %d == %d '%s'.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pSendResult, pPatternMatches, Result );
        }
        if( Result != 0 )
        {
            if( DiscrepancyIndex != -1 )
            {
                if( bVerboseLog )
                {
                    REPORT( "[%s] Socket: %d. Pattern discrepancy at index: %d.  0x%02X != 0x%02X.\n", ApplicationName, Socket, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
                }
            }
        }

        ActualNumberOfBytesSent += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands  += (Result == 0) ? 1 : 0;
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket %d. LargeSendTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Large Send Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t LargeReceiveTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
        REPORT( "[%s] LargeReceiveTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ LargeReceiveTest_ExpectedNumberOfReceiveCommands };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    const int32_t ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption on the "Remote host".
        memset( pDataBuffer, 0xEE, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Client to the Server.
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Receive;
            Header.m_DataLength     = LargeReceiveTest_DataLength;
            Header.m_SequenceNumber = index;
        }

        // Prefill the data to send with a pattern.
        if( Result == 0 )
        {
            Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
        }

        // Send the data from the Client to the Server.
        const char *pReceiveResult{ "Untested" };
        if (Result == 0)
        {
            Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pReceiveResult = (Result == 0) ? "Yes" : "No";
        }

        // Ask if the Server if the pattern matched or not.

        ActualNumberOfBytesReceived += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands      += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "  Received %d == %d.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pReceiveResult, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. LargeReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Large Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t SmallSendTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
        REPORT( "[%s] SmallSendTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ SmallSendTest_ExpectedNumberOfSendCommands };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    const int32_t ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
        memset( pDataBuffer, 0xDD, StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Server to Client.
        const char *pSendResult{ "Untested" };
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Send;
            Header.m_DataLength     = GetSmallRandomNumber();
            Header.m_SequenceNumber = index;
            Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pSendResult = (Result == 0) ? "Yes" : "No";
        }

        // Validate the sent data.
        int32_t DiscrepancyIndex{ -1 };
        uint8_t DataBufferValue{ 0 };
        uint8_t ExpectedBufferValue{ 0 };
        const char *pPatternMatches{ "Untested" };
        if (Result == 0)
        {
            Result = StressTestsCommon::ValidatePatternFill( pDataBuffer, Header.m_DataLength, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
            pPatternMatches = (Result == 0) ? "Yes" : "No";
        }

        if( bVerboseLog )
        {
            REPORT( "  Sent %d == %d '%s'.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pSendResult, pPatternMatches, Result );
        }
        if( Result != 0 )
        {
            if( DiscrepancyIndex != -1 )
            {
                if( bVerboseLog )
                {
                    REPORT( "[%s] Socket: %d. Pattern discrepancy at index: %d.  0x%02X != 0x%02X.\n", ApplicationName, Socket, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
                }
            }
        }

        ActualNumberOfBytesSent += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands  += (Result == 0) ? 1 : 0;
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket %d. SmallSendTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Small Send Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t SmallReceiveTest(Socket Socket)
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
        REPORT( "[%s] SmallReceiveTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ SmallReceiveTest_ExpectedNumberOfReceiveCommands };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    const int32_t ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption on the "Remote host".
        memset( pDataBuffer, 0xEE, StressTestsCommon::SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Client to the Server.
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Receive;
            Header.m_DataLength     = GetSmallRandomNumber();
            Header.m_SequenceNumber = index;
        }

        // Prefill the data to send with a pattern.
        if( Result == 0 )
        {
            Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
        }

        // Send the data from the Client to the Server.
        const char *pReceiveResult{ "Untested" };
        if (Result == 0)
        {
            Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pReceiveResult = (Result == 0) ? "Yes" : "No";
        }

        // Ask if the Server if the pattern matched or not.

        ActualNumberOfBytesReceived += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands      += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "  Received %d == %d.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pReceiveResult, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. SmallReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Small Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t RandomSendTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] ---------------------------------------\n", ApplicationName );
        REPORT( "[%s] RandomSendTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] ---------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ RandomSendTest_ExpectedNumberOfSendCommands };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    const int32_t ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
        memset( pDataBuffer, 0xDD, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Server to Client.
        const char *pSendResult{ "Untested" };
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Send;
            Header.m_DataLength     = GetRandomNumber();
            Header.m_SequenceNumber = index;
            Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pSendResult = (Result == 0) ? "Yes" : "No";
        }

        // Validate the sent data.
        int32_t DiscrepancyIndex{ -1 };
        uint8_t DataBufferValue{ 0 };
        uint8_t ExpectedBufferValue{ 0 };
        const char *pPatternMatches{ "Untested" };
        if (Result == 0)
        {
            Result = StressTestsCommon::ValidatePatternFill( pDataBuffer, Header.m_DataLength, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
            pPatternMatches = (Result == 0) ? "Yes" : "No";
        }

        if( bVerboseLog )
        {
            REPORT( "  Sent %d == %d '%s'.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pSendResult, pPatternMatches, Result );
        }
        if( Result != 0 )
        {
            if( DiscrepancyIndex != -1 )
            {
                if( bVerboseLog )
                {
                    REPORT( "[%s] Socket: %d. Pattern discrepancy at index: %d.  0x%02X != 0x%02X.\n", ApplicationName, Socket, DiscrepancyIndex, DataBufferValue, ExpectedBufferValue );
                }
            }
        }

        ActualNumberOfBytesSent += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands  += (Result == 0) ? 1 : 0;
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket %d. RandomSendTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Random Send Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t RandomReceiveTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] ---------------------------------------\n", ApplicationName );
        REPORT( "[%s] RandomReceiveTest.  Socket: 0x%08X.\n", ApplicationName, Socket );
        REPORT( "[%s] ---------------------------------------\n", ApplicationName );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ RandomReceiveTest_ExpectedNumberOfReceiveCommands };
    const int32_t ExpectedNumberOfCommands{ ExpectedNumberOfSendCommands + ExpectedNumberOfReceiveCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    const int32_t ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for( int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++ )
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption on the "Remote host".
        memset( pDataBuffer, 0xEE, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Request the data to be sent from the Client to the Server.
        if( Result == 0 )
        {
            Header.m_Command        = StressTestsCommon::Command::Receive;
            Header.m_DataLength     = GetRandomNumber();
            Header.m_SequenceNumber = index;
        }

        // Prefill the data to send with a pattern.
        if( Result == 0 )
        {
            Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
        }

        // Send the data from the Client to the Server.
        const char *pReceiveResult{ "Untested" };
        if (Result == 0)
        {
            Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
            pReceiveResult = (Result == 0) ? "Yes" : "No";
        }

        // Ask if the Server if the pattern matched or not.

        ActualNumberOfBytesReceived += (Result == 0) ? Header.m_DataLength : 0;
        ActualNumberOfCommands      += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "  Received %d == %d.  Pattern matches: '%s'.  Result: %d.\n", ActualByteCount, Header.m_DataLength, pReceiveResult, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. RandomReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Random Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t SmallRandomSendReceiveTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] -------------------------------------------------\n", ApplicationName  );
        REPORT( "[%s] SmallRandomSendReceiveTest.  Socket: 0x%08X.\n",      ApplicationName, Socket );
        REPORT( "[%s] -------------------------------------------------\n", ApplicationName  );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
//    const int32_t ExpectedNumberOfReceiveCommands{ 10 };
    const int32_t ExpectedNumberOfCommands{ SmallRandomSendReceiveTest_ExpectedNumberOfCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for (int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++)
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
        memset( pDataBuffer, 0xFF, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Send or Receive the data?
        const bool Send{ (GetRandomNumber() & 1) != 0 };
        if (Result == 0)
        {
            // Send or Receive?
            if( Send )
            {
                Header.m_Command        = StressTestsCommon::Command::Send;
                Header.m_DataLength     = GetSmallRandomNumber();
                Header.m_SequenceNumber = index;
                Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
                ActualNumberOfBytesSent += ActualByteCount;
                ActualNumberOfSendCommands++;
            }
            else
            {
                // Set up the Header.
                Header.m_Command        = StressTestsCommon::Command::Receive;
                Header.m_DataLength     = GetSmallRandomNumber();
                Header.m_SequenceNumber = index;

                // Prefill the data to send with a pattern.
                if (Result == 0)
                {
                    Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
                }

                // Send the data from the Client to the Server.
                if( Result == 0 )
                {

                    Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
                    ActualNumberOfBytesReceived += ActualByteCount;
                    ActualNumberOfReceiveCommands++;
                }
            }
        }

        ActualNumberOfCommands += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "   Byte count: '%s' %d == %d.  Result: %d.\n", Send ? "Send" : "Receive", ActualByteCount, Header.m_DataLength, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. Small Send/ReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d.  Small Send/Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

int32_t LargeRandomSendReceiveTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] -------------------------------------------------\n", ApplicationName  );
        REPORT( "[%s] LargeRandomSendReceiveTest.  Socket: 0x%08X.\n",      ApplicationName, Socket );
        REPORT( "[%s] -------------------------------------------------\n", ApplicationName  );
    }

    uint8_t* pDataBuffer{ new uint8_t[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND] };

    const int32_t ExpectedNumberOfSendCommands{ 0 };
    const int32_t ExpectedNumberOfReceiveCommands{ 0 };
//    const int32_t ExpectedNumberOfReceiveCommands{ 10 };
    const int32_t ExpectedNumberOfCommands{ LargeRandomSendReceiveTest_ExpectedNumberOfCommands };
    int32_t       ActualNumberOfSendCommands{ 0 };
    int32_t       ActualNumberOfReceiveCommands{ 0 };
    int32_t       ActualNumberOfBytesSent{ 0 };
    int32_t       ActualNumberOfBytesReceived{ 0 };

    // Send the "initialize test" to the Server.
    if( Result == 0 )
    {
        Result = SendInitializeServer( Socket, ExpectedNumberOfCommands, ExpectedNumberOfSendCommands, ExpectedNumberOfReceiveCommands );
    }

    // Perform the test.
    StressTestsCommon::Header Header;
    int32_t ActualNumberOfCommands{ 0 };
    int32_t ActualByteCount{ 0 };
    for (int32_t index = 0; (index < ExpectedNumberOfCommands) && (Result == 0); index++)
    {
        if( bVerboseLog )
        {
            REPORT( "[%s]  %5d/%5d) Socket: %d. ", ApplicationName, index + 1, ExpectedNumberOfCommands, Socket );
        }

        // Clear the memory - to help catch corruption from the "Remote host".
        memset( pDataBuffer, 0xFF, StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND );

        // Send or Receive the data?
        const bool Send{ (GetRandomNumber() & 1) != 0 };
        if (Result == 0)
        {
            // Send or Receive?
            if( Send )
            {
                Header.m_Command        = StressTestsCommon::Command::Send;
                Header.m_DataLength     = GetRandomNumber();
                Header.m_SequenceNumber = index;
                Result = DoSendCommand( Socket, Header, pDataBuffer, ActualByteCount );
                ActualNumberOfBytesSent += ActualByteCount;
                ActualNumberOfSendCommands++;
            }
            else
            {
                // Set up the Header.
                Header.m_Command        = StressTestsCommon::Command::Receive;
                Header.m_DataLength     = GetRandomNumber();
                Header.m_SequenceNumber = index;

                // Prefill the data to send with a pattern.
                if (Result == 0)
                {
                    Result = StressTestsCommon::WritePatternFill( pDataBuffer, Header.m_DataLength );
                }

                // Send the data from the Client to the Server.
                if( Result == 0 )
                {

                    Result = DoReceiveCommand( Socket, Header, pDataBuffer, ActualByteCount );
                    ActualNumberOfBytesReceived += ActualByteCount;
                    ActualNumberOfReceiveCommands++;
                }
            }
        }

        ActualNumberOfCommands += (Result == 0) ? 1 : 0;

        if( bVerboseLog )
        {
            REPORT( "   Byte count: '%s' %d == %d.  Result: %d.\n", Send ? "Send" : "Receive", ActualByteCount, Header.m_DataLength, Result );
        }
    }

    // Request the FinalizeTest results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        Result = RequestServerTestResults( Socket, FinalizeTest );
    }

    // Compare results.
    if( Result == 0 )
    {
        const StressTestsCommon::FinalizeTest Client
        {
            ExpectedNumberOfCommands,                               // m_ExpectedNumberOfCommands.
            ExpectedNumberOfSendCommands,                           // m_ExpectedNumberOfSendCommands.
            ExpectedNumberOfReceiveCommands,                        // m_ExpectedNumberOfReceiveCommands.
            ActualNumberOfCommands,                                 // m_ActualNumberOfCommands.
            ActualNumberOfSendCommands,                             // m_ActualNumberOfSendCommands.
            ActualNumberOfReceiveCommands,                          // m_ActualNumberOfReceiveCommands.
            ActualNumberOfBytesSent + ActualNumberOfBytesReceived,  // m_NumberOfBytesTransferred.
            ActualNumberOfBytesSent,                                // m_NumberOfSendBytes.
            ActualNumberOfBytesReceived,                            // m_NumberOfReceiveBytes.
        };
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. Large Send/ReceiveTest results:\n", ApplicationName, Socket );
        }
        Result = CompareFinalizeTestResults( Client, FinalizeTest );
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d.  Large Send/Receive Test finished after running %d of %d commands.  Result: %d.\n", ApplicationName, Socket, ActualNumberOfCommands, ExpectedNumberOfCommands, Result );
    }

    delete[] pDataBuffer;
    pDataBuffer = nullptr;

    return Result;
}

// --------------------------------------------------------------------------

struct TestThreadArgument
{
    Socket       m_Socket;
    TestFunction m_pfFunction;
    int32_t      m_Result;
};
ThreadReturn ThreadedTestThread( void* lpTestThreadArgument )
{
    TestThreadArgument& rTestThreadArgument{ *reinterpret_cast<TestThreadArgument *>(lpTestThreadArgument) };

    rTestThreadArgument.m_Result = rTestThreadArgument.m_pfFunction( rTestThreadArgument.m_Socket );

    return 0;
}

// --------------------------------------------------------------------------

int32_t SmallThreadedTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName  );
        REPORT( "[%s] SmallThreadedTest.  Socket: 0x%08X.\n",    ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName  );
    }

    const int32_t NumberOfSockets{ SmallThreadedTest_NumberOfSockets };

    // Send the "initialize test" to the Server.
    StressTestsCommon::Header Header{};
    if( Result == 0 )
    {
        const StressTestsCommon::ThreadsTest ThreadsTest
        {
            NumberOfSockets
        };
        Header.m_Command        = StressTestsCommon::Command::ThreadsTest;
        Header.m_SequenceNumber = 0;
        Header.m_DataLength     = sizeof(ThreadsTest);
        Result = SendHeaderAndData( Socket, Header, &ThreadsTest );
    }

    // Initialize the Thread TestThread Arguments.
    TestThreadArgument TestThreadArguments[NumberOfSockets]{};
    for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
    {
        TestThreadArguments[Index].m_Socket     = INVALID_SOCKET;
        TestThreadArguments[Index].m_pfFunction = nullptr;
        TestThreadArguments[Index].m_Result     = 0;
    }

    // Receive the created Port Names to connect to.
    if( Result == 0 )
    {
        StressTestsCommon::ThreadsTestResponse ThreadsTestResponse{};
        Result = ReceiveHeaderAndData( Socket, Header, &ThreadsTestResponse );
        if( Result == 0 )
        {
            Result = ThreadsTestResponse.m_Result;
        }
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. ThreadedTest: Create HTCS Sockets?  '%s'.  Result: %d.\n", ApplicationName, Socket, (Result == 0) ? "Yes" : "No", Result );
        }
    }

    // Retrieve the Socket handles.
    if( Result == 0 )
    {
        for (int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            Result = HTCSCreateConnection( TestThreadArguments[Index].m_Socket, Index + 1 );
            if( bVerboseLog )
            {
                REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Create HTCS Connection.  Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
            }
        }
    }

    // Set up the Tests that will run.
    if( Result == 0 )
    {
        for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
        {
            const int32_t Roll{ GetRandomNumber() };
            switch( Roll % 4 )
            {
            case 0: TestThreadArguments[Index].m_pfFunction = SmallSendTest;                break;
            case 1: TestThreadArguments[Index].m_pfFunction = SmallReceiveTest;             break;
            default: // Just in case.
            case 2:
            case 3: TestThreadArguments[Index].m_pfFunction = SmallRandomSendReceiveTest;   break;
            }
        }
    }

    // Create the threads.
#if defined(NN_NINTENDO_SDK)
    char LocalPortName[64]{};
    nn::os::ThreadType hThreads[NumberOfSockets]{};
    const uint32_t StackSize{ 1024 * 64 };
    const uint32_t AllocateStackSize{ ((StackSize + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1)) + (nn::os::StackRegionAlignment - 1) };
    void *pStacks[NumberOfSockets]{};
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            pStacks[Index] = new uint8_t[AllocateStackSize];
            void *pAlignedStack{ (void*)((((uint64_t)pStacks[Index]) - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1)) };
            const nn::Result StartResult{ nn::os::CreateThread( hThreads + Index, reinterpret_cast<nn::os::ThreadFunction>(ThreadedTestThread), TestThreadArguments + Index, pAlignedStack, StackSize, 0 ) };
            if( StartResult.IsSuccess() )
            {
                HTCSGeneratePortName( LocalPortName, sizeof(LocalPortName), Index + 1 );
                nn::os::SetThreadName( hThreads + Index, LocalPortName );
                nn::os::StartThread( hThreads + Index );
            }
            else
            {
                Result = EXIT_FAILURE;
            }
            if( bVerboseLog )
            {
                REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Created thread for Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
            }
        }
    }

    // Wait for all of the threads to finish.
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            // Wait for the thread to exit.
            uint8_t State = hThreads[Index]._state;
            if( ( State == nn::os::ThreadType::State_Initialized ) ||
                ( State == nn::os::ThreadType::State_Started ) ||
                ( State == nn::os::ThreadType::State_Exited) )
            {
                nn::os::WaitThread( hThreads + Index );
                nn::os::DestroyThread( hThreads + Index );
            }

            delete[] static_cast<uint8_t *>(pStacks[Index]);
            pStacks[Index] = nullptr;
        }
    }
#else
    HANDLE hThreads[NumberOfSockets]{};
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            hThreads[Index] = ::CreateThread( nullptr, 0, reinterpret_cast<PTHREAD_START_ROUTINE>(ThreadedTestThread), TestThreadArguments + Index, 0, nullptr );
            if( hThreads[Index] == nullptr )
            {
                Result = EXIT_FAILURE;
            }
            REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Created thread for Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
        }
    }

    // Wait for all of the threads to finish.
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            const DWORD TimeOutMS{ INFINITE };
            const DWORD WaitResult{ ::WaitForSingleObject( hThreads[Index], TimeOutMS ) };
            if( WaitResult != WAIT_OBJECT_0 )
            {
                Result = EXIT_FAILURE;
                REPORT( "[%s]: Socket: %d. Failed to WaitForSingleObject: 0x%08X.\n", ApplicationName, Socket, hThreads[Index] );
            }
        }
    }
#endif

    // Tally the Results - to output below.
    if( Result == 0 )
    {
        for (int32_t Index = 0; Index < NumberOfSockets; Index++ )
        {
            if( TestThreadArguments[Index].m_Result != 0 )
            {
                Result = EXIT_FAILURE;
            }
            if( bVerboseLog )
            {
                REPORT( "[%s] ThreadedTest: %d/%d) Socket: %d. Test Result: '%s'.\n", ApplicationName, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, (TestThreadArguments[Index].m_Result == 0) ? "Passed" : "Failed" );
            }
        }
    }

    // *Always* Close the HTCS Entries.
    for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
    {
        if( TestThreadArguments[Index].m_Socket != INVALID_SOCKET )
        {
            const ::Socket SavedSocketDescriptor{ TestThreadArguments[Index].m_Socket };
            const int32_t CloseResult{ HTCSCloseConnection( TestThreadArguments[Index].m_Socket ) };
            // Close Server header.
            if( bVerboseLog )
            {
                REPORT( "[%s] ThreadedTest: %d/%d) Socket: %d.  Sending the CloseServer command.\n", ApplicationName, Index + 1, NumberOfSockets, SavedSocketDescriptor );
            }
            if( CloseResult != 0 )
            {
                if( Result == 0 )
                {
                    Result = CloseResult;
                }
                if( bVerboseLog )
                {
                    REPORT( "[%s] ThreadedTest: Socket: %d. Failed to Close HTCS.  Result: %d.\n", ApplicationName, SavedSocketDescriptor, Result );
                }
            }
        }
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Threaded Test finished after running %d Threads.  Result: %d.\n", ApplicationName, Socket, NumberOfSockets, Result );
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t LargeThreadedTest( Socket Socket )
{
    int32_t Result{ 0 };

    if( bVerboseLog )
    {
        REPORT( "[%s] --------------------------------------\n", ApplicationName  );
        REPORT( "[%s] LargeThreadedTest.  Socket: 0x%08X.\n",    ApplicationName, Socket );
        REPORT( "[%s] --------------------------------------\n", ApplicationName  );
    }

    const int32_t NumberOfSockets{ LargeThreadedTest_NumberOfSockets };

    // Send the "initialize test" to the Server.
    StressTestsCommon::Header Header{};
    if( Result == 0 )
    {
        const StressTestsCommon::ThreadsTest ThreadsTest
        {
            NumberOfSockets
        };
        Header.m_Command        = StressTestsCommon::Command::ThreadsTest;
        Header.m_SequenceNumber = 0;
        Header.m_DataLength     = sizeof(ThreadsTest);
        Result = SendHeaderAndData( Socket, Header, &ThreadsTest );
    }

    // Initialize the Thread TestThread Arguments.
    TestThreadArgument TestThreadArguments[NumberOfSockets]{};
    for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
    {
        TestThreadArguments[Index].m_Socket     = INVALID_SOCKET;
        TestThreadArguments[Index].m_pfFunction = nullptr;
        TestThreadArguments[Index].m_Result     = 0;
    }

    // Receive the created Port Names to connect to.
    if( Result == 0 )
    {
        StressTestsCommon::ThreadsTestResponse ThreadsTestResponse{};
        Result = ReceiveHeaderAndData( Socket, Header, &ThreadsTestResponse );
        if( Result == 0 )
        {
            Result = ThreadsTestResponse.m_Result;
        }
        if( bVerboseLog )
        {
            REPORT( "[%s] Socket: %d. ThreadedTest: Create HTCS Sockets?  '%s'.  Result: %d.\n", ApplicationName, Socket, (Result == 0) ? "Yes" : "No", Result );
        }
    }

    // Retrieve the Socket handles.
    if( Result == 0 )
    {
        for (int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            Result = HTCSCreateConnection( TestThreadArguments[Index].m_Socket, Index + 1 );
            if( bVerboseLog )
            {
                REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Create HTCS Connection.  Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
            }
        }
    }

    // Set up the Tests that will run.
    if( Result == 0 )
    {
        for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
        {
            const int32_t Roll{ GetRandomNumber() };
            switch( Roll % 16 )
            {
            case 0: TestThreadArguments[Index].m_pfFunction = LargeSendTest;                break;
            case 1: TestThreadArguments[Index].m_pfFunction = LargeReceiveTest;             break;
            case 2:
            case 3:
            case 4:
            case 5: TestThreadArguments[Index].m_pfFunction = SmallSendTest;                break;
            case 6:
            case 7:
            case 8:
            case 9: TestThreadArguments[Index].m_pfFunction = SmallReceiveTest;             break;
            case 10:
            case 11: TestThreadArguments[Index].m_pfFunction = RandomSendTest;              break;
            case 12:
            case 13: TestThreadArguments[Index].m_pfFunction = RandomReceiveTest;           break;
            case 14:
            case 15: TestThreadArguments[Index].m_pfFunction = SmallRandomSendReceiveTest;  break;
            default: // Just in case.
            case 16:
            case 17: TestThreadArguments[Index].m_pfFunction = LargeRandomSendReceiveTest;  break;
            }
        }
    }

    // Create the threads.
#if defined(NN_NINTENDO_SDK)
    char LocalPortName[64]{};
    nn::os::ThreadType hThreads[NumberOfSockets]{};
    const uint32_t StackSize{ 1024 * 64 };
    const uint32_t AllocateStackSize{ ((StackSize + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1)) + (nn::os::StackRegionAlignment - 1) };
    void *pStacks[NumberOfSockets]{};
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            pStacks[Index] = new uint8_t[AllocateStackSize];
            void *pAlignedStack{ (void*)((((uint64_t)pStacks[Index]) - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1)) };
            const nn::Result StartResult{ nn::os::CreateThread( hThreads + Index, reinterpret_cast<nn::os::ThreadFunction>(ThreadedTestThread), TestThreadArguments + Index, pAlignedStack, StackSize, 0 ) };
            if( StartResult.IsSuccess() )
            {
                HTCSGeneratePortName( LocalPortName, sizeof(LocalPortName), Index + 1 );
                nn::os::SetThreadName( hThreads + Index, LocalPortName );
                nn::os::StartThread( hThreads + Index );
            }
            else
            {
                Result = EXIT_FAILURE;
            }
            if( bVerboseLog )
            {
                REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Created thread for Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
            }
        }
    }

    // Wait for all of the threads to finish.
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            // Wait for the thread to exit.
            uint8_t State = hThreads[Index]._state;
            if( ( State == nn::os::ThreadType::State_Initialized ) ||
                ( State == nn::os::ThreadType::State_Started ) ||
                ( State == nn::os::ThreadType::State_Exited) )
            {
                nn::os::WaitThread( hThreads + Index );
                nn::os::DestroyThread( hThreads + Index );
            }

            delete[] static_cast<uint8_t *>(pStacks[Index]);
            pStacks[Index] = nullptr;
        }
    }
#else
    HANDLE hThreads[NumberOfSockets]{};
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            hThreads[Index] = ::CreateThread( nullptr, 0, reinterpret_cast<PTHREAD_START_ROUTINE>(ThreadedTestThread), TestThreadArguments + Index, 0, nullptr );
            if( hThreads[Index] == nullptr )
            {
                Result = EXIT_FAILURE;
            }
            REPORT( "[%s] Socket: %d. ThreadedTest: %d/%d) Created thread for Socket: %d.  Result: %d.\n", ApplicationName, Socket, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
        }
    }

    // Wait for all of the threads to finish.
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < NumberOfSockets) && (Result == 0); Index++ )
        {
            const DWORD TimeOutMS{ INFINITE };
            const DWORD WaitResult{ ::WaitForSingleObject( hThreads[Index], TimeOutMS ) };
            if( WaitResult != WAIT_OBJECT_0 )
            {
                Result = EXIT_FAILURE;
                REPORT( "[%s]: Socket: %d. Failed to WaitForSingleObject: 0x%08X.\n", ApplicationName, Socket, hThreads[Index] );
            }
        }
    }
#endif

    // Tally the Results - to output below.
    if( Result == 0 )
    {
        for (int32_t Index = 0; Index < NumberOfSockets; Index++ )
        {
            if( TestThreadArguments[Index].m_Result != 0 )
            {
                Result = EXIT_FAILURE;
            }
            if( bVerboseLog )
            {
                REPORT( "[%s] ThreadedTest: %d/%d) Socket: %d. Test Result: '%s'.\n", ApplicationName, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, (TestThreadArguments[Index].m_Result == 0) ? "Passed" : "Failed" );
            }
        }
    }

    // *Always* Close the HTCS Entries.
    for( int32_t Index = 0; Index < NumberOfSockets; Index++ )
    {
        if( TestThreadArguments[Index].m_Socket != INVALID_SOCKET )
        {
            const ::Socket SavedSocketDescriptor{ TestThreadArguments[Index].m_Socket };
            const int32_t CloseResult{ HTCSCloseConnection( TestThreadArguments[Index].m_Socket ) };
            // Close Server header.
            if( bVerboseLog )
            {
                REPORT( "[%s] ThreadedTest: %d/%d) Socket: %d.  Sending the CloseServer command.\n", ApplicationName, Index + 1, NumberOfSockets, SavedSocketDescriptor );
            }
            if( CloseResult != 0 )
            {
                if( Result == 0 )
                {
                    Result = CloseResult;
                }
                if( bVerboseLog )
                {
                    REPORT( "[%s] ThreadedTest: Socket: %d. Failed to Close HTCS.  Result: %d.\n", ApplicationName, SavedSocketDescriptor, Result );
                }
            }
        }
    }

    if( bVerboseLog )
    {
        REPORT( "[%s] Socket: %d. Threaded Test finished after running %d Threads.  Result: %d.\n", ApplicationName, Socket, NumberOfSockets, Result );
    }

    return Result;
}

// --------------------------------------------------------------------------

void ParseArguments( int argc, char** argv )
{
    // If this becomes true, then set all TestResult[*].bEnabled to true.
    bool bRunAllTests{ false };

    // Set default values.
    bDisplayHelp = false;
    for( int32_t Index = 0; Index < NumberOfTests; Index++ )
    {
        // Default all tests to *not* run.
        TestResults[Index].bEnabled = false;
    }

    // Are there more command line arguments other than the executable name?
    if( argc > 1 )
    {
        // argv[0] is the executable name.  Skip it.
        for( int32_t ArgumentIndex = 1; ArgumentIndex < argc; ArgumentIndex++ )
        {
            bool bFound{ false };
            for( int32_t TestIndex = 0; (TestIndex < NumberOfTests) && !bFound; TestIndex++ )
            {
                if( (strcmp(Tests[TestIndex]->m_pShortCommandLine, argv[ArgumentIndex]) == 0) || (strcmp(Tests[TestIndex]->m_pCommandLine, argv[ArgumentIndex]) == 0) )
                {
                    TestResults[TestIndex].bEnabled = true;
                    bFound = true;
                }
            }

            // If not found, is this a request for help?
            if( !bFound )
            {
                if (
                    (strcmp( argv[ArgumentIndex], "h") == 0 )       ||
                    (strcmp( argv[ArgumentIndex], "-h") == 0 )      ||
                    (strcmp( argv[ArgumentIndex], "/h") == 0 )      ||
                    (strcmp( argv[ArgumentIndex], "--help") == 0 )  ||
                    (strcmp( argv[ArgumentIndex], "?") == 0 )       ||
                    (strcmp( argv[ArgumentIndex], "-?") == 0 )      ||
                    (strcmp( argv[ArgumentIndex], "/?") == 0 )
                   )
                {
                    bDisplayHelp = true;
                }
                else
                {
                    if( strcmp( argv[ArgumentIndex], "all" ) == 0 )
                    {
                        bRunAllTests = true;
                    }

                    if( strcmp( argv[ArgumentIndex], "verbose" ) == 0 )
                    {
                        bVerboseLog = true;
                    }
                }
            }
        }
    }
    else
    {
        // No command line arguments.  Assume the user is requesting help.
        bDisplayHelp = true;
    }

    // Was "all" specified?
    if( bRunAllTests )
    {
        for( int32_t Index = 0; Index < NumberOfTests; Index++ )
        {
            TestResults[Index].bEnabled = true;
        }
    }
}

// --------------------------------------------------------------------------

#ifdef NN_NINTENDO_SDK
extern "C" int nnMain( void )
{
    int argc{ nn::os::GetHostArgc() };
    char** argv{ nn::os::GetHostArgv() };
#else
int32_t main( int argc, char** argv )
{
#endif
    int32_t Result{ 0 };

    // Parse the command line arguments.
    ParseArguments( argc, argv );

    // Display the help to the user?
    if( !bDisplayHelp )
    {
        InitSystem();

#if defined(NN_NINTENDO_SDK) & defined(_WIN32)
        // --------------------------------------------------------------------------
        // On Win32, sleep long enough for us to use Target Manager to
        // connect to this instance.
        // --------------------------------------------------------------------------
        Sleep( 10000 );
#elif WIN32
        // --------------------------------------------------------------------------
        // PC-based applications have to use the HTCS directory service to
        // resolve the IP address and port from a HTCS server & HTCS port name.
        // This htcs_directory object provides an interface for doing that.
        // (You can find the source in the Samples\HTCS\HTCSDirectory folder.)
        // --------------------------------------------------------------------------
        htcs_directory HTCSDirectory;
        if( Result == 0 )
        {
            if( HTCSDirectory.HasError() )
            {
                HTCSDirectory.ReportError();
                Result = EXIT_FAILURE;
            }
        }
#endif

        REPORT( "[%s] -------------------------------\n", ApplicationName );
        REPORT( "[%s]        %s\n",                       ApplicationName, ApplicationName );
        REPORT( "[%s] -------------------------------\n", ApplicationName );

        // -----------------------
        //  Connect to the Server.
        Socket ConnectingSocket{ 0 };
        // -----------------------

        if( Result == 0 )
        {
            Result = HTCSCreateConnection( ConnectingSocket, 0 );
        }

        // ----------------
        // Run the test(s).
        // ----------------

        if( Result == 0 )
        {
            const int32_t WorkBufferSize{ 256 };
            char WorkBuffer[WorkBufferSize]{ "" };
            for( int32_t Index = 0; Index < NumberOfTests; Index++ )
            {
                if( TestResults[Index].bEnabled )
                {
                    SPRINTF_S( WorkBuffer, WorkBufferSize, "%s.", Tests[Index]->m_pName );
                    time( &TestResults[Index].StartTime );
                    REPORT( "=========================================\n" );
                    REPORT( "Running Test: %-26s\n", WorkBuffer );
                    TestResults[Index].Result = Tests[Index]->m_pfTestFunction( ConnectingSocket );
                    time( &TestResults[Index].EndTime );
                    const int32_t TestTimeInSeconds{ static_cast<int32_t>(TestResults[Index].EndTime - TestResults[Index].StartTime) };
                    REPORT( "Done.  Result: %d.  Time: %d seconds.\n", TestResults[Index].Result, TestTimeInSeconds );
                    REPORT( "=========================================\n" );
                }
            }
        }

        // ------------------------------
        // Display the test time results.
        // ------------------------------
        REPORT( "------------------------\n" );
        REPORT( "Test Timing information.\n" );
        REPORT( "------------------------\n" );
        int64_t TotalTime{ 0 };
        int32_t TotalTests{ 0 };
        for( int32_t Index = 0; Index < NumberOfTests; Index++ )
        {
            if( TestResults[Index].bEnabled )
            {
                const int64_t ElapsedTime{ static_cast<int64_t>(TestResults[Index].EndTime - TestResults[Index].StartTime) };
                REPORT( "%-26s: % 8d seconds.\n", Tests[Index]->m_pName, ElapsedTime );
                TotalTime += ElapsedTime;
                TotalTests++;
            }
        }
        REPORT( "%-26s: % 8d seconds.\n", "Total Time", static_cast<int32_t>(TotalTime) );
        REPORT( "%-26s: %8.3f seconds.\n", "Average Time", static_cast<float>(TotalTime) / static_cast<float>((TotalTests != 0) ? TotalTests : 1) );

        // -------------
        // Close Server.
        // -------------
        REPORT( "[%s] Socket: %d. Sending the CloseServer command.\n", ApplicationName, ConnectingSocket );
        if( ConnectingSocket != INVALID_SOCKET )
        {
            const int32_t CloseSocketResult{ HTCSCloseConnection( ConnectingSocket ) };
            if( CloseSocketResult != 0 )
            {
                if( Result == 0 )
                {
                    Result = CloseSocketResult;
                }
                REPORT( "[%s] Failed to close 'original' HTCS Socket: %d.  Result: %d.", ApplicationName, ConnectingSocket, Result );
            }
        }

        // ---------
        // Shutdown.
        // ---------

        CloseSocket( ConnectingSocket );
        CloseSystem();

        REPORT( "[%s] Quitting.  Result: %d.  Passed: '%s'.\n", ApplicationName, Result, (Result == 0) ? "Yes" : "No" );
    }
    else
    {
        // Display help.
        const int32_t WorkBufferSize{ 256 };
        char WorkBuffer1[WorkBufferSize]{ "" };
        char WorkBuffer2[WorkBufferSize]{ "" };
        REPORT( "%s.exe - HTCS stress test tool.\n", ApplicationName );
        REPORT( "Usage: %s.exe [test options]\n", ApplicationName );
        REPORT( "Where [tests options] are provided from the following list:\n\n" );
        REPORT( "%-25s %-27s [Server role].\n", "Test Name", "[Short | Long Argument]" );
        for( int32_t Index = 0; Index < NumberOfTests; Index++ )
        {
            const Test *pTest{ Tests[Index] };
#if 1
            SPRINTF_S( WorkBuffer1, WorkBufferSize, "%-4s | %s", pTest->m_pShortCommandLine, pTest->m_pCommandLine );
            SPRINTF_S( WorkBuffer2, WorkBufferSize, pTest->m_pDescriptionFormat, pTest->m_DescriptionValue1, pTest->m_DescriptionValue2, pTest->m_DescriptionValue3 );
#else
#if !defined(NN_NINTENDO_SDK)
            sprintf_s( WorkBuffer1, "%s | %s", pTest->m_pShortCommandLine, pTest->m_pCommandLine );
            sprintf_s( WorkBuffer2, pTest->m_pDescriptionFormat, pTest->m_DescriptionValue1, pTest->m_DescriptionValue2, pTest->m_DescriptionValue3 );
#else
            sprintf( WorkBuffer1, "%s | %s", pTest->m_pShortCommandLine, pTest->m_pCommandLine );
            sprintf( WorkBuffer2, pTest->m_pDescriptionFormat, pTest->m_DescriptionValue1, pTest->m_DescriptionValue2, pTest->m_DescriptionValue3 );
#endif
#endif
            REPORT( "%-25s %-27s %s\n", pTest->m_pName, WorkBuffer1, WorkBuffer2 );
        }
        REPORT( "Run all: %s.exe all.\n", ApplicationName );
        REPORT( "Help:    %s.exe [-h | --help | ? | /? | /h].\n", ApplicationName );
        REPORT( "Example: %s.exe -h\n", ApplicationName );
        REPORT( "Example: %s.exe %s %s\n", ApplicationName, Tests[0]->m_pShortCommandLine, Tests[1]->m_pShortCommandLine );
    }

    return Result;
}

