﻿//==============================================================================
//
//  mem_mgr.cpp
//
//==============================================================================
//  Copyright (C) DejaTools, LLC.  All rights reserved.
//==============================================================================

#include "stdafx.h"
#include "DejaInsight.h"
#include <stdlib.h>
#include "PXMalloc.h"

//==============================================================================

#ifdef new
#undef new
#endif

#ifdef delete
#undef
#endif

#ifdef malloc
#undef malloc
#endif

#ifdef realloc
#undef realloc
#endif

#ifdef free
#undef free
#endif

//==============================================================================
// PXMalloc
//==============================================================================

#define HEAP_SIZE (8*1024*1024)

#if (defined( WIN32 ) && !defined(_M_X64)) || defined( __NX__ )
#define USE_PXMALLOC
#endif

#if defined( WIN32 )
__declspec(align(4096)) static char s_Heap[HEAP_SIZE];
#else
static char s_Heap[HEAP_SIZE] __attribute__ ((__aligned__(4096)));
#endif

#if defined( USE_PXMALLOC )
static px::Memory::Allocator    s_Allocator;
#else
static char* s_pHeapCursor = s_Heap;
#endif

void tiny_memmgr_init( void )
{
#if defined( USE_PXMALLOC )
    s_Allocator.Initialize( sizeof(s_Heap) );
#endif
}

void* tiny_memmgr_reserve( size_t Size )
{
    ASSERT( Size <= sizeof(s_Heap) );
    return( s_Heap );
}

//==============================================================================
//
//  First, the functions which actually perform the heap operations.  These just
//  wrap the standard C library heap functions.
//
//  Both the standard C library heap function replacements (MyMalloc, MyRealloc,
//  and MyFree) and the C++ global operators new and delete are routed through
//  here.
//
//==============================================================================

static
void* HeapMalloc( size_t Size )
{
#if defined( USE_PXMALLOC )
    void* pResult = s_Allocator.VirtualAlloc( Size, 4, "" );
#else
    void* pResult = s_pHeapCursor;
    ASSERT( (s_pHeapCursor + Size) <= &s_Heap[HEAP_SIZE] );
    s_pHeapCursor += Size;
#endif

    return( pResult );
}

//------------------------------------------------------------------------------

static
void* HeapRealloc( void* pMemory, size_t Size )
{
#if defined( USE_PXMALLOC )
    void* pResult = s_Allocator.VirtualRealloc( pMemory, Size, 4, "" );
#else
    // Not supported with this simple allocator.
    void* pResult = NULL;
    ASSERT( 0 );
#endif

    return( pResult );
}

//------------------------------------------------------------------------------

static
void HeapFree( void* pMemory )
{
#if defined( USE_PXMALLOC )
    s_Allocator.VirtualFree( pMemory );
#else
    // Not supported with this simple allocator.
#endif
}

//==============================================================================
//
//  Next, the public functions which provide the standard C heap functionality.
//  These functions include appropriate DejaLib instrumentation.
//
//==============================================================================

void* tiny_malloc( size_t Size )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_MALLOC( pResult, Size );
    return( pResult );
}

//------------------------------------------------------------------------------

void* tiny_malloc( size_t Size, const char* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_MALLOC( pResult, Size, pFileName, Line );
    return( pResult );
}

//------------------------------------------------------------------------------

void* tiny_malloc( size_t Size, const wchar_t* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_MALLOC( pResult, Size, pFileName, Line );
    return( pResult );
}

//==============================================================================

void* tiny_realloc( void* pMemory, size_t Size )
{
    void* pResult = HeapRealloc( pMemory, Size );
    DEJA_LOG_REALLOC( pResult, pMemory, Size );
    return( pResult );
}

//------------------------------------------------------------------------------

void* tiny_realloc(       void*  pMemory,
                        size_t Size,
                  const char*  pFileName,
                        int    Line )
{
    void* pResult = HeapRealloc( pMemory, Size );
    DEJA_LOG_REALLOC( pResult, pMemory, Size, pFileName, Line );
    return( pResult );
}

//------------------------------------------------------------------------------

void* tiny_realloc(       void*    pMemory,
                        size_t   Size,
                  const wchar_t* pFileName,
                        int      Line )
{
    void* pResult = HeapRealloc( pMemory, Size );
    DEJA_LOG_REALLOC( pResult, pMemory, Size, pFileName, Line );
    return( pResult );
}

//==============================================================================

void tiny_free( void* pMemory )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_FREE( pMemory );
    }

    HeapFree( pMemory );
}

//------------------------------------------------------------------------------

void tiny_free( void* pMemory, const char* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_FREE( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//------------------------------------------------------------------------------

void tiny_free( void* pMemory, const wchar_t* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_FREE( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//==============================================================================
//
//  Finally, the public versions of global operators new and delete.  These
//  functions include appropriate DejaLib instrumentation.
//
//==============================================================================

#ifndef CAFE

void* operator new ( size_t Size )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW( pResult, Size );
    return( pResult );
}

#endif

//------------------------------------------------------------------------------

void* operator new ( size_t Size, const char* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW( pResult, Size, pFileName, Line );
    return( pResult );
}

//------------------------------------------------------------------------------

void* operator new ( size_t Size, const wchar_t* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW( pResult, Size, pFileName, Line );
    return( pResult );
}

//==============================================================================

#ifndef CAFE

void* operator new [] ( size_t Size )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW_ARRAY( pResult, Size );
    return( pResult );
}

#endif

//------------------------------------------------------------------------------

void* operator new [] ( size_t Size, const char* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW_ARRAY( pResult, Size, pFileName, Line );
    return( pResult );
}

//------------------------------------------------------------------------------

void* operator new [] ( size_t Size, const wchar_t* pFileName, int Line )
{
    void* pResult = HeapMalloc( Size );
    DEJA_LOG_NEW_ARRAY( pResult, Size, pFileName, Line );
    return( pResult );
}

//==============================================================================

#ifndef CAFE
#ifdef __NX__
void operator delete ( void* pMemory ) NN_NOEXCEPT
#else
void operator delete ( void* pMemory )
#endif
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE( pMemory );
    }

    HeapFree( pMemory );
}
#endif

//------------------------------------------------------------------------------

void operator delete ( void* pMemory, const char* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//------------------------------------------------------------------------------

void operator delete ( void* pMemory, const wchar_t* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//==============================================================================

#ifndef CAFE
#ifdef __NX__
void operator delete [] ( void* pMemory ) NN_NOEXCEPT
#else
void operator delete [] ( void* pMemory )
#endif
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE_ARRAY( pMemory );
    }

    HeapFree( pMemory );
}
#endif

//------------------------------------------------------------------------------

void operator delete [] ( void* pMemory, const char* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE_ARRAY( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//------------------------------------------------------------------------------

void operator delete [] ( void* pMemory, const wchar_t* pFileName, int Line )
{
    if( pMemory )
    {
        // Recommendation:  Only log when the pointer is non-NULL.
        DEJA_LOG_DELETE_ARRAY( pMemory, pFileName, Line );
    }

    HeapFree( pMemory );
}

//==============================================================================
