﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// main DoDialog UpdateDialog
// DoUnite UniteInputImages
// InitOptions GetOptions CreateOptionsDesc

/*

// BEGIN__HARVEST_EXCEPTION_ZSTRING

<javascriptresource>
<name>Nintendo Unite Images...</name>
<menu>automate</menu>
<type>automate</type>
<eventid>10C95BF7-FF6C-4ed0-8D50-16A78B1E0A6D</eventid>
<category>Nintendo</category>
<terminology><![CDATA[<<
	/Version 1
	/Events <<
		/10C95BF7-FF6C-4ed0-8D50-16A78B1E0A6D [(NintendoUniteImages) /noDirectParam <<
			/InputFolder [(Input Folder) /char]
			/PixelsW [(Width) /integer]
			/PixelsH [(Height) /integer]
			/CountH [(Horizontal) /integer]
			/CountV [(Vertical) /integer]
			/UnitesAlpha [(Unite Alpha) /boolean]
			/FillsColorEdge [(Fill Color Edge) /boolean]
			/FillsAlphaEdge [(Fill Alpha Edge) /boolean]
		>>]
	>>
>>]]></terminology>
</javascriptresource>

// END__HARVEST_EXCEPTION_ZSTRING

*/

#strict on
#target photoshop

#include "./NintendoCommon.jsxinc"

//=============================================================================
// constants
//=============================================================================

// Photoshop のレジストリに最後に実行した際のオプション設定を記憶する際の ID です。
var OPTIONS_ID = "5B427C81-6D6A-47e9-B9D1-15B6CD2A994A";

// オプション設定が有効か判断するためのメッセージ文字列です。
var OPTIONS_MESSAGE = "NW4F_UniteImages Options";

// アクション記述子のキーの定義です。
var KEY_MESSAGE          = charIDToTypeID('Msge');
var KEY_INPUT_FOLDER     = stringIDToTypeID("InputFolder");
var KEY_PIXELS_W         = stringIDToTypeID("PixelsW");
var KEY_PIXELS_H         = stringIDToTypeID("PixelsH");
var KEY_COUNT_H          = stringIDToTypeID("CountH");
var KEY_COUNT_V          = stringIDToTypeID("CountV");
var KEY_UNITES_ALPHA     = stringIDToTypeID("UnitesAlpha");
var KEY_FILLS_COLOR_EDGE = stringIDToTypeID("FillsColorEdge");
var KEY_FILLS_ALPHA_EDGE = stringIDToTypeID("FillsAlphaEdge");

var CUSTOM_ITEM = "custom"; //!< ドロップダウンリストのカスタムアイテム名です。

//-----------------------------------------------------------------------------
//! @brief デフォルトの入力フォルダのパスを返します。
//-----------------------------------------------------------------------------
function GetDefaultInputFolder()
{
	if (app.documents.length != 0)
	{
		var doc = app.activeDocument;
		try // 名称未設定だと doc.fullName のアクセスでエラーになるので try を使用します。
		{
			var folder = Folder(doc.fullName.parent);
			if (folder.exists)
			{
				return folder.fsName;
			}
		}
		catch (e)
		{
		}
	}

	//return Folder.current.fsName;
	return Folder.myDocuments.fsName;
}

//-----------------------------------------------------------------------------
//! @brief オプション設定を初期化します。
//!
//! @param[out] opts オプション設定です。
//-----------------------------------------------------------------------------
function InitOptions(opts)
{
	opts.m_InputFolder = GetDefaultInputFolder();

	opts.m_PixelsW = 32;
	opts.m_PixelsH = 32;

	opts.m_CountH = 4;
	opts.m_CountV = 4;

	opts.m_UnitesAlpha = true;

	opts.m_FillsColorEdge = false;
	opts.m_FillsAlphaEdge = false;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子からオプションを取得します。
//!
//! @param[in,out] opts オプション設定です。
//-----------------------------------------------------------------------------
function GetOptions(opts, desc)
{
	//-----------------------------------------------------------------------------
	// メッセージ文字列が正しくなければ取得しません。
	if (desc.count != 0          &&
		desc.hasKey(KEY_MESSAGE) &&
		desc.getString(KEY_MESSAGE) != OPTIONS_MESSAGE)
	{
		return;
	}

	//-----------------------------------------------------------------------------
	// 各キーを解析します。
	for (var iKey = 0; iKey < desc.count; ++iKey)
	{
		var key = desc.getKey(iKey);
		//alert("key" + iKey + ": " + app.typeIDToStringID(key) + ": " + desc.getType(key));
		switch (key)
		{
			case KEY_INPUT_FOLDER:
				var path = desc.getString(key);
				if (path.length != 0) opts.m_InputFolder = path;
				break;

			case KEY_PIXELS_W:
				var value = desc.getInteger(key);
				if (value >= 1) opts.m_PixelsW = value;
				break;
			case KEY_PIXELS_H:
				var value = desc.getInteger(key);
				if (value >= 1) opts.m_PixelsH = value;
				break;

			case KEY_COUNT_H:
				var value = desc.getInteger(key);
				if (value >= 1) opts.m_CountH = value;
				break;
			case KEY_COUNT_V:
				var value = desc.getInteger(key);
				if (value >= 1) opts.m_CountV = value;
				break;

			case KEY_UNITES_ALPHA:
				opts.m_UnitesAlpha = desc.getBoolean(key);
				break;

			case KEY_FILLS_COLOR_EDGE:
				opts.m_FillsColorEdge = desc.getBoolean(key);
				break;
			case KEY_FILLS_ALPHA_EDGE:
				opts.m_FillsAlphaEdge = desc.getBoolean(key);
				break;
		}
	}
}

//-----------------------------------------------------------------------------
//! @brief オプション設定からアクション記述子を作成します。
//!
//! @param[in] opts オプション設定です。
//-----------------------------------------------------------------------------
function CreateOptionsDesc(opts)
{
	var desc = new ActionDescriptor;

	desc.putString(KEY_MESSAGE, OPTIONS_MESSAGE);

	if (opts.m_InputFolder.length != 0)
	{
		desc.putString(KEY_INPUT_FOLDER, opts.m_InputFolder);
	}

	desc.putInteger(KEY_PIXELS_W, opts.m_PixelsW);
	desc.putInteger(KEY_PIXELS_H, opts.m_PixelsH);

	desc.putInteger(KEY_COUNT_H, opts.m_CountH);
	desc.putInteger(KEY_COUNT_V, opts.m_CountV);

	desc.putBoolean(KEY_UNITES_ALPHA, opts.m_UnitesAlpha);

	desc.putBoolean(KEY_FILLS_COLOR_EDGE, opts.m_FillsColorEdge);
	desc.putBoolean(KEY_FILLS_ALPHA_EDGE, opts.m_FillsAlphaEdge);

	return desc;
}

//-----------------------------------------------------------------------------
//! @brief 上下左右の端の 1 ピクセルを選択します。
//-----------------------------------------------------------------------------
function SelectEdgePixels(doc)
{
	var docW = doc.width.value;
	var docH = doc.height.value;
	if (docW <= 2 || docH <= 2)
	{
		doc.selection.selectAll();
	}
	else
	{
		var x0 = 1;
		var y0 = 1;
		var x1 = docW - 1;
		var y1 = docH - 1;
		var region = Array(
			Array(x0, y0),
			Array(x1, y0),
			Array(x1, y1),
			Array(x0, y1),
			Array(x0, y0))
		doc.selection.select(region);
		doc.selection.invert();
	}
}

//-----------------------------------------------------------------------------
//! @brief 入力画像を出力画像に結合します。
//!        opts.m_IsSucceeded に処理成功なら true、処理失敗なら false を格納します。
//!
//! @param[in,out] dstDoc 出力画像です。
//! @param[in,out] opts オプション設定です。
//! @param[in] filePaths 入力画像のファイルパス配列です。
//-----------------------------------------------------------------------------
function UniteInputImages(dstDoc, opts, filePaths)
{
	//-----------------------------------------------------------------------------
	// 処理成功フラグを true で初期します。
	opts.m_IsSucceeded = true;

	//-----------------------------------------------------------------------------
	// 塗りつぶし用のカラーを作成します。
	var blackColor = new SolidColor();
	blackColor.rgb.red   = 0;
	blackColor.rgb.green = 0;
	blackColor.rgb.blue  = 0;

	//-----------------------------------------------------------------------------
	// 出力画像にアルファチャンネルを追加します。
	if (opts.m_UnitesAlpha)
	{
		dstDoc.channels.add();
	}

	//-----------------------------------------------------------------------------
	// アルファチャンネルの端のピクセルの塗りつぶしは
	// アルファチャンネルを結合する場合のみ有効なので
	// 両方の条件をみたすかどうかのフラグを変数に代入します。
	var fillsAlphaEdge = opts.m_UnitesAlpha && opts.m_FillsAlphaEdge;

	//-----------------------------------------------------------------------------
	// 入力フォルダのファイルについてループします。
	var ALPHA_CHAN_IDX = 3;
	var imageCountMax = opts.m_CountH * opts.m_CountV;
	var imageCount = 0;
	var someInputHasAlpha = false; // 入力画像のいずれかにアルファチャンネルが存在すれば true です。
	for (var iFile = 0; iFile < filePaths.length; ++iFile)
	{
		//-----------------------------------------------------------------------------
		// 出力画像の領域に納まらなければ処理を終了します。
		if (imageCount >= imageCountMax)
		{
			break;
		}

		//-----------------------------------------------------------------------------
		// 入力画像を開きます。
		var filePath = filePaths[iFile];
		var fileExt = filePath.GetExtensionFromFilePath();
		var doc;
		try
		{
			if (fileExt == "exr")
			{
				// EXR ファイルの場合、Unite Alpha が ON なら、アルファチャンネルとして読み取ります。
				OpenAsExr(filePath, opts.m_UnitesAlpha, true);
				doc = app.activeDocument;
			}
			else
			{
				doc = app.open(File(filePath));
			}
		}
		catch (e)
		{
			// Photoshop でロードできない ftx ファイルなら処理を中断します。
			if (fileExt == "ftxb" || fileExt == "ftxa")
			{
				return;
			}

			// Photoshop が対応している形式の画像ファイルでなければスキップします。
			//alert("Can't open the file: " + filePath + "\n" + e);
			continue;
		}

		//-----------------------------------------------------------------------------
		// イメージモードが RGB でなければ RGB にします。
		if (doc.mode != DocumentMode.RGB)
		{
			doc.changeMode(ChangeMode.RGB);
		}

		//-----------------------------------------------------------------------------
		// レイヤーが複数あれば表示レイヤーを結合します。
		if (doc.layers.length >= 2)
		{
			doc.mergeVisibleLayers();
		}

		//-----------------------------------------------------------------------------
		// 必要ならサイズを変更します。
		var srcW = doc.width.value;
		var srcH = doc.height.value;
		if (srcW != opts.m_PixelsW || srcH != opts.m_PixelsH)
		{
			doc.resizeImage(opts.m_PixelsW, opts.m_PixelsH, doc.resolution, ResampleMethod.BILINEAR);
			// BICUBIC, BICUBICSHARPER, BICUBICSMOOTHER, BILINEAR, NEARESTNEIGHBOR
		}

		//-----------------------------------------------------------------------------
		// アルファチャンネルがなければ追加します。
		if (doc.channels.length >= R_RGBA_COUNT)
		{
			someInputHasAlpha = true;
		}
		else if (opts.m_UnitesAlpha)
		{
			doc.channels.add();
			FillAllRgb(doc, 255, 255, 255);
		}

		//-----------------------------------------------------------------------------
		// 端 1 ピクセルを黒で塗りつぶします。
		if (opts.m_FillsColorEdge || fillsAlphaEdge)
		{
			SelectEdgePixels(doc);

			if (opts.m_FillsColorEdge)
			{
				doc.activeChannels = doc.componentChannels;
				doc.selection.fill(blackColor);
			}

			if (fillsAlphaEdge)
			{
				doc.activeChannels = [ doc.channels[ALPHA_CHAN_IDX] ];
				doc.selection.fill(blackColor);
			}
		}

		//-----------------------------------------------------------------------------
		// 出力画像にコンポーネントチャンネルをコピーします。
		// レイヤーに透明ピクセルが存在する場合に中央にペーストされないために、
		// 出力画像をレイヤーの bounds で選択してからペーストします。
		var ox =           (imageCount % opts.m_CountH) * opts.m_PixelsW;
		var oy = Math.floor(imageCount / opts.m_CountH) * opts.m_PixelsH;

		doc.selection.selectAll();
		doc.activeChannels = doc.componentChannels;
		var layer = doc.activeLayer;
		var bx0 = layer.bounds[0].value;
		var by0 = layer.bounds[1].value;
		var bx1 = layer.bounds[2].value;
		var by1 = layer.bounds[3].value;
		var isNoPixel = (bx0 == bx1 && by0 == by1);
		if (!isNoPixel)
		{
			doc.selection.copy(false); // merge: false

			app.activeDocument = dstDoc;
			dstDoc.activeChannels = dstDoc.componentChannels;
			bx0 += ox;
			by0 += oy;
			bx1 += ox;
			by1 += oy;
			//alert("dst region: " + imageCount + ": " + ox + " " + oy + ": " + bx0 + " " + by0 + " " + bx1 + " " + by1);
			var selRegion = Array(
				Array(bx0, by0), Array(bx1, by0),
				Array(bx1, by1), Array(bx0, by1), Array(bx0, by0));
			dstDoc.selection.select(selRegion);
			dstDoc.paste(false); // intoSelection: false
		}

		//-----------------------------------------------------------------------------
		// 出力画像にアルファチャンネルをコピーします。
		if (opts.m_UnitesAlpha)
		{
			app.activeDocument = doc;
			doc.activeChannels = [ doc.channels[ALPHA_CHAN_IDX] ];
			doc.selection.copy(false); // merge: false

			app.activeDocument = dstDoc;
			dstDoc.activeChannels = [ dstDoc.channels[ALPHA_CHAN_IDX] ];
			var x0 = ox;
			var y0 = oy;
			var x1 = x0 + opts.m_PixelsW;
			var y1 = y0 + opts.m_PixelsH;
			var selRegion = Array(
				Array(x0, y0), Array(x1, y0),
				Array(x1, y1), Array(x0, y1), Array(x0, y0));
			dstDoc.selection.select(selRegion);
			dstDoc.paste(false); // intoSelection: false
			dstDoc.selection.deselect();
			dstDoc.activeChannels = dstDoc.componentChannels;
		}

		//-----------------------------------------------------------------------------
		// 入力画像を閉じます。
		app.activeDocument = doc;
		doc.close(SaveOptions.DONOTSAVECHANGES);

		++imageCount;
	}

	//-----------------------------------------------------------------------------
	// 入力画像がなければエラーを表示します。
	app.activeDocument = dstDoc;
	if (imageCount == 0)
	{
		opts.m_IsSucceeded = false;
		ShowError("There is no image file in the Input Folder");
		return;
	}

	//-----------------------------------------------------------------------------
	// レイヤーが複数あれば表示レイヤーを結合します。
	if (dstDoc.layers.length >= 2)
	{
		dstDoc.mergeVisibleLayers();
	}

	//-----------------------------------------------------------------------------
	// アルファチャンネルを持つ入力画像がなく、
	// 端 1 ピクセルを黒で塗りつぶすのでなければ、
	// 出力画像のアルファチャンネルを削除します。
	if (!someInputHasAlpha && opts.m_UnitesAlpha && !opts.m_FillsAlphaEdge)
	{
		dstDoc.channels[ALPHA_CHAN_IDX].remove();
	}
}

//-----------------------------------------------------------------------------
//! @brief 結合します。
//!
//! @param[in] opts オプション設定です。
//!
//! @return 処理成功なら true、処理失敗なら false を返します。
//-----------------------------------------------------------------------------
function DoUnite(opts)
{
	//alert("do unite");

	//-----------------------------------------------------------------------------
	// 入力フォルダのファイル一覧を取得します。
	// 画像ファイル以外も含まれます。
	var inputFolder = Folder(opts.m_InputFolder);
	if (!inputFolder.exists)
	{
		ShowError("Input Folder is wrong");
		return false;
	}
	var filePaths = GetFilePathsInFolder(opts.m_InputFolder);
	if (filePaths.length == 0)
	{
		ShowError("There is no image file in the Input Folder");
		return false;
	}
	filePaths.sort(CompareStringCaseInsensitive);
	//alert("input: " + filePaths.length + "\n" + filePaths.join("\n"));

	//-----------------------------------------------------------------------------
	// 出力画像を作成します。
	var dstW = opts.m_PixelsW * opts.m_CountH;
	var dstH = opts.m_PixelsH * opts.m_CountV;
	var dstDoc = app.documents.add(dstW, dstH, 72, null, NewDocumentMode.RGB,
		DocumentFill.TRANSPARENT);

	//-----------------------------------------------------------------------------
	// 入力画像を出力画像に結合します。
	//UniteInputImages(dstDoc, opts, filePaths);
	dstDoc.suspendHistory("NintendoUniteImages", "UniteInputImages(dstDoc, opts, filePaths)");

	//-----------------------------------------------------------------------------
	// 処理失敗なら出力画像を閉じます。
	if (!opts.m_IsSucceeded)
	{
		dstDoc.close(SaveOptions.DONOTSAVECHANGES);
	}

	return opts.m_IsSucceeded;
}

//-----------------------------------------------------------------------------
//! @brief ダイアログの状態を更新します。
//!
//! @param[in,out] dialog ダイアログです。
//! @param[in] opts オプション設定です。
//-----------------------------------------------------------------------------
function UpdateDialog(dialog, opts)
{
	//-----------------------------------------------------------------------------
	// image pixels
	dialog.customWTxt.enabled = (dialog.pixelsWLst.selection.text == CUSTOM_ITEM);
	if (dialog.samePixelsChk.value)
	{
		dialog.pixelsHLst.enabled = false;
		dialog.customHTxt.enabled = false;
	}
	else
	{
		dialog.pixelsHLst.enabled = true;
		dialog.customHTxt.enabled = (dialog.pixelsHLst.selection.text == CUSTOM_ITEM);
	}

	dialog.customWTxt.visible = (dialog.pixelsWLst.selection.text == CUSTOM_ITEM);
	dialog.customHTxt.visible = (dialog.pixelsHLst.selection.text == CUSTOM_ITEM);

	//-----------------------------------------------------------------------------
	// image count
	dialog.customCountHTxt.enabled = (dialog.countHLst.selection.text == CUSTOM_ITEM);
	if (dialog.sameCountChk.value)
	{
		dialog.countVLst.enabled = false;
		dialog.customCountVTxt.enabled = false;
	}
	else
	{
		dialog.countVLst.enabled = true;
		dialog.customCountVTxt.enabled = (dialog.countVLst.selection.text == CUSTOM_ITEM);
	}

	dialog.customCountHTxt.visible = (dialog.countHLst.selection.text == CUSTOM_ITEM);
	dialog.customCountVTxt.visible = (dialog.countVLst.selection.text == CUSTOM_ITEM);

	//-----------------------------------------------------------------------------
	// fill alpha edge
	dialog.fillAlphaEdgeChk.enabled = dialog.uniteAlphaChk.value;
}

//-----------------------------------------------------------------------------
//! @brief ダイアログのピクセルの幅を高さにコピーします。
//!
//! @param[in,out] dialog ダイアログです。
//! @param[in,out] opts オプション設定です。
//-----------------------------------------------------------------------------
function CopyPixelsWToH(dialog, opts)
{
	dialog.pixelsHLst.items[dialog.pixelsWLst.selection.index].selected = true;
	opts.m_PixelsH = opts.m_PixelsW;
	dialog.customHTxt.text = String(opts.m_PixelsH);
}

//-----------------------------------------------------------------------------
//! @brief ダイアログの水平方向のイメージ数を垂直方向のイメージ数にコピーします。
//!
//! @param[in,out] dialog ダイアログです。
//! @param[in,out] opts オプション設定です。
//-----------------------------------------------------------------------------
function CopyCountHToV(dialog, opts)
{
	dialog.countVLst.items[dialog.countHLst.selection.index].selected = true;
	opts.m_CountV = opts.m_CountH;
	dialog.customCountVTxt.text = String(opts.m_CountV);
}

//-----------------------------------------------------------------------------
//! @brief ダイアログを表示します。
//!
//! @param[in,out] opts オプション設定です。
//!
//! @return OK がクリックされたら true、Cancel がクリックされたら false を返します。
//-----------------------------------------------------------------------------
function DoDialog(opts)
{
	//-----------------------------------------------------------------------------
	// create dialog
	var isJP = IsJapaneseUI();
	var dialog = new Window("dialog", "Nintendo Unite Images");
	dialog.orientation = 'row';
	dialog.alignChildren = 'top';
	dialog.margins = new Array(15, 10, 15, 10); // left, top, right, bottom
	dialog.spacing = 10;
	dialog.graphics.backgroundColor =
		dialog.graphics.newBrush(dialog.graphics.BrushType.THEME_COLOR, "appDialogBackground");	

	//-----------------------------------------------------------------------------
	// options group
	dialog.optionsGroup = dialog.add("group");
	dialog.optionsGroup.orientation = 'column';
	dialog.optionsGroup.alignChildren = 'left';
	dialog.optionsGroup.spacing = 5;

	//-----------------------------------------------------------------------------
	// input folder
	dialog.inputFolderPnl = dialog.optionsGroup.add("panel", undefined, "Input Folder");
	dialog.inputFolderPnl.orientation = 'column';
	dialog.inputFolderPnl.alignChildren = 'left';
	dialog.inputFolderPnl.alignment = 'fill';
	dialog.inputFolderPnl.spacing = 5;

	dialog.inputFolderGrp = dialog.inputFolderPnl.add("group");
	dialog.inputFolderGrp.orientation = 'row';
	dialog.inputFolderGrp.alignChildren = 'left';

	dialog.inputFolderTxt = dialog.inputFolderGrp.add("edittext", undefined, opts.m_InputFolder);
	dialog.inputFolderTxt.preferredSize.width = 330;
	dialog.inputFolderTxt.onChange = function()
	{
		if (!dialog.inputFolderTxt.text.IsValidFilePath())
		{
			alert("Input Folder is wrong");
			dialog.inputFolderTxt.text = opts.m_InputFolder;
		}
		else
		{
			opts.m_InputFolder = dialog.inputFolderTxt.text;
		}
	}

	dialog.inputFolderBrowseBtn = dialog.inputFolderGrp.add("button", undefined, "Browse...");
	dialog.inputFolderBrowseBtn.onClick = function()
	{
		var defaultFolder = dialog.inputFolderTxt.text;
		if (!Folder(defaultFolder).exists) defaultFolder = GetDefaultInputFolder();
		var selFolder = Folder(defaultFolder).selectDlg("Select Input Folder");
		if (selFolder != null)
		{
			dialog.inputFolderTxt.text = selFolder.fsName;
			opts.m_InputFolder = dialog.inputFolderTxt.text;
		}
		dialog.defaultElement.active = true;
	}
	dialog.inputFolderPnl.helpTip = dialog.inputFolderTxt.helpTip = (isJP) ?
		"結合する画像ファイルが存在するフォルダを指定します。" :
		"Specifies the folder in which image files to be merged can be found";
	dialog.inputFolderBrowseBtn.helpTip = (isJP) ? "結合する画像ファイルが存在するフォルダを選択するダイアログを表示します" :
		"Open the dialog box for selecting the folder in which image files to be merged can be found";

	//-----------------------------------------------------------------------------
	// image pixels
	dialog.pixelsPnl = dialog.optionsGroup.add("panel", undefined, "Number of Pixels in an Image");
	dialog.pixelsPnl.orientation = 'column';
	dialog.pixelsPnl.alignChildren = 'left';
	dialog.pixelsPnl.alignment = 'fill';
	dialog.pixelsPnl.spacing = 5;

	dialog.pixelsGrp = dialog.pixelsPnl.add("group");
	dialog.pixelsGrp.orientation = 'row';
	dialog.pixelsGrp.alignChildren = 'left';
	dialog.pixelsGrp.margins = new Array(0, 5, 0, 0);

	var pixelsItems = [ "8", "16", "32", "64", "128", "256", "512", CUSTOM_ITEM ];

	dialog.pixelsWLbl = dialog.pixelsGrp.add("statictext", undefined, "Width");
	dialog.pixelsWLbl.preferredSize.width = 60;
	dialog.pixelsWLbl.justify = "right";

	dialog.pixelsWLst = dialog.pixelsGrp.add("dropdownlist", undefined, pixelsItems);
	dialog.pixelsWLst.preferredSize.width = 70;
	var iPixelsW = FindValueInArray(pixelsItems, String(opts.m_PixelsW));
	if (iPixelsW == -1)
	{
		iPixelsW = pixelsItems.length - 1;
	}
	dialog.pixelsWLst.items[iPixelsW].selected = true;
	dialog.pixelsWLst.onChange = function()
	{
		var text = dialog.pixelsWLst.selection.text;
		if (text == CUSTOM_ITEM) text = dialog.customWTxt.text;
		var value = parseInt(text);
		if (value >= 1)
		{
			opts.m_PixelsW = value;
			dialog.customWTxt.text = String(opts.m_PixelsW);
			if (dialog.samePixelsChk.value)
			{
				CopyPixelsWToH(dialog, opts);
			}
		}
		UpdateDialog(dialog, opts);
	}

	dialog.pixelsHLbl = dialog.pixelsGrp.add("statictext", undefined, "Height");
	dialog.pixelsHLbl.preferredSize.width = 60;
	dialog.pixelsHLbl.justify = "right";

	dialog.pixelsHLst = dialog.pixelsGrp.add("dropdownlist", undefined, pixelsItems);
	dialog.pixelsHLst.preferredSize.width = 70;
	var iPixelsH = FindValueInArray(pixelsItems, String(opts.m_PixelsH));
	if (iPixelsH == -1)
	{
		iPixelsH = pixelsItems.length - 1;
	}
	dialog.pixelsHLst.items[iPixelsH].selected = true;
	dialog.pixelsHLst.onChange = function()
	{
		var text = dialog.pixelsHLst.selection.text;
		if (text == CUSTOM_ITEM) text = dialog.customHTxt.text;
		var value = parseInt(text);
		if (value >= 1)
		{
			opts.m_PixelsH = value;
			dialog.customHTxt.text = String(opts.m_PixelsH);
		}
		UpdateDialog(dialog, opts);
	}

	// same pixels check
	dialog.samePixelsLbl = dialog.pixelsGrp.add("statictext", undefined, "");

	dialog.samePixelsChk = dialog.pixelsGrp.add("checkbox", undefined, "Same Value");
	dialog.samePixelsChk.value = (opts.m_PixelsW == opts.m_PixelsH);
	dialog.samePixelsChk.onClick = function()
	{
		if (dialog.samePixelsChk.value)
		{
			CopyPixelsWToH(dialog, opts);
		}
		UpdateDialog(dialog, opts);
	}

	dialog.pixelsPnl.helpTip = (isJP) ? "1 つの画像の幅と高さを指定します" : "Specifies the height and width of a single image";
	dialog.pixelsWLbl.helpTip = dialog.pixelsWLst.helpTip = (isJP) ?
		"1 つの画像の幅をピクセル数で指定します。\ncustom : ピクセル数を直接入力" :
		"Specifies the width of a single image in terms of number of pixels. \ncustom : Enters pixel values directly";
	dialog.pixelsHLbl.helpTip = dialog.pixelsHLst.helpTip = (isJP) ?
		"1 つの画像の高さをピクセル数で指定します。\ncustom : ピクセル数を直接入力" :
		"Specifies the height of a single image in terms of number of pixels. \ncustom : Enters pixel values directly";
	dialog.samePixelsChk.helpTip = (isJP) ? "高さを自動的に幅と同じ値にするなら ON にします" :
		"Automatically uses the same value for height if ON is selected";

	//-----------------------------------------------------------------------------
	// custom image pixels
	dialog.customPixelsGrp = dialog.pixelsPnl.add("group");
	dialog.customPixelsGrp.orientation = 'row';
	dialog.customPixelsGrp.alignChildren = 'left';

	dialog.customWLbl = dialog.customPixelsGrp.add("statictext", undefined, "");
	dialog.customWLbl.preferredSize.width = 60;

	dialog.customWTxt = dialog.customPixelsGrp.add("edittext", undefined, String(opts.m_PixelsW));
	dialog.customWTxt.preferredSize.width = 70 - 15;
	dialog.customWTxt.onChange = function()
	{
		var value = parseInt(dialog.customWTxt.text);
		if (value >= 1)
		{
			opts.m_PixelsW = value;
			if (dialog.samePixelsChk.value)
			{
				CopyPixelsWToH(dialog, opts);
			}
		}
		dialog.customWTxt.text = String(opts.m_PixelsW);
	}

	dialog.customHLbl = dialog.customPixelsGrp.add("statictext", undefined, "");
	dialog.customHLbl.preferredSize.width = 15 + 60;

	dialog.customHTxt = dialog.customPixelsGrp.add("edittext", undefined, String(opts.m_PixelsH));
	dialog.customHTxt.preferredSize.width = 70 - 15;
	dialog.customHTxt.onChange = function()
	{
		var value = parseInt(dialog.customHTxt.text);
		if (value >= 1)
		{
			opts.m_PixelsH = value;
		}
		dialog.customHTxt.text = String(opts.m_PixelsH);
	}

	dialog.customWTxt.helpTip = (isJP) ? "1 つの画像の幅のピクセル数を入力します" : "Enters the width of a single image in terms of number of pixels";
	dialog.customHTxt.helpTip = (isJP) ? "1 つの画像の高さのピクセル数を入力します" : "Enters the height of a single image in terms of number of pixels";

	//-----------------------------------------------------------------------------
	// image count
	dialog.countPnl = dialog.optionsGroup.add("panel", undefined, "Number of Images in a Row (in a Column)");
	dialog.countPnl.orientation = 'column';
	dialog.countPnl.alignChildren = 'left';
	dialog.countPnl.alignment = 'fill';
	dialog.countPnl.spacing = 5;

	dialog.countGrp = dialog.countPnl.add("group");
	dialog.countGrp.orientation = 'row';
	dialog.countGrp.alignChildren = 'left';
	dialog.countGrp.margins = new Array(0, 5, 0, 0);

	var countItems = [ "1", "2", "4", "8", "16", "32", CUSTOM_ITEM ];

	dialog.countHLbl = dialog.countGrp.add("statictext", undefined, "Horizontal");
	dialog.countHLbl.preferredSize.width = 60;
	dialog.countHLbl.justify = "right";

	dialog.countHLst = dialog.countGrp.add("dropdownlist", undefined, countItems);
	dialog.countHLst.preferredSize.width = 70;
	var iCountH = FindValueInArray(countItems, String(opts.m_CountH));
	if (iCountH == -1)
	{
		iCountH = countItems.length - 1;
	}
	dialog.countHLst.items[iCountH].selected = true;
	dialog.countHLst.onChange = function()
	{
		var text = dialog.countHLst.selection.text;
		if (text == CUSTOM_ITEM) text = dialog.customCountHTxt.text;
		var value = parseInt(text);
		if (value >= 1)
		{
			opts.m_CountH = value;
			dialog.customCountHTxt.text = String(opts.m_CountH);
			if (dialog.sameCountChk.value)
			{
				CopyCountHToV(dialog, opts);
			}
		}
		UpdateDialog(dialog, opts);
	}

	dialog.countVLbl = dialog.countGrp.add("statictext", undefined, "Vertical");
	dialog.countVLbl.preferredSize.width = 60;
	dialog.countVLbl.justify = "right";

	dialog.countVLst = dialog.countGrp.add("dropdownlist", undefined, countItems);
	dialog.countVLst.preferredSize.width = 70;
	var iCountV = FindValueInArray(countItems, String(opts.m_CountV));
	if (iCountV == -1)
	{
		iCountV = countItems.length - 1;
	}
	dialog.countVLst.items[iCountV].selected = true;
	dialog.countVLst.onChange = function()
	{
		var text = dialog.countVLst.selection.text;
		if (text == CUSTOM_ITEM) text = dialog.customCountVTxt.text;
		var value = parseInt(text);
		if (value >= 1)
		{
			opts.m_CountV = value;
			dialog.customCountVTxt.text = String(opts.m_CountV);
		}
		UpdateDialog(dialog, opts);
	}

	// same count check
	dialog.sameCountLbl = dialog.countGrp.add("statictext", undefined, "");

	dialog.sameCountChk = dialog.countGrp.add("checkbox", undefined, "Same Value");
	dialog.sameCountChk.value = (opts.m_CountH == opts.m_CountV);
	dialog.sameCountChk.onClick = function()
	{
		if (dialog.sameCountChk.value)
		{
			CopyCountHToV(dialog, opts);
		}
		UpdateDialog(dialog, opts);
	}

	dialog.countPnl.helpTip = (isJP) ? "横方向と縦方向に配置する画像の個数を指定します" :
		"Specifies the number of images being layed out in the horizontal and vertical directions";
	dialog.countHLbl.helpTip = dialog.countHLst.helpTip = (isJP) ?
		"横方向に配置する画像の個数を指定します。\ncustom : 個数を直接入力" :
		"Specifies the number of images being layed out in the horizontal direction. \ncustom : Enters the number of images directly";
	dialog.countVLbl.helpTip = dialog.countVLst.helpTip = (isJP) ?
		"縦方向に配置する画像の個数を指定します。\ncustom : 個数を直接入力" :
		"Specifies the number of images being layed out in the vertical direction. \ncustom : Enters the number of images directly";
	dialog.sameCountChk.helpTip = (isJP) ? "縦方向の個数を自動的に横方向の個数と同じ値にするなら ON にします" :
		"Automatically uses the same number of images for the vertical direction if ON is selected";

	//-----------------------------------------------------------------------------
	// custom image count
	dialog.customCountGrp = dialog.countPnl.add("group");
	dialog.customCountGrp.orientation = 'row';
	dialog.customCountGrp.alignChildren = 'left';

	dialog.customCountHLbl = dialog.customCountGrp.add("statictext", undefined, "");
	dialog.customCountHLbl.preferredSize.width = 60;

	dialog.customCountHTxt = dialog.customCountGrp.add("edittext", undefined, String(opts.m_CountH));
	dialog.customCountHTxt.preferredSize.width = 70 - 15;
	dialog.customCountHTxt.onChange = function()
	{
		var value = parseInt(dialog.customCountHTxt.text);
		if (value >= 1)
		{
			opts.m_CountH = value;
			if (dialog.sameCountChk.value)
			{
				CopyCountHToV(dialog, opts);
			}
		}
		dialog.customCountHTxt.text = String(opts.m_CountH);
	}

	dialog.customCountVLbl = dialog.customCountGrp.add("statictext", undefined, "");
	dialog.customCountVLbl.preferredSize.width = 15 + 60;

	dialog.customCountVTxt = dialog.customCountGrp.add("edittext", undefined, String(opts.m_CountV));
	dialog.customCountVTxt.preferredSize.width = 70 - 15;
	dialog.customCountVTxt.onChange = function()
	{
		var value = parseInt(dialog.customCountVTxt.text);
		if (value >= 1)
		{
			opts.m_CountV = value;
		}
		dialog.customCountVTxt.text = String(opts.m_CountV);
	}

	dialog.customCountHTxt.helpTip = (isJP) ? "横方向に配置する画像の個数を入力します" :
		"Enters the number of images being layed out in the horizontal direction";
	dialog.customCountVTxt.helpTip = (isJP) ? "縦方向に配置する画像の個数を入力します" :
		"Enters the number of images being layed out in the vertical direction";

	//-----------------------------------------------------------------------------
	// unite alpha
	dialog.uniteAlphaPnl = dialog.optionsGroup.add("panel", undefined, "Process");
	dialog.uniteAlphaPnl.orientation = 'column';
	dialog.uniteAlphaPnl.alignChildren = 'left';
	dialog.uniteAlphaPnl.alignment = 'fill';
	dialog.uniteAlphaPnl.spacing = 5;

	dialog.uniteAlphaGrp = dialog.uniteAlphaPnl.add("group");
	dialog.uniteAlphaGrp.orientation = 'row';
	dialog.uniteAlphaGrp.alignChildren = 'left';
	dialog.uniteAlphaGrp.margins = new Array(0, 5, 0, 0);

	dialog.uniteAlphaChk = dialog.uniteAlphaGrp.add("checkbox", undefined, "Unite Alpha");
	dialog.uniteAlphaChk.value = opts.m_UnitesAlpha;
	dialog.uniteAlphaChk.onClick = function()
	{
		opts.m_UnitesAlpha = dialog.uniteAlphaChk.value;
		UpdateDialog(dialog, opts);
	}
	dialog.uniteAlphaChk.helpTip = (isJP) ? "アルファチャンネルを結合するなら ON にします" : "Unites alpha channels if ON is selected";

	//-----------------------------------------------------------------------------
	// fill edge
	dialog.fillEdgePnl = dialog.optionsGroup.add("panel", undefined, "Edge");
	dialog.fillEdgePnl.orientation = 'column';
	dialog.fillEdgePnl.alignChildren = 'left';
	dialog.fillEdgePnl.alignment = 'fill';
	dialog.fillEdgePnl.spacing = 5;

	dialog.fillEdgeGrp = dialog.fillEdgePnl.add("group");
	dialog.fillEdgeGrp.orientation = 'row';
	dialog.fillEdgeGrp.alignChildren = 'left';
	dialog.fillEdgeGrp.margins = new Array(0, 5, 0, 0);

	dialog.fillColorEdgeChk = dialog.fillEdgeGrp.add("checkbox", undefined, "Fill Color Edge with Black");
	dialog.fillColorEdgeChk.value = opts.m_FillsColorEdge;
	dialog.fillColorEdgeChk.onClick = function()
	{
		opts.m_FillsColorEdge = dialog.fillColorEdgeChk.value;
	}

	dialog.fillAlphaEdgeLbl = dialog.fillEdgeGrp.add("statictext", undefined, "");

	dialog.fillAlphaEdgeChk = dialog.fillEdgeGrp.add("checkbox", undefined, "Fill Alpha Edge with Black");
	dialog.fillAlphaEdgeChk.value = opts.m_FillsAlphaEdge;
	dialog.fillAlphaEdgeChk.onClick = function()
	{
		opts.m_FillsAlphaEdge = dialog.fillAlphaEdgeChk.value;
	}

	dialog.fillColorEdgeChk.helpTip = (isJP) ? "各画像のカラーチャンネルの端の 1 ピクセルを黒で塗りつぶすなら ON にします" :
		"Fills the end of each color channel of each image with a single black pixel if ON is selected";
	dialog.fillAlphaEdgeChk.helpTip = (isJP) ? "各画像のアルファチャンネルの端の 1 ピクセルを黒で塗りつぶすなら ON にします" :
		"Fills the end of the alpha channel of each image with a single black pixel if ON is selected";

	//-----------------------------------------------------------------------------
	// button
	dialog.btnGrp = dialog.add("group");
	dialog.btnGrp.orientation = 'column';
	dialog.btnGrp.alignChildren = 'left';
	dialog.btnGrp.margins = Array(0, 5, 0, 0);

	dialog.m_IsOkPressed = false;
	dialog.okBtn = dialog.btnGrp.add("button", undefined, "OK");
	dialog.okBtn.preferredSize.width = 120;
	dialog.okBtn.onClick = function()
	{
		dialog.m_IsOkPressed = true;
		dialog.close(true);
	}

	dialog.cancelBtn = dialog.btnGrp.add("button", undefined, "Cancel");
	dialog.cancelBtn.preferredSize.width = 120;
	dialog.cancelBtn.onClick = function()
	{
		dialog.close(false);
	}

	dialog.btnSpace1Lbl = dialog.btnGrp.add("statictext", undefined, "");

	dialog.helpBtn = dialog.btnGrp.add("button", undefined, "Help...");
	dialog.helpBtn.preferredSize.width = 85;
	dialog.helpBtn.onClick = function()
	{
		NintendoShowHelp("html/NW4F_UniteImages.html");
	}

	dialog.helpIdxBtn = dialog.btnGrp.add("button", undefined, "Help Index...");
	dialog.helpIdxBtn.preferredSize.width = 85;
	dialog.helpIdxBtn.onClick = function()
	{
		NintendoShowHelp("PhotoshopPlugin.html");
	}

	dialog.defaultElement = dialog.okBtn;
	dialog.cancelElement = dialog.cancelBtn;

	dialog.okBtn.helpTip     = (isJP) ? "結合した画像を作成します" : "Creates a united image";
	dialog.cancelBtn.helpTip = (isJP) ? "キャンセルします" : "Cancels";
	dialog.helpBtn.helpTip = (isJP) ? "画像結合プラグインのヘルプを表示します" :
		"Displays help on Unite Images Plug-in";
	dialog.helpIdxBtn.helpTip = (isJP) ? "ヘルプの索引を表示します" : "Displays help index";

	//-----------------------------------------------------------------------------
	// update state
	UpdateDialog(dialog, opts);

	//-----------------------------------------------------------------------------
	// show dialog
	dialog.center();
	if (!dialog.show()) // cancelled
	{
		return false;
	}
	return dialog.m_IsOkPressed;
}

//-----------------------------------------------------------------------------
//! @brief メイン関数です。
//-----------------------------------------------------------------------------
function main()
{
	//-----------------------------------------------------------------------------
	// test
	//var doc = app.activeDocument;
	//var layer = doc.activeLayer;
	//return;

	//-----------------------------------------------------------------------------
	// bring to front
	app.bringToFront();

	//-----------------------------------------------------------------------------
	// Photoshop のレジストリからオプション設定を取得します。
	var opts = new Object();
	InitOptions(opts);
	try // レジストリにオプションが存在しない場合にエラーになるので try を使用します。
	{
		var desc = app.getCustomOptions(OPTIONS_ID);
		GetOptions(opts, desc);
	}
	catch (e)
	{
	}

	//-----------------------------------------------------------------------------
	// アクションからオプション設定を取得します。
	GetOptions(opts, app.playbackParameters);

	//-----------------------------------------------------------------------------
	// オプションダイアログを表示します。
	//alert("display dialogs: " + app.displayDialogs + " " + app.playbackDisplayDialogs);
	if (app.playbackDisplayDialogs == DialogModes.ALL) // NO, ERROR, ALL
	{
		if (!DoDialog(opts))
		{
			return;
		}
	}

	//-----------------------------------------------------------------------------
	// 結合します。
	if (!DoUnite(opts))
	{
		return;
	}

	//-----------------------------------------------------------------------------
	// Photoshop のレジストリにオプション設定を記録します。
	app.putCustomOptions(OPTIONS_ID, CreateOptionsDesc(opts));

	//-----------------------------------------------------------------------------
	// アクションにオプション設定を設定します。
	app.playbackParameters = CreateOptionsDesc(opts);
}

// 定規の単位を pixels にして main を呼びます。
CallFunctionUnitsPixels(main);

