﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/manager/vi_Manager.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/oe.h>
#include "Graphics.h"

namespace nn {
    namespace vi {
        namespace dev {
            extern LayerId g_ExternalLayerId;
        }
    }
}

static void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::aligned_alloc(alignment, size);
}

static void NvFree(void* ptr, void* userPtr)
{
    NN_UNUSED(userPtr);
    std::free(ptr);
}

static void* NvReallocate(void* ptr, size_t size, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::realloc(ptr, size);
}

extern "C" void nnMain()
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );

    nn::vi::Initialize();
    nn::vi::ProxyName proxyName = { "dispmgr" };
    nn::vi::manager::DisplayManagerService viManager;
    viManager.Initialize(nn::vi::PolicyLevel_Composition, &proxyName);

    nn::vi::Display* pDefault;
    nn::vi::DisplayId defaultId;

    {
        nn::Result result = viManager.GetApplicationDisplayService()->OpenDefaultDisplay(&defaultId);
        NN_ASSERT(result.IsSuccess(), "Failed to open default display.");
        result = nn::vi::OpenDefaultDisplay(&pDefault);
    }

    nn::vi::LayerSettings settings;
    nn::vi::SetLayerSettingsDefaults(&settings);

    nn::vi::LayerId hdmiLayerId;
    nn::vi::Layer* pHdmiLayer;

    const int LayerWidth = 240;
    const int LayerHeight = 135;

    nn::applet::AppletResourceUserId aruid = nn::applet::GetAppletResourceUserId();

    {
        nn::Result result = viManager.GetManagerDisplayService()->CreateManagedLayer(&hdmiLayerId,
                                                                                     defaultId,
                                                                                     settings._storage[0],
                                                                                     aruid);
        NN_ASSERT(result.IsSuccess(), "Failed to create managed layer.");
        nn::vi::dev::g_ExternalLayerId = hdmiLayerId;
        result = nn::vi::CreateLayer(&pHdmiLayer, pDefault);
        NN_ASSERT(result.IsSuccess(), "Failed to open layer.");
    }

    nn::vi::LayerId lcdLayerId;
    nn::vi::Layer* pLcdLayer;

    {
        nn::Result result = viManager.GetManagerDisplayService()->CreateManagedLayer(&lcdLayerId,
                                                                                     defaultId,
                                                                                     settings._storage[0],
                                                                                     aruid);
        NN_ASSERT(result.IsSuccess(), "Failed to create managed layer.");
        nn::vi::dev::g_ExternalLayerId = lcdLayerId;
        result = nn::vi::CreateLayer(&pLcdLayer, pDefault);
        NN_ASSERT(result.IsSuccess(), "Failed to open layer.");
    }

    nn::vi::Display* pLcd;
    nn::vi::OpenDisplay(&pLcd, "Internal");

    nn::vi::SetDisplayPowerState(pLcd, nn::vi::PowerState_On);

    viManager.GetManagerDisplayService()->RemoveFromLayerStack(lcdLayerId, nn::vi::LayerStack_Default);
    viManager.GetManagerDisplayService()->AddToLayerStack(lcdLayerId, nn::vi::LayerStack_Lcd);

    nn::vi::SetDisplayLayerStack(pLcd, nn::vi::LayerStack_Lcd);

    Graphics g;

    // This is the red layer.
    g.RegisterLayer(pHdmiLayer, LayerWidth, LayerHeight, Color(1.f, 0.f, 0.f));

    // This is the white layer.
    g.RegisterLayer(pLcdLayer, LayerWidth, LayerHeight, Color(1.f, 1.f, 1.f));

    while (true)
    {
        g.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    nn::vi::Finalize();
}
