﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <nn/nn_Assert.h>
#include "MemoryPool.h"

MemoryPool::MemoryPool() NN_NOEXCEPT
    : m_Memory(nullptr)
    , m_Pool(nullptr)
    , m_Size(0)
{
}

void MemoryPool::Initialize(NVNdevice* pDevice, size_t size) NN_NOEXCEPT
{
    m_Memory = std::aligned_alloc(NVN_MEMORY_POOL_STORAGE_ALIGNMENT, size);

    NVNmemoryPoolBuilder builder;
    nvnMemoryPoolBuilderSetDefaults(&builder);
    nvnMemoryPoolBuilderSetDevice(&builder, pDevice);
    nvnMemoryPoolBuilderSetStorage(&builder, m_Memory, size);
    nvnMemoryPoolBuilderSetFlags(&builder,
                                 NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT |
                                 NVN_MEMORY_POOL_FLAGS_GPU_UNCACHED_BIT);

    m_Pool = new NVNmemoryPool;
    nvnMemoryPoolInitialize(m_Pool, &builder);

    m_Size = size;
}

MemoryPool::~MemoryPool() NN_NOEXCEPT
{
    if( m_Pool != nullptr )
    {
        nvnMemoryPoolFinalize(m_Pool);
        delete m_Pool;
    }

    if( m_Memory != nullptr )
    {
        std::free(m_Memory);
    }
}

NVNmemoryPool* MemoryPool::GetPool() const NN_NOEXCEPT
{
    return m_Pool;
}

size_t MemoryPool::GetSize() const NN_NOEXCEPT
{
    return m_Size;
}

size_t MemoryPool::GetAlignedOffset(size_t size, size_t alignment, size_t offset) const NN_NOEXCEPT
{
    void* ptr = static_cast<char*>(m_Memory) + offset;
    size_t space = GetSize() - offset;

    void* address = std::align(alignment, size, ptr, space);
    NN_ASSERT_NOT_NULL(address);

    return static_cast<char*>(address) - static_cast<char*>(ptr);
}
