﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include "Graphics.h"

// Please refer to the NvnTutorials for more details about the NVN functions.
// NvnTutorials seem to do this...
extern "C" PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char* name);

Graphics::Graphics() NN_NOEXCEPT
{
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>(nvnBootstrapLoader("nvnDeviceInitialize"));
    NN_ASSERT(pfnc_nvnDeviceInitialize != nullptr, "Bootstrap loading failed.");

    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(nvnBootstrapLoader("nvnDeviceGetProcAddress"));
    NN_ASSERT(pfnc_nvnDeviceGetProcAddress != nullptr, "Failed to query function address.");

    nvnLoadCProcs(nullptr, pfnc_nvnDeviceGetProcAddress);

    {
        NVNdeviceBuilder builder;
        nvnDeviceBuilderSetDefaults(&builder);
        NN_ABORT_UNLESS(nvnDeviceInitialize(&m_Device, &builder), "Failed to initialize device.");
    }

    nvnSyncInitialize(&m_CommandSync, &m_Device);

    {
        NVNqueueBuilder builder;
        nvnQueueBuilderSetDefaults(&builder);
        nvnQueueBuilderSetDevice(&builder, &m_Device);
        nvnQueueInitialize(&m_Queue, &builder);
    }
    nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device);

    int controlAlignment;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &controlAlignment);

    m_ControlMemory = std::aligned_alloc(controlAlignment, ControlMemoryAlignedSize);
    NN_ASSERT_NOT_NULL(m_ControlMemory);

    m_Pool.Initialize(&m_Device, 4096);
}

Graphics::~Graphics() NN_NOEXCEPT
{
    for( Window* window : m_Windows )
    {
        delete window;
    }

    nvnQueueFinalize(&m_Queue);
    nvnCommandBufferFinalize(&m_CommandBuffer);
    std::free(m_ControlMemory);
    nvnSyncFinalize(&m_CommandSync);
    nvnDeviceFinalize(&m_Device);
}

void Graphics::RegisterLayer(nn::vi::Layer* pLayer, int width, int height, Color backgroundColor) NN_NOEXCEPT
{
    // dynamically allocating to avoid copy constructor from vector
    m_Windows.push_back(new Window(&m_Device, pLayer, width, height, backgroundColor));
}

void Graphics::Update() NN_NOEXCEPT
{
    for( Window* window : m_Windows )
    {
        // Note: This is not the best approach when generating more than one layer.  Multiple command buffers
        //       (or not resetting the command buffer for each layer) may improve performance.
        int textureIndex;
        nvnQueueAcquireTexture(&m_Queue, window->GetWindow(), &textureIndex);


        nvnQueueFenceSync(&m_Queue, &m_CommandSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
        nvnQueueFlush(&m_Queue);
        // This may block here waiting for the previous layer's commands to complete.
        nvnSyncWait(&m_CommandSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_ControlMemory, ControlMemoryAlignedSize);
        nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_Pool.GetPool(), 0, m_Pool.GetSize());

        nvnCommandBufferBeginRecording(&m_CommandBuffer);
        NVNtexture* texture = window->GetTexture(textureIndex);
        nvnCommandBufferSetRenderTargets(&m_CommandBuffer, 1, &texture, nullptr, nullptr, nullptr);
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, window->GetTextureWidth(), window->GetTextureHeight());
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, window->GetBackgroundColorChannels(), NVN_CLEAR_COLOR_MASK_RGBA);
        NVNcommandHandle handle = nvnCommandBufferEndRecording(&m_CommandBuffer);

        nvnQueueSubmitCommands(&m_Queue, 1, &handle);

        nvnQueuePresentTexture(&m_Queue, window->GetWindow(), textureIndex);
    }
}
