﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testVi_Scene.h"

#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include "../Common/framework/Framework.h"
#include "../Common/panel/PanelRenderer.h"
#include "../Common/framework/Hid.h"

#if defined(NNT_VI_FW_DISPLAY_SYSTEM_INDIRECT_TESTSERVER)
#include <nnt/vi/testVi_TestServerProxy.h>
namespace framework{ namespace detail{
    extern nnt::vi::TestServiceProxy g_DisplayTestProxy;
}}
#endif


namespace nnt{ namespace vi{

    nn::os::Tick g_PrevTick;
    std::list<float> g_FpsHistory;

    Scene::Scene() NN_NOEXCEPT
        : m_FrameCount(0)
    {
        m_pPanelFrameCount = std::make_shared<PanelText>();
        m_pPanelFrameCount->SetTransparent(true);
        m_pPanelFrameCount->SetColor(nn::util::Color4f(1, 1, 1, 0.75));
        m_pPanelFrameCount->SetPosition(nn::util::Vector3f(0, 360, 0));
        m_pPanelFrameCount->SetSize(nn::util::MakeFloat2(1280, 360));
        m_pPanelFrameCount->SetTextColor(nn::util::Color4f(0, 0, 0, 1));
        m_pPanelFrameCount->SetTextPosition(500, 120);
        m_pPanelFrameCount->SetTextSize(30);

        m_pDelayChecker = std::make_shared<DelayChecker>();

        m_pColorPanelList.resize(12);
        for(int i = 0; i < 12; i++)
        {
            float xBase = 0;
            float yBase = 360;
            float w = 100;
            float h = 100;

            float x = xBase + w * (i % 4);
            float y = yBase + h * (i / 4);

            float r = ((i % 4) == 0) ? 0.5f : 0;
            float g = ((i % 4) == 1) ? 0.5f : 0;
            float b = ((i % 4) == 2) ? 0.5f : 0;
            float a = ((i / 4) == 0) ? 1 : (((i / 4) == 1) ? .5f : .25f);

            auto p = std::make_shared<PanelText>();
            p->SetTransparent(false);
            p->SetColor(nn::util::Color4f(r, g, b, a));
            p->SetPosition(nn::util::Vector3f(x, y, 0));
            p->SetSize(nn::util::MakeFloat2(w, h));

            m_pColorPanelList[i] = p;
        }

        m_FlipOffsetMicrosec = 0;
    }

    void Scene::Update()
    {
        float fps = 0;
        float fpsMin = 0;
        float fpsMax = 0;
        auto tick = nn::os::GetSystemTick();
        if(m_FrameCount > 0)
        {
            auto time = (tick - g_PrevTick).ToTimeSpan();
            fps = static_cast<float>(1000ll * 1000ll / time.GetMicroSeconds());
            g_FpsHistory.push_back(fps);
            if(g_FpsHistory.size() > 100)
            {
                g_FpsHistory.pop_front();
            }
            fpsMin = 9999;
            fpsMax = 0;
            for(auto v : g_FpsHistory)
            {
                fpsMin = std::min(fpsMin, v);
                fpsMax = std::max(fpsMax, v);
            }
        }
        g_PrevTick = tick;
        m_FrameCount++;

        auto buttons = Hid::GetButtonState();
        if(!buttons.isPressed.X && buttons.isTriggered.R)
        {
            m_FlipOffsetMicrosec += buttons.isPressed.Y ? 100 : 10;
            m_FlipOffsetMicrosec = std::min(m_FlipOffsetMicrosec, 10 * 1000 * 1000); // 10 sec
            framework::detail::g_DisplayTestProxy.GetService()->SetIndirectLayerProducerFlipOffset(0, nn::TimeSpan::FromMicroSeconds(m_FlipOffsetMicrosec));
        }
        if(!buttons.isPressed.X && buttons.isTriggered.L)
        {
            m_FlipOffsetMicrosec -= buttons.isPressed.Y ? 100 : 10;
            m_FlipOffsetMicrosec = std::max(m_FlipOffsetMicrosec, 0);
            framework::detail::g_DisplayTestProxy.GetService()->SetIndirectLayerProducerFlipOffset(0, nn::TimeSpan::FromMicroSeconds(m_FlipOffsetMicrosec));
        }
        if(buttons.isDown.A)
        {
            m_pDelayChecker->Activate();
        }

        char buffer[256] = {};
        nn::util::SNPrintf(buffer, sizeof(buffer),
            "IndirectProducer\n"
            "Frame: %d\n"
            "%.1ffps(%.1ffps-%.1ffps)\n"
            "offset: %dus\n",
            m_FrameCount,
            fps, fpsMin, fpsMax,
            m_FlipOffsetMicrosec
        );
        m_pPanelFrameCount->SetText(buffer);
        m_pPanelFrameCount->Update();

        m_pDelayChecker->Update();
    }

    void Scene::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int resourceLockIndex) NN_NOEXCEPT
    {
        PanelRenderer::Render(pCommandBuffer, m_pPanelFrameCount, Framework::GetScreenRectangle(), resourceLockIndex);
        PanelRenderer::Render(pCommandBuffer, m_pDelayChecker->GetPanel(), Framework::GetScreenRectangle(), resourceLockIndex);
        for(auto& e : m_pColorPanelList)
        {
            PanelRenderer::Render(pCommandBuffer, e, Framework::GetScreenRectangle(), resourceLockIndex);
        }
    }

}}
