﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt.h>
#include <nn/oe.h>
#include <nn/vi.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/buffer/vi_Buffer.h>
#include <nnt/viUtil/testVi_PowerFixture.h>

namespace
{
    const int TestCount = 128;
    const size_t GraphicsHeapSize = 8 * 1024 * 1024;

    void* NvAllocate(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);
        return std::aligned_alloc(alignment, size);
    }

    void NvFree(void* ptr, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);
        std::free(ptr);
    }

    void* NvReallocate(void* ptr, size_t size, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);
        return std::realloc(ptr, size);
    }

    class Blank : public nnt::vi::util::PowerFixture
    {
    public:
        static void SetUpTestCase()
        {
            nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
            m_GraphicsHeap = std::malloc(GraphicsHeapSize);
            nv::InitializeGraphics(m_GraphicsHeap, GraphicsHeapSize);
            nn::vi::Initialize();

            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDefault));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDefault));
        }

        static void TearDownTestCase()
        {
            nn::os::DestroySystemEvent(&vsync);
            nn::vi::CloseDisplay(pDefault);
            nn::vi::Finalize();
            nv::FinalizeGraphics();
            std::free(m_GraphicsHeap);
        }

        void FlipTest() NN_NOEXCEPT
        {
            nn::vi::Layer* pLayer;
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDefault));
            NN_UTIL_SCOPE_EXIT
            {
                nn::vi::DestroyLayer(pLayer);
            };

            nn::vi::buffer::BufferQueue queue;
            nn::vi::buffer::BufferInfo info;

            info.bufferCount = 2;
            info.format = nn::vi::PixelFormat_Rgba8888;
            info.width = 1920;
            info.height = 1080;

            NNT_ASSERT_RESULT_SUCCESS(queue.Initialize(pLayer, info));

            size_t alignment = queue.GetRequiredMemoryAlignment(info);
            size_t size = queue.GetRequiredMemorySize(info);

            void* scanBuffers[info.bufferCount];
            for( int i = 0; i < info.bufferCount; ++i )
            {
                scanBuffers[i] = nullptr;
            }

            NN_UTIL_SCOPE_EXIT
            {
                queue.Finalize();

                for( int i = 0; i < info.bufferCount; ++i )
                {
                    if( scanBuffers[i] != nullptr )
                    {
                        std::free(scanBuffers[i]);
                    }
                }
            };

            for( int i = 0; i < info.bufferCount; ++i )
            {
                scanBuffers[i] = std::aligned_alloc(alignment, size);
                ASSERT_NE(nullptr, scanBuffers[i]);
                queue.SetScanBuffer(i, scanBuffers[i], size);
            }

            for( int i = 0; i < TestCount; ++i )
            {
                nn::vi::buffer::BufferQueueHandle* handle;
                NNT_EXPECT_RESULT_SUCCESS(queue.DequeueBuffer(&handle));
                NNT_EXPECT_RESULT_SUCCESS(queue.QueueBuffer(handle));
            }
        }

        void VsyncTest() NN_NOEXCEPT
        {
            for( int i = 0; i < TestCount; ++i )
            {
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&vsync, nn::TimeSpan::FromSeconds(1)));
            }
        }
    protected:
        static nn::vi::Display* pDefault;
        static nn::vi::Display* pInternal;
        static nn::vi::Display* pExternal;

        static nn::os::SystemEventType vsync;

        void InitializeIcosa() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pInternal, "Internal"));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternal, "External"));

            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Off));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Off));
        }

        void InitializeCopper() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternal, "External"));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Off));
        }

        void InitializeHoag() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pInternal, "Internal"));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Off));
        }
    private:
        static void* m_GraphicsHeap;
    };

    nn::vi::Display* Blank::pDefault = nullptr;
    nn::vi::Display* Blank::pInternal = nullptr;
    nn::vi::Display* Blank::pExternal = nullptr;

    nn::os::SystemEventType Blank::vsync;
    void* Blank::m_GraphicsHeap = nullptr;
}

TEST_F(Blank, LcdVsync_Icosa)
{
    InitializeIcosa();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Blank));

    VsyncTest();
}

TEST_F(Blank, TvVsync_Icosa)
{
    InitializeIcosa();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Blank));

    VsyncTest();
}

TEST_F(Blank, LcdFlip_Icosa)
{
    InitializeIcosa();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Blank));

    FlipTest();
}

TEST_F(Blank, TvFlip_Icosa)
{
    InitializeIcosa();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Blank));

    FlipTest();
}

TEST_F(Blank, TvVsync_Copper)
{
    InitializeCopper();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Blank));

    VsyncTest();
}

TEST_F(Blank, TvFlip_Copper)
{
    InitializeCopper();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pExternal, nn::vi::PowerState_Blank));

    FlipTest();
}

TEST_F(Blank, LcdVsync_Hoag)
{
    InitializeHoag();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Blank));

    VsyncTest();
}

TEST_F(Blank, LcdFlip_Hoag)
{
    InitializeHoag();

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pInternal, nn::vi::PowerState_Blank));

    FlipTest();
}
