﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include <string>

#include "testFixtures.h"
#include "utils.h"

TEST_F(CurlTest, BasicRedirectTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://natf.com/redirect.php";

    // Prepare a file to write the response to
    nn::fs::FileHandle outputFile;
    const char *outputFileName = "sd:/redirect-response.xml";
    nResult = CreateFileForWriting(&outputFile, outputFileName);
    ASSERT_TRUE(nResult.IsSuccess());

    FileWriterInfo fwInfo;
    fwInfo.handle = outputFile;
    fwInfo.bytesWritten = 0;

    // Set up libcurl options
    cResult = curl_easy_setopt(cHandle, CURLOPT_COOKIELIST, "Set-Cookie: Flavor=Chocolate-Chip; path=/; domain=natf.com");
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &fwInfo);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, FileWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_AUTOREFERER, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_FOLLOWLOCATION, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    // Perform the request
    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Make sure we got the expected response
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    // Close the response file
    nn::fs::CloseFile(outputFile);

    // Load the reponse as an XML document
    pugi::xml_document xmlResponse;
    ASSERT_EQ(ReadFileResult_Success, ReadXmlDocFromFile(&xmlResponse, outputFileName));
    pugi::xml_node headers = xmlResponse.child("RequestHeaders");

    std::map<std::string, std::string> headerMap;
    for (pugi::xml_node header = headers.child("Header"); header; header = header.next_sibling("Header"))
    {
        const char *headerName = header.child("Name").child_value();
        const char *headerValue = header.child("Value").child_value();

        headerMap.insert(std::make_pair(headerName, headerValue));
    }

    // Verify the headers we sent
    ASSERT_STREQ(testUrl, headerMap["Referer"].c_str());
    ASSERT_STREQ("Flavor=Chocolate-Chip", headerMap["Cookie"].c_str());
}

TEST_F(CurlTest, MaxRedirectTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://natf.com/doubleredirect.php";

    cResult = curl_easy_setopt(cHandle, CURLOPT_MAXREDIRS, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_FOLLOWLOCATION, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    // Perform the request
    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_TOO_MANY_REDIRECTS, cResult);
}
