﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <new>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/xcd/xcd.h>
#include "NpadController.h"
#include "util.h"

// タグの読み書きをループ実行する場合に定義
//#define NNT_XCD_LOOP_MODE

// ResultMcuBusy の発生をテストする
#define NNT_XCD_CHECK_BUSY

// 連続 Read 要求をテストする
#define NNT_XCD_MULTIPLE_READ

namespace
{

const int CharacterIdSize = 3;
const int NumberingIdSize = 2;

/*
 * amiibo の ROM データの生値
 */
struct NN_ALIGNAS(1) AmiiboRomAreaRaw
{
    char characterId[CharacterIdSize];
    uint8_t nfpType;
    char numberingId[NumberingIdSize];
    uint8_t seriesId;
    uint8_t formatVersion;
};

bool g_IsDeviceConnected   = false;
bool g_IsNfcMonitoring     = false;
bool g_IsNfcInitialized    = false;
bool g_IsMcuVersionChecked = false;
nn::hid::NpadIdType g_ConnectedDeviceNpadId;

nn::os::SystemEventType     g_NfcBindEvent;
nn::os::SystemEventType     g_NfcEvent;
nn::os::SystemEventType     g_NfcDetectEvent;
nn::os::TimerEventType      g_TerminateEvent;

nn::os::MultiWaitHolderType g_NfcBindHolder;
nn::os::MultiWaitHolderType g_NfcHolder;
nn::os::MultiWaitHolderType g_NfcDetectHolder;
nn::os::MultiWaitHolderType g_TerminateHolder;

nn::os::MultiWaitType       g_MultiWait;

nn::os::ThreadType g_Thread;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[nn::os::ThreadStackAlignment * 2];

nn::os::Tick g_WriteStartTick;

nnt::xcd::INpadController* g_pController = nullptr;

// テストの状態
enum TestSequence
{
    WaitingForConnect,  // デバイス接続待ち
    WaitingCommand,     // コマンド入力待ち
    StateTransition,    // ステート遷移処理
    StartingTest,       // テスト開始処理
    RunningTest,        // テスト実行中
    Finished            // テスト終了
};

TestSequence g_CurrentSequence = TestSequence::WaitingForConnect;

#ifdef NNT_XCD_MULTIPLE_READ
const int MultipleReadCount = 15;

int g_RequestedReadCount = 0;
#endif

/*
 * NFC 機能のセットアップ
 */
void SetupNfc() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::hid::system::ActivateNfc(g_ConnectedDeviceNpadId));
    while (!nn::hid::system::IsNfcActivated(g_ConnectedDeviceNpadId))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetDataFormat(nn::xcd::PeriodicDataFormat_MCU, xcdHandle));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetNfcEvent(&g_NfcEvent, &g_NfcDetectEvent, xcdHandle));
    NN_LOG("[TeraTest] NFC events are registered\n");

    nn::os::InitializeMultiWaitHolder(&g_NfcHolder, &g_NfcEvent);
    nn::os::InitializeMultiWaitHolder(&g_NfcDetectHolder, &g_NfcDetectEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcHolder);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcDetectHolder);

    NN_LOG("[TeraTest] Set state to NFC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetMcuState(nn::xcd::McuState_Nfc, xcdHandle));

    g_IsNfcInitialized = true;
}

/*
 * NFC 機能の終了処理
 */
void FinalizeNfc() NN_NOEXCEPT
{
    if (!g_IsNfcInitialized)
    {
        return;
    }

    nn::os::UnlinkMultiWaitHolder(&g_NfcDetectHolder);
    nn::os::UnlinkMultiWaitHolder(&g_NfcHolder);
    nn::os::FinalizeMultiWaitHolder(&g_NfcDetectHolder);
    nn::os::FinalizeMultiWaitHolder(&g_NfcHolder);
    nn::os::DestroySystemEvent(&g_NfcDetectEvent);
    nn::os::DestroySystemEvent(&g_NfcEvent);
    g_IsNfcInitialized = false;
}

#ifndef NNT_XCD_LOOP_MODE
/*
 * テストループの終了
 */
void TerminateTest() NN_NOEXCEPT
{
    if (g_IsNfcInitialized)
    {
        NN_LOG("[TeraTest] Set state to Standby\n");

        auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::xcd::SetMcuState(nn::xcd::McuState_Standby, xcdHandle));
    }

    nn::os::StartOneShotTimerEvent(&g_TerminateEvent, nn::TimeSpan::FromSeconds(1));
    g_CurrentSequence = TestSequence::Finished;
}
#endif

/*
 * タグの検出開始
 */
void StartDetection(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_LOG("[TeraTest] Start discovery\n");
    nn::xcd::NfcDiscoveryParameter param = {};
    param.pollingMask       = nn::xcd::NfcPollingMask_All;
    param.activationTimeout = 0;    // タイムアウトしない
    param.discoveryPeriod   = 300;  // 300 msec 間隔でポーリング

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartNfcDiscovery(param, handle));

    NN_LOG("Please touch a tag!\n");
}

/*
 * タグの検出終了
 */
void StopDetection(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_LOG("[TeraTest] Stop discovery\n");

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StopNfcDiscovery(handle));
}

/*
 * amiibo の読み取り開始
 */
void ReadStartForAmiibo(nn::xcd::DeviceHandle handle, int blocks) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX(blocks, 1, 3);  // 受け付けるブロック数は 3 まで

    NN_LOG("[TeraTest] Read start\n");
    nn::xcd::NtagReadParameter param = {};
    param.timeoutMsec        = 3000;    // タイムアウト: 3000 msec
    param.isPasswordRequired = false;   // パスワード認証なし
    param.tagId.length       = 0;       // UID 指定なし
    param.blockCount         = blocks;  // ブロック数

    auto* addresses = param.addresses;
    addresses[0].startPage = 0x00;  // 以下 amiibo 用のアドレス情報
    addresses[0].endPage   = 0x39;
    if (blocks >= 2)
    {
        addresses[1].startPage = 0x3A;
        addresses[1].endPage   = 0x73;
    }
    if (blocks >= 3)
    {
        addresses[2].startPage = 0x74;
        addresses[2].endPage   = 0x86;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartNtagRead(param, handle));
}

/*
 * ブランク Type2 タグへの書き込み
 */
void WriteStartForBlankType2Tag(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_LOG("[TeraTest] Write start\n");
    nn::xcd::NtagWriteParameter param = {};
    param.timeoutMsec        = 5000;    // タイムアウト: 5000 msec
    param.isPasswordRequired = false;   // パスワード認証なし
    param.tagId.length       = 0;       // UID 指定なし
    param.type2TagVersion    = 0;       // Type2 タグバージョン 0

    param.ntagWriteData.isActivationNeeded = true;  // アクティベーションあり

    // アクティベーション情報の設定
    const nn::Bit8 ClearData[4] = { 0xFF, 0xFF, 0xFF, 0xFF };
    std::memcpy(
        param.ntagWriteData.clearData,
        ClearData,
        nn::xcd::NtagActivationAreaSize);
    const nn::Bit8 ActivationData[4] = { 0xA5, 0x00, 0x00, 0x00 };
    std::memcpy(
        param.ntagWriteData.activationData,
        ActivationData,
        nn::xcd::NtagActivationAreaSize);

    // 0x39 で挟んだ乱数データをユーザー領域の全域に書き込む
    const int UserAreaStartPage = 0x04;
    const int UserAreaPageCount = 0x7E;
    auto* blocks = param.ntagWriteData.dataBlocks;
    nn::Bit8 writeData[nn::xcd::Type2TagPageSize * UserAreaPageCount] = { 0x39 };
    srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));
    for (int i = 1; i < sizeof(writeData) - 1; ++i)
    {
        writeData[i] = static_cast<nn::Bit8>(::rand());
    }
    writeData[sizeof(writeData) - 1] = 0x39;

    // 書き込むデータを各ブロックに分割して格納
    int blockCount = (sizeof(writeData) + nn::xcd::NtagWriteBlockSizeMax - 1) /
        nn::xcd::NtagWriteBlockSizeMax;
    size_t remainSize = sizeof(writeData);
    for (int i = 0; i < blockCount; i++)
    {
        size_t blockSize = std::min(remainSize, nn::xcd::NtagWriteBlockSizeMax);
        std::memcpy(
            blocks[i].data,
            writeData + i * nn::xcd::NtagWriteBlockSizeMax,
            blockSize);
        blocks[i].startPageAddress = static_cast<uint8_t>(
            UserAreaStartPage + i * nn::xcd::NtagWriteBlockPageCountMax);
        blocks[i].dataSize = static_cast<uint8_t>(blockSize);
        remainSize -= blockSize;
    }
    param.ntagWriteData.blockCount = blockCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartNtagWrite(param, handle));
}

/*
 * 読み込んだタグが amiibo か否か判定
 */
bool IsAmiibo(const nn::xcd::NfcNtagData& tag) NN_NOEXCEPT
{
    // CC エリアで amiibo か否かを判定する
    const uint8_t CcPageAddress = 0x03;
    const uint8_t AmiiboCc[4]   = { 0xF1, 0x10, 0xFF, 0xEE };

#if 0
    NN_LOG("[TeraTest] Signature:\n  ");
    for (int i = 0; i < nn::xcd::Type2TagSignatureSize; ++i)
    {
        NN_LOG("%02X ", tag.signature[i]);
    }
    NN_LOG("\n");
#endif

    for (int i = 0; i < tag.blockCount; ++i)
    {
        const auto& block   = tag.readDataBlocks[i];
        const auto& address = block.address;
        if (address.startPage > CcPageAddress || address.endPage < CcPageAddress)
        {
            // CC エリアが含まれていないブロックは無視
            continue;
        }

        // CC エリアが一致するか
        size_t ccOffset = (CcPageAddress - address.startPage) * 4;
        const auto* cc  = reinterpret_cast<const char*>(&block.data[ccOffset]);

        return std::memcmp(cc, AmiiboCc, sizeof(AmiiboCc)) == 0;
    }

    return false;
}

/*
 * amiibo のデータをパースして表示
 */
void ParseAmiibo(const nn::xcd::NfcNtagData& tag) NN_NOEXCEPT
{
    if (!IsAmiibo(tag))
    {
        return;
    }

    NN_LOG("  [Amiibo info]\n");

    const uint8_t RomPageAddress = 0x15;

    for (int i = 0; i < tag.blockCount; ++i)
    {
        const auto& block   = tag.readDataBlocks[i];
        const auto& address = block.address;
        if (address.startPage > RomPageAddress || address.endPage < RomPageAddress)
        {
            continue;
        }

        // ROM 領域の表示
        size_t romOffset = (RomPageAddress - address.startPage) * 4;
        const auto* pRom = reinterpret_cast<const AmiiboRomAreaRaw*>(&block.data[romOffset]);

        NN_LOG("    Character Id  : %02X %02X %02X\n",
            static_cast<nn::Bit8>(pRom->characterId[0]),
            static_cast<nn::Bit8>(pRom->characterId[1]),
            static_cast<nn::Bit8>(pRom->characterId[2]));
        NN_LOG("    Numbering Id  : %02X %02X\n",
            static_cast<nn::Bit8>(pRom->numberingId[0]),
            static_cast<nn::Bit8>(pRom->numberingId[1]));
        NN_LOG("    Series Id     : %02X\n", pRom->seriesId);
        NN_LOG("    NFP type      : %02X\n", pRom->nfpType);
        NN_LOG("    Format version: %02X\n", pRom->formatVersion);
    }
}

/*
 * タグ検出・喪失時の処理
 */
void ProcessNfcDetectEvent() NN_NOEXCEPT
{
    if (!g_IsDeviceConnected)
    {
        return;
    }

    NN_LOG("[TeraTest] NFC detect event!!\n");

    auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

    nn::xcd::NfcInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::GetNfcInfo(&info, handle));
    NN_LOG("[TeraTest] reason: %d (%s)\n", info.reason, nnt::xcd::GetEventReasonName(info.reason));

    switch (info.reason)
    {
    case nn::xcd::NfcEventReason_Detected:
        {
            const auto& tagInfo = info.tagInfo;
            NN_LOG("[TeraTest] Detected tag info:\n");
            NN_LOG("  Protocol: %d (%s)\n", tagInfo.protocol, nnt::xcd::GetProtocolName(tagInfo.protocol));
            NN_LOG("  Tag type: %d (%s)\n", tagInfo.tagType, nnt::xcd::GetTagTypeName(tagInfo.tagType));
            NN_LOG("  UID: ");
            for (int i = 0; i < tagInfo.tagId.length; ++i)
            {
                NN_LOG("%02X ", static_cast<nn::Bit8>(tagInfo.tagId.uid[i]));
            }
            NN_LOG("\n");
            if (tagInfo.tagType == nn::xcd::NfcTagType_Type2)
            {
#ifdef NNT_XCD_MULTIPLE_READ
                ReadStartForAmiibo(handle, 1);
                g_RequestedReadCount++;
#else
                ReadStartForAmiibo(handle, 3);
#endif
            }
        }
        break;
    case nn::xcd::NfcEventReason_Deactivated:
        {
            NN_LOG("[TeraTest] Tag released\n");
            StopDetection(handle);

#ifdef NNT_XCD_CHECK_BUSY
            {
                // あえてコマンドを大量に積み、Busy が発生することを確認
                NN_LOG("[TeraTest] Checking nn::xcd::ResultMcuBusy occurrence...\n");
                nn::xcd::NfcDiscoveryParameter param = {};
                param.pollingMask       = nn::xcd::NfcPollingMask_All;
                param.activationTimeout = 0;
                param.discoveryPeriod   = 300;

                nn::Result result;
                for (int i = 0; i < 10; i++)
                {
                    result = nn::xcd::StartNfcDiscovery(param, handle);
                    if (nn::xcd::ResultMcuBusy::Includes(result))
                    {
                        NN_LOG("[TeraTest] nn::xcd::ResultMcuBusy is occurred\n");
                        break;
                    }
                }
                NN_ABORT_UNLESS(nn::xcd::ResultMcuBusy::Includes(result));

                nn::xcd::StopNfcDiscovery(handle);
            }
#endif  // ifdef NNT_XCD_CHECK_BUSY

#ifdef NNT_XCD_LOOP_MODE
            g_CurrentSequence = TestSequence::StartingTest;
#else
            TerminateTest();
#endif  // ifdef NNT_XCD_LOOP_MODE
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * NFC の検出・喪失以外のイベントに関する処理
 */
void ProcessNfcGeneralEvent() NN_NOEXCEPT
{
    if (!g_IsNfcInitialized)
    {
        return;
    }

    NN_LOG("[TeraTest] NFC event!!\n");

    auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

    nn::xcd::NfcInfo info = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::GetNfcInfo(&info, handle));
    NN_LOG("[TeraTest] reason: %d (%s)\n", info.reason, nnt::xcd::GetEventReasonName(info.reason));

    switch (info.reason)
    {
    case nn::xcd::NfcEventReason_ReadFinish:
        {
            const auto& tagData = info.ntagData;

            // 読み込みブロック数が要求通りであることを確認
#ifdef NNT_XCD_MULTIPLE_READ
            int expectedBlockCount = (g_RequestedReadCount - 1) % 3 + 1;
            NN_ABORT_UNLESS_EQUAL(tagData.blockCount, expectedBlockCount);

            if (g_RequestedReadCount < MultipleReadCount)
            {
                // 間髪入れず次の読み込みを開始
                NN_LOG("[TeraTest] Multiple read test #%d\n", g_RequestedReadCount);
                g_RequestedReadCount++;
                ReadStartForAmiibo(handle, (g_RequestedReadCount - 1) % 3 + 1);
                return;
            }
            g_RequestedReadCount = 0;
#else
            NN_ABORT_UNLESS_EQUAL(tagData.blockCount, 3);
#endif  // ifdef NNT_XCD_MULTIPLE_READ

            if (IsAmiibo(tagData))
            {
                NN_LOG("[TeraTest] Detected tag is amiibo.\n");
                ParseAmiibo(tagData);
            }
            else
            {
                // 非 amiibo は暫定的に blank と見なす
                NN_LOG("[TeraTest] Detected tag is not amiibo.\n");
                WriteStartForBlankType2Tag(handle);
                g_WriteStartTick = nn::os::GetSystemTick();
            }
        }
        break;
    case nn::xcd::NfcEventReason_WriteFinish:
        {
            NN_LOG("[TeraTest] Write finished!\n");

            auto diffTime = (nn::os::GetSystemTick() - g_WriteStartTick).ToTimeSpan();
            NN_LOG("  Write time: %d msec\n", diffTime.GetMilliSeconds());
        }
        break;
    case nn::xcd::NfcEventReason_Error:
        {
            NN_LOG("[TeraTest] An error has occurred\n");
            NN_LOG("  Result code: %d\n", info.errorInfo.resultCode);

            // エラー時はタグ検出を止める
            //NN_ABORT_UNLESS_RESULT_SUCCESS(
            //    nn::xcd::StopNfcDiscovery(handle));
            //NN_ABORT("ERROR %d\n", info.errorInfo.resultCode);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * 現在のシーケンスに応じた処理
 */
void ProcessSequence() NN_NOEXCEPT
{
    if (g_pController != nullptr)
    {
        g_pController->Update();
    }

    switch (g_CurrentSequence)
    {
    case TestSequence::WaitingCommand:
        {
            NN_ABORT_UNLESS_NOT_NULL(g_pController);

            if (g_pController->IsTriggered(nn::hid::NpadButton::A::Mask))
            {
                // A ボタンで開始
                SetupNfc();
                g_CurrentSequence = TestSequence::StateTransition;
            }
            else if (g_pController->IsTriggered(
                nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask))
            {
                // R+ZR ボタンで終了
                TerminateTest();
            }

            return;
        }
    case TestSequence::StateTransition:
        {
            NN_ABORT_UNLESS_NOT_NULL(g_pController);

            auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
            nn::xcd::McuState state;
            nn::xcd::GetMcuState(&state, handle);
            if (state == nn::xcd::McuState_Nfc)
            {
                // NFC ステートに遷移したらテストを開始
                g_CurrentSequence = TestSequence::StartingTest;
            }
            else if (!g_IsMcuVersionChecked)
            {
                // Tera MCU FW のバージョンを確認
                nn::xcd::McuVersionDataForNfc version;
                auto result = nn::xcd::GetMcuVersionForNfc(&version, handle);
                if (nn::xcd::ResultMcuHardwareError::Includes(result))
                {
                    NN_LOG("ERROR: Tera MCU is broken!\n");
                }
                else if (nn::xcd::ResultMcuFirmwareCorrupted::Includes(result))
                {
                    NN_LOG("ERROR: Tera MCU FW is corrupted. Need to update.\n");
                }

                if (!nn::xcd::ResultMcuVersionNotAvailable::Includes(result))
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    NN_LOG("Tera MCU FW Version: %d.%d\n", version.major, version.minor);
                    g_IsMcuVersionChecked = true;
                }
            }
            return;
        }
    case TestSequence::StartingTest:
        {
            NN_ABORT_UNLESS_NOT_NULL(g_pController);

            auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
            StartDetection(handle);
            g_CurrentSequence = TestSequence::RunningTest;
            return;
        }
    case TestSequence::WaitingForConnect:
    case TestSequence::RunningTest:
        {
            // R+ZR で終了
            if (g_pController != nullptr &&
                g_pController->IsTriggered(
                    nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask))
            {
                TerminateTest();
            }

            return;
        }
    case TestSequence::Finished:
        {
            // 何もしない
            return;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * デバイスの電池残量を確認
 */
nn::Result CheckDeviceBattery() NN_NOEXCEPT
{
    auto result = nn::xcd::CheckNfcDevicePower(nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId));
    if (result.IsSuccess())
    {
        NN_LOG("%s: Device battery is enough!!\n\n", NN_CURRENT_FUNCTION_NAME);
        return result;
    }

    if (nn::xcd::ResultLowBattery::Includes(result))
    {
        NN_LOG("%s: Low battery!!\n", NN_CURRENT_FUNCTION_NAME);
    }
    else
    {
        NN_LOG("%s: Unexpected result\n", NN_CURRENT_FUNCTION_NAME);
    }

    return result;
}

/*
 * デバイスの接続状態を更新
 */
void UpdateNfcDeviceConnection() NN_NOEXCEPT
{
    // デバイスのハンドルを取得
    int count = nn::hid::system::GetNpadsWithNfc(&g_ConnectedDeviceNpadId, 1);
    if (count > 0)
    {
        if (g_IsDeviceConnected)
        {
            return;
        }

        g_IsDeviceConnected   = true;
        g_IsMcuVersionChecked = false;

        // 電池残量確認
        NN_ABORT_UNLESS_RESULT_SUCCESS(CheckDeviceBattery());

        // XXX: NFC デバイス接続イベント通知直後はスタイルが反映されていない場合があるので、
        //      反映されるのを待つための W/A
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        // 接続されたデバイスに応じたコントローラクラスを作成始
        g_pController = nnt::xcd::CreateNpadController(g_ConnectedDeviceNpadId);
        g_pController->Initialize();

        NN_LOG("  A    : Start NFC access\n");
        NN_LOG("  R+ZR : Exit\n");

        g_CurrentSequence = TestSequence::WaitingCommand;
    }
    else if (g_IsDeviceConnected)
    {
        if (g_pController != nullptr)
        {
            // 振動を停止
            g_pController->Finalize();
            g_pController->~INpadController();
            g_pController = nullptr;
        }

        FinalizeNfc();

        g_IsDeviceConnected = false;
        g_CurrentSequence = TestSequence::WaitingForConnect;

        NN_LOG("Device is disconnected.\n");
    }
}

/*
 * シグナルされたイベントに応じた処理
 */
void HandleEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pHolder);

    if (pHolder == &g_NfcBindHolder)
    {
        nn::os::ClearSystemEvent(&g_NfcBindEvent);
        UpdateNfcDeviceConnection();
    }
    else if (pHolder == &g_NfcDetectHolder)
    {
        // タグの検知・喪失
        nn::os::ClearSystemEvent(&g_NfcDetectEvent);
        ProcessNfcDetectEvent();
    }
    else if (pHolder == &g_NfcHolder)
    {
        // 他の NFC 関連イベント
        nn::os::ClearSystemEvent(&g_NfcEvent);
        ProcessNfcGeneralEvent();
    }
}

/*
 * NFC のイベントモニタスレッド
 */
void NfcMonitoringThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    while (g_IsNfcMonitoring)
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::TimedWaitAny(
            &g_MultiWait,
            nn::TimeSpan::FromMilliSeconds(10));

        if (pHolder == &g_TerminateHolder)
        {
            g_IsNfcMonitoring = false;
        }
        else if (pHolder != nullptr && g_CurrentSequence != TestSequence::Finished)
        {
            HandleEvent(pHolder);
        }
    }
}

/*
 * 初期化処理
 */
void InitializeTest() NN_NOEXCEPT
{
    nnt::xcd::InitializeNpad();

    nn::os::InitializeMultiWait(&g_MultiWait);

    // 中断用イベント
    nn::os::InitializeTimerEvent(&g_TerminateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TerminateHolder, &g_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TerminateHolder);

    // NFC 用イベント
    nn::hid::system::BindNfcDeviceUpdateEvent(&g_NfcBindEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_NfcBindHolder, &g_NfcBindEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcBindHolder);

    g_IsNfcMonitoring = true;
    nn::os::CreateThread(&g_Thread, NfcMonitoringThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_Thread);

    // 既に接続済みの場合があるのでチェックする
    UpdateNfcDeviceConnection();
}

/*
 * 終了処理
 */
void FinalizeTest() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);

    if (g_IsDeviceConnected)
    {
        // NFC の使用終了
        nn::hid::system::DeactivateNfc(g_ConnectedDeviceNpadId);

        // データフォーマットを戻す
        auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
        nn::xcd::SetDataFormat(nn::xcd::PeriodicDataFormat_Basic, handle);

        g_pController->Finalize();
        g_pController->~INpadController();

        FinalizeNfc();

        g_IsDeviceConnected = false;
    }

    nn::os::UnlinkMultiWaitHolder(&g_NfcBindHolder);
    nn::os::UnlinkMultiWaitHolder(&g_TerminateHolder);
    nn::os::FinalizeMultiWaitHolder(&g_NfcBindHolder);
    nn::os::FinalizeMultiWaitHolder(&g_TerminateHolder);
    nn::os::DestroySystemEvent(&g_NfcBindEvent);
    nn::os::FinalizeTimerEvent(&g_TerminateEvent);

    nn::os::FinalizeMultiWait(&g_MultiWait);
}

}  // anonymous

extern "C" void nnMain()
{
    NN_LOG("XCD TeraNfcManualTest tool starts.\n");
    NN_LOG("\n");

    InitializeTest();

    while (g_IsNfcMonitoring)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));

        ProcessSequence();
    }

    FinalizeTest();

    NN_LOG("Finished\n");
}
