﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ClusteringModeTestTask.h"
#include <test_ConsoleFrameworkOpenCv.h>
#include <test_CvUtil.h>
#include <nn/xcd/xcd.h>
#include <nn/os/os_Tick.h>

namespace
{

    const CvScalar ColorRed = {36, 28, 255};
    const CvScalar ColorWhite    = { 255,255,255, 255 };
    const CvScalar ColorGray    = { 128,128,128, 255 };
    const CvScalar ColorCyan     = { 239,174,  0, 255 };
    const CvScalar ColorMagenta     = { 255,0,  255, 255 };
    const CvScalar ColorYellow     = {0, 255, 255, 255 };
    const CvScalar ColorGreen   = {0, 255, 0, 255 };

    void DrawLine(CvArr* pImage, int x1, int y1, int x2, int y2, CvScalar color, int width)
    {
        CvPoint p1 = cvPoint(x1, y1);
        CvPoint p2 = cvPoint(x2, y2);
        cvLine(pImage, p1, p2, color, width);
    }

    void DrawCircle(CvArr* pImage, int ox, int oy, int radius, CvScalar color, int width)
    {
        CvPoint p = cvPoint(ox, oy);
        cvCircle(pImage, p, radius, color, width);
    }

    const int CLUSTERING_PIXCNT_SIZE_MAX = 65535;

    const char* OUTPUT_IMAGE_FILE_NAME = "%s\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const char* OUTPUT_CLUSTERING_FILE_NAME = "%s\\%s_%s.csv";
    FILE* fp;
    const size_t FILE_NAME_SIZE = 256;
    const size_t LINE_SIZE = 4096;
}

const char* ClusteringModeTestTask::MENU_ITEM_CLUSTERING_NAMES[MENU_ITEM_CLUSTERING_NUM] =
{
    "WOI",
    "PixCntMin",
    "PixCntMax",
    "IntensityMin"
};

//============================================================
void ClusteringModeTestTask::DoInitializeCore()
{
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Clustering))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Clustering;
    // 内部パラメータの初期化
    SetDefaultClusteringConfig(m_Config);
    SetDefaultClusteringConfig(m_CurrentConfig);
} //ClusteringModeTestTask::DoInitializeCore()

//============================================================
void ClusteringModeTestTask::DoFinalizeCore()
{
    m_pDriver->SetSixAxisSampling(false);
    m_pDriver->SetLraSending(false);
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Ready))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Ready;
} //ClusteringModeTestTask::DoFinalizeCore()

//============================================================
void ClusteringModeTestTask::DoCalcCore(IplImage* pImage, IplImage* pDstImage)
{
    NN_UNUSED(pImage);
    NN_UNUSED(pDstImage);

    m_SamplingResult = m_pDriver->GetClusteringData(&m_ImageHeader, &m_State);
} //ClusteringModeTestTask::DoCalc()

//============================================================
void ClusteringModeTestTask::DoDrawCore(IplImage* pImage)
{
    cvSetZero(pImage);
    DrawClustering(m_State, pImage);
    if (m_IsLogEnabled)
    {
        SaveData(pImage, m_State);
    }
} //ClusteringModeTestTask::DoDraw()


void ClusteringModeTestTask::DrawClustering( nn::xcd::IrClusteringProcessorState& clusteringProcessorState, IplImage* pResultImage)
{
    int offsetX = 0;
    int offsetY = 0;

    nn::xcd::IrRect woi = {0, 0, 320, 240};
    int woiRightX = woi.x + woi.width - 1;
    int woiLeftX = woi.x;
    int woiUpY = woi.y;
    int woiDownY = woi.y + woi.height - 1;

    // WOI の表示
    DrawLine(pResultImage, woiLeftX, woiUpY, woiLeftX, woiDownY, ColorYellow, 1);
    DrawLine(pResultImage, woiLeftX, woiDownY, woiRightX, woiDownY, ColorYellow, 1);
    DrawLine(pResultImage, woiRightX, woiDownY, woiRightX, woiUpY, ColorYellow, 1);
    DrawLine(pResultImage, woiRightX, woiUpY, woiLeftX, woiUpY, ColorYellow, 1);

    // 検出されたオブジェクトを表示する
    for (int i = 0; i < clusteringProcessorState.objectCount; i++)
    {
        const nn::xcd::IrClusteringData& result = clusteringProcessorState.objects[i];

        DrawCircle(pResultImage,
            offsetX + static_cast<int>(result.centroid.x),
            offsetY + static_cast<int>(result.centroid.y),
            2, ColorGreen, -1);

        int leftX = offsetX + result.bound.x;
        int rightX = offsetX + result.bound.x + result.bound.width - 1;
        int upY = offsetY + result.bound.y;
        int downY = offsetY + result.bound.y + result.bound.height - 1;
        DrawLine(pResultImage, leftX, upY, leftX, downY, ColorRed, 2);
        DrawLine(pResultImage, leftX, downY, rightX, downY, ColorRed, 2);
        DrawLine(pResultImage, rightX, downY, rightX, upY, ColorRed, 2);
        DrawLine(pResultImage, rightX, upY, leftX, upY, ColorRed, 2);
    }
}

//============================================================
void ClusteringModeTestTask::DoDrawMenuCore(IplImage* pImage)
{
    const CvSize FONT_SIZE = cvSize(16, 16);
    //const char* WOI_NAMES[] = {"QVGA", "QQVGA", "QQQVGA"};

    for (s32 i = MENU_ITEM_NUM; i < (MENU_ITEM_NUM + MENU_ITEM_CLUSTERING_NUM); i++)
    {
        s32 tx = 2 * FONT_SIZE.width;
        s32 ty = i * FONT_SIZE.height;

        m_TextWriter.SetFontColor((m_Cursor == i) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);
        m_TextWriter.PutText(pImage, 0, ty, "%s", (m_Cursor == i) ? ">" : " ");

        //if (i == (MENU_ITEM_CLUSTERING_WOI + MENU_ITEM_NUM))
        //{
        //    m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_CLUSTERING_NAMES[MENU_ITEM_CLUSTERING_WOI], m_Config.woi);
        //}
        if (i == (MENU_ITEM_CLUSTERING_PIX_CNT_MIN + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_CLUSTERING_NAMES[MENU_ITEM_CLUSTERING_PIX_CNT_MIN], m_Config.pixCntMin);
        }
        if (i == (MENU_ITEM_CLUSTERING_PIX_CNT_MAX + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_CLUSTERING_NAMES[MENU_ITEM_CLUSTERING_PIX_CNT_MAX], m_Config.pixCntMax);
        }
        if (i == (MENU_ITEM_CLUSTERING_INTENSITY_MIN + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_CLUSTERING_NAMES[MENU_ITEM_CLUSTERING_INTENSITY_MIN], m_Config.intensityMin);
        }
    }
} //ClusteringModeTestTask::DoDraw()


//============================================================
void ClusteringModeTestTask::DoCalcMenuCore(int* pMenuCnt)
{
    static nn::os::Tick keyPressedTime = nn::os::Tick();
    static int counter = 0;
    s32 keycode = GetKeycode();
    if (m_Cursor == (MENU_ITEM_CLUSTERING_WOI + MENU_ITEM_NUM))
    {
        //if (keycode == KEY_CODE_LEFT)
        //{
        //    m_Config.woi = static_cast<SizeFormat>(m_Format - 1);
        //    if (m_Format < 0)
        //    {
        //        m_Format = static_cast<SizeFormat>(SIZE_FORMAT_MAX_NUM - 1);
        //    }
        //}
        //if (keycode == KEY_CODE_RIGHT)
        //{
        //    m_Format= static_cast<SizeFormat>(m_Format + 1);
        //    if (m_Format >= SIZE_FORMAT_MAX_NUM)
        //    {
        //        m_Format = static_cast<SizeFormat>(0);
        //    }
        //}
    }
    if (m_Cursor == (MENU_ITEM_CLUSTERING_PIX_CNT_MIN + MENU_ITEM_NUM))
    {
        if ((nn::os::GetSystemTick() - keyPressedTime).ToTimeSpan().GetMilliSeconds() > 200)
        {
            counter = 0;
        }

        if ((keycode == KEY_CODE_LEFT) && (counter > 100))
        {
            m_Config.pixCntMin -= 100;
            if (m_Config.pixCntMin < 0)
            {
                m_Config.pixCntMin = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if ((keycode == KEY_CODE_LEFT) && (counter > 50))
        {
            m_Config.pixCntMin -= 10;
            if (m_Config.pixCntMin < 0)
            {
                m_Config.pixCntMin = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_LEFT)
        {
            m_Config.pixCntMin--;
            if (m_Config.pixCntMin < 0)
            {
                m_Config.pixCntMin = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
        if (keycode == KEY_CODE_RIGHT && (counter > 100))
        {
            m_Config.pixCntMin += 100;
            if (m_Config.pixCntMin >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMin = 0;
            }
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_RIGHT && (counter > 50))
        {
            m_Config.pixCntMin += 10;
            if (m_Config.pixCntMin >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMin = 0;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.pixCntMin++;
            if (m_Config.pixCntMin >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMin = 0;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
    }
    if (m_Cursor == (MENU_ITEM_CLUSTERING_PIX_CNT_MAX + MENU_ITEM_NUM))
    {
        if ((nn::os::GetSystemTick() - keyPressedTime).ToTimeSpan().GetMilliSeconds() > 200)
        {
            counter = 0;
        }

        if ((keycode == KEY_CODE_LEFT) && (counter > 100))
        {
            m_Config.pixCntMax -= 100;
            if (m_Config.pixCntMax < 0)
            {
                m_Config.pixCntMax = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if ((keycode == KEY_CODE_LEFT) && (counter > 50))
        {
            m_Config.pixCntMax -= 10;
            if (m_Config.pixCntMax < 0)
            {
                m_Config.pixCntMax = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_LEFT )
        {
            m_Config.pixCntMax--;
            if (m_Config.pixCntMax < 0)
            {
                m_Config.pixCntMax = CLUSTERING_PIXCNT_SIZE_MAX;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }

        if (keycode == KEY_CODE_RIGHT && (counter > 100))
        {
            m_Config.pixCntMax += 100;
            if (m_Config.pixCntMax >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMax = 0;
            }
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_RIGHT && (counter > 50))
        {
            m_Config.pixCntMax += 10;
            if (m_Config.pixCntMax >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMax = 0;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
        else if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.pixCntMax++;
            if (m_Config.pixCntMax >= CLUSTERING_PIXCNT_SIZE_MAX)
            {
                m_Config.pixCntMax = 0;
            }
            counter++;
            keyPressedTime = nn::os::GetSystemTick();
        }
    }
    if (m_Cursor == (MENU_ITEM_CLUSTERING_INTENSITY_MIN + MENU_ITEM_NUM))
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config.intensityMin--;
            if (m_Config.intensityMin < 0)
            {
                m_Config.intensityMin = 255;
            }
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.intensityMin++;
            if (m_Config.intensityMin >= 255)
            {
                m_Config.intensityMin = 0;
            }
        }
    }
    *pMenuCnt = MENU_ITEM_CLUSTERING_NUM;
}//NOLINT(readability/fn_size)

void ClusteringModeTestTask::DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
{
    //if (m_Config.woi != m_CurrentConfig.woi)
    //{
    //    //WOI 変更の修正 (TO BE IMPLEMENTED)
    //    setting.registerBlock[index].bankId = 1;
    //    setting.registerBlock[index].address = 0x30;
    //    setting.registerBlock[index].data = 0x00;
    //    index++;
    //    setting.registerCount = index;
    //}
    if (m_Config.pixCntMin != m_CurrentConfig.pixCntMin)
    {
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x21;
        setting.registerBlock[index].data = (m_Config.pixCntMin & 0x000000FF);
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x22;
        setting.registerBlock[index].data = ((m_Config.pixCntMin >> 8) & 0x000000FF);
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x23;
        setting.registerBlock[index].data = ((m_Config.pixCntMin >> 16) & 0x000000FF);
        index++;
        setting.registerCount = index;
    }
    if (m_Config.pixCntMax != m_CurrentConfig.pixCntMax)
    {
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x24;
        setting.registerBlock[index].data = (m_Config.pixCntMax & 0x000000FF);
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x25;
        setting.registerBlock[index].data = ((m_Config.pixCntMax >> 8) & 0x000000FF);
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x26;
        setting.registerBlock[index].data = ((m_Config.pixCntMax >> 16) & 0x000000FF);
        index++;
        setting.registerCount = index;
    }
    if (m_Config.intensityMin != m_CurrentConfig.intensityMin)
    {
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x20;
        setting.registerBlock[index].data = static_cast<uint8_t>(m_Config.intensityMin);
        index++;
        setting.registerCount = index;
    }

    m_CurrentConfig = m_Config;
}

void ClusteringModeTestTask::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    // Clusteringモード設定時のデフォルト設定
    config.digitalGain = GAIN_X2;
    config.exposureTimeUsec = 200;
    config.irLedMode = IR_LED_BOTH;
    config.irLedIntensity = 16;
    config.postProcess = POSTPROCESS_NONE;
    config.isOnOffSubtractionEnabled = true;
    config.isHdrEnabled = false;
    config.isLensShadingEnabled = true;
    config.fps = 133;
}

void ClusteringModeTestTask::SetDefaultClusteringConfig(ClusteringConfig& config)
{
    config.woi.x = 0;
    config.woi.y = 0;
    config.woi.width = 320;
    config.woi.height = 240;
    config.intensityMin = 150;
    config.pixCntMax = 65535;
    config.pixCntMin = 3;
}

void ClusteringModeTestTask::SaveData(IplImage* pImage, nn::xcd::IrClusteringProcessorState& state)
{
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);

    m_TotalFrameCount++;
    bool isCaptureEnabled = false;
    if (m_SkipFrame == 0 || (m_TotalFrameCount % m_SkipFrame == 0))
    {
        m_CaptureFrameCount++;
        isCaptureEnabled = true;
    }

    if (isCaptureEnabled)
    {
        if (m_IsVisualLogEnabled && m_SamplingResult.IsSuccess())
        {
            sprintf_s(path, OUTPUT_IMAGE_FILE_NAME, m_DirectoryPath, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            s32 success = cvSaveImage(path, pImage);
            if (success == 0)
            {
                TEST_PRINTF("Failed to save an image: %s\n", path);
                return;
            }
        }
        // データログ
        if (fp != NULL)
        {
            char data[LINE_SIZE];
            sprintf_s(data, "%04d%02d%02d_%02d%02d%02d_%04d", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            if (m_SamplingResult.IsSuccess())
            {
                sprintf_s(data, "%s,%d,%d,%d", data, state.samplingNumber, state.diffTimeStampCount, state.objectCount);
                for (int i = 0; i < nn::xcd::IrClusteringProcessorObjectCountMax; i++)
                {
                    sprintf_s(data, "%s,%f,%f,%f,%d,%d,%d,%d,%d",
                        data,
                        state.objects[i].averageIntensity,
                        state.objects[i].centroid.x,
                        state.objects[i].centroid.y,
                        state.objects[i].pixelCount,
                        state.objects[i].bound.x,
                        state.objects[i].bound.y,
                        state.objects[i].bound.width,
                        state.objects[i].bound.height
                    );
                }
                sprintf_s(data, "%s\n", data);
            }
            else
            {
                sprintf_s(data, "%s,SamplingFailed,%0X\n", data, m_SamplingResult.GetInnerValueForDebug());
            }
            fputs(data, fp);
        }
    }
}

void ClusteringModeTestTask::DoCreateLogFileCore()
{
    char path[FILE_NAME_SIZE];
    sprintf_s(path, OUTPUT_CLUSTERING_FILE_NAME, m_DirectoryPath, GetName(), m_DirectoryPath);
    if ((fp = fopen(path, "w")) == NULL)
    {
        TEST_PRINTF("Failed to create file\n");
    }
    char data[LINE_SIZE];
    sprintf_s(data, "time,samplingNumber,deltaTime,objectsCount");
    for (int i = 0; i< nn::xcd::IrClusteringProcessorObjectCountMax; i++)
    {
        sprintf_s(data, "%s,objects[%d].averageIntensity,objects[%d].centroidX,objects[%d].centoridY,objects[%d].pixelCount,objects[%d].boundX,objects[%d].boundY,objects[%d].boundWidth,objects[%d].boundHeight", data, i, i, i, i, i, i, i, i);
    }
    sprintf_s(data, "%s\n", data);
    fputs(data, fp);
}

void ClusteringModeTestTask::DoCloseLogFileCore()
{
    fclose(fp);
}

