﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>
#include "TeraIrsensorDriver.h"


namespace nnt {

// テストでは、チェックしないため、0.0 以上を要求しておく。
static const ::nn::xcd::IrMcuVersion requiredVersion = { 0, 0 };

const nn::TimeSpan XcdIrSensorDriver::SamplingInterval = ::nn::TimeSpan::FromMilliSeconds(15);
const int XcdIrSensorDriver::TrialRetryCountMax = 10;

XcdIrSensorDriver::XcdIrSensorDriver() NN_NOEXCEPT
    : m_Mutex(false)
    , m_IsDriverInitialized(false)
    , m_DriverState(IrSensorDriverState::Ready)
    , m_XcdDevice()
    , m_CurrentProcessorType(nn::xcd::IrProcessorType::Ready)
    , m_NextProcessorType(nn::xcd::IrProcessorType::Ready)
    , m_Thread()
    , m_CommonData()
    , m_CommonWorkBuffer()
    , m_MomentData()
    , m_MomentWorkBuffer()
    , m_ClusteringData()
    , m_ClusteringWorkBuffer()
    , m_ImageTransferData()
    , m_ImageTransferWorkBuffer()
    , m_OutPacketCnt()
    , m_Result()
    , m_IrSamplingEvent()
    , m_IrCommandCompletionEvent()
    , m_RequestCompletionEvent()
    , m_SamplingTimer()
#if defined (USE_OPENCV)
    , m_IsLraEnabled(false)
#endif
{
    std::memset(m_ThreadStack, 0, 4096);

    for (auto i = 0; i < nn::xcd::IrDpdProcessorStateCountMax; i++)
    {
        memset(&m_DpdData[i], 0, sizeof(nn::xcd::IrDpdProcessorState));
        memset(&m_DpdWorkBuffer[i], 0, sizeof(nn::xcd::IrDpdProcessorState));
    }
}

XcdIrSensorDriver::~XcdIrSensorDriver() NN_NOEXCEPT
{
}

/*
 * IR センサ機能のセットアップ
 */
void XcdIrSensorDriver::Initialize(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    if (m_IsDriverInitialized)
    {
        return;
    }

    // ハンドルを設定
    m_XcdDevice.handle = handle;

    // DeviceInfo を取得
    nn::xcd::GetDeviceInfo(&m_XcdDevice.info, m_XcdDevice.handle);
    nn::xcd::GetSensorCalibrationValue(&m_XcdDevice.cal, m_XcdDevice.handle);

    nn::xcd::DeviceInfo& devInfo = m_XcdDevice.info;
    NN_LOG("[TeraIrTest] #%lld Addr: %02X %02X %02X %02X %02X %02X   Type: %s\n",
        m_XcdDevice.handle, devInfo.address.address[0], devInfo.address.address[1], devInfo.address.address[2],
        devInfo.address.address[3], devInfo.address.address[4], devInfo.address.address[5],
        (devInfo.deviceType == nn::xcd::DeviceType_Right) ? "Right" : "Left");

    if (devInfo.deviceType == nn::xcd::DeviceType_Right)
    {
        // 通信フォーマットの設定
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::xcd::SetDataFormat(nn::xcd::PeriodicDataFormat_MCU, m_XcdDevice.handle));

        // リクエストハンドリングスレッドを生成
        nn::os::CreateThread(&m_Thread, RequestHandlerThreadLauncher, this, m_ThreadStack, sizeof(m_ThreadStack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&m_Thread);

        // IR センサ用イベント
        nn::os::CreateSystemEvent(&m_IrSamplingEvent, nn::os::EventClearMode_ManualClear, false);
        nn::os::CreateSystemEvent(&m_IrCommandCompletionEvent, nn::os::EventClearMode_ManualClear, false);
        nn::os::CreateSystemEvent(&m_RequestCompletionEvent, nn::os::EventClearMode_ManualClear, false);
        nn::os::SignalSystemEvent(&m_RequestCompletionEvent);

        nn::os::InitializeTimerEvent(&m_SamplingTimer, nn::os::EventClearMode_ManualClear);
        nn::os::StartPeriodicTimerEvent(&m_SamplingTimer, 0, SamplingInterval);

        m_ImageTransferData.pImage = new char[nn::xcd::IrImageSizeMax];
        m_ImageTransferWorkBuffer.pImage = new char[nn::xcd::IrImageSizeMax];

        m_IsDriverInitialized = true;
    }
}

/*
 * IR センサ機能の終了処理
 */
void XcdIrSensorDriver::Finalize() NN_NOEXCEPT
{
    if (m_IsDriverInitialized)
    {
        // スレッドの停止
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);

        // メモリの解放
        delete[] static_cast<char*>(m_ImageTransferData.pImage);
        delete[] static_cast<char*>(m_ImageTransferWorkBuffer.pImage);

        nn::os::DestroySystemEvent(&m_IrSamplingEvent);
        nn::os::DestroySystemEvent(&m_IrCommandCompletionEvent);
        nn::os::DestroySystemEvent(&m_RequestCompletionEvent);
        nn::os::FinalizeTimerEvent(&m_SamplingTimer);

        m_IsDriverInitialized = false;
    }
}

/*
 * スレッドのランチャー
 */
void XcdIrSensorDriver::RequestHandlerThreadLauncher(void* arg) NN_NOEXCEPT
{
    auto pDriver = reinterpret_cast<XcdIrSensorDriver*>(arg);
    pDriver->RequestHandlerThreadFunc();
}

/*
 * リクエストハンドルスレッド
 */
void XcdIrSensorDriver::RequestHandlerThreadFunc() NN_NOEXCEPT
{
    static bool isRunning = true;

    while (isRunning)
    {
        if (nn::os::TryWaitTimerEvent(&m_SamplingTimer))
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);

            nn::xcd::GetPadState(&m_XcdDevice.pad, m_XcdDevice.handle);
            nn::xcd::GetDeviceStatus(&m_XcdDevice.status, m_XcdDevice.handle);

#if defined (USE_OPENCV)
            if (m_IsLraEnabled)
            {
                // LRAが有効の場合は、定常振動させる
                const nn::xcd::VibrationValue maxVibration =
                {
                    1.0f,
                    40.0f,
                    0.0f,
                    320.0f
                };
                nn::xcd::SendVibrationValue(maxVibration, nn::xcd::VibratorPosition_Right, m_XcdDevice.handle);
            }
#endif

            switch (m_DriverState)
            {
            case IrSensorDriverState::InitializeMcu:
                {
                    // Mcu 状態をIRモードに遷移するリクエストを行う
                    DoInitializeMcuSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                    m_DriverState = IrSensorDriverState::Ready;
                }
                break;
            case IrSensorDriverState::FinalizeMcu:
                {
                    // Mcu の終了処理をするリクエストを行う
                    DoFinalizeMcuSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                    m_DriverState = IrSensorDriverState::Ready;
                    isRunning = false;
                }
                break;
            case IrSensorDriverState::ModeSetting:
                {
                    // IR センサーのモードを設定する
                    DoModeSettingSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                    if (m_CurrentProcessorType == nn::xcd::IrProcessorType::Ready)
                    {
                        m_DriverState = IrSensorDriverState::Ready;
                    }
                    else
                    {
                        m_DriverState = IrSensorDriverState::DataRead;
                    }
                }
                break;
            case IrSensorDriverState::DataRead:
                {
                    // IR データのキャプチャを行う
                    DoDataReadSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                }
                break;
            case IrSensorDriverState::ReadRegister:
                {
                    // レジスタリードを行う
                    DoReadRegisterSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                    if (m_CurrentProcessorType == nn::xcd::IrProcessorType::Ready)
                    {
                        m_DriverState = IrSensorDriverState::Ready;
                    }
                    else
                    {
                        m_DriverState = IrSensorDriverState::DataRead;
                    }
                }
                break;
            case IrSensorDriverState::WriteRegister:
                {
                    // レジスタライトを行う
                    DoWriteRegisterSequence();
                    nn::os::SignalSystemEvent(&m_RequestCompletionEvent);
                    if (m_CurrentProcessorType == nn::xcd::IrProcessorType::Ready)
                    {
                        m_DriverState = IrSensorDriverState::Ready;
                    }
                    else
                    {
                        m_DriverState = IrSensorDriverState::DataRead;
                    }
                }
                break;
            case IrSensorDriverState::Ready:
                {
                    // Nothing
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

/*
*  IR センサーのレジスタライトを行います
*/
void XcdIrSensorDriver::DoWriteRegisterSequence() NN_NOEXCEPT
{
    auto handle = m_XcdDevice.handle;

    // モード実行中は一旦止める
    if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
    {
        nn::xcd::StopIrSampling(handle);
    }

    // レジスタ設定
    nn::xcd::StartIrWriteRegister(m_WriteRegisterSetting, handle);

    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::os::TryWaitSystemEvent(&m_IrCommandCompletionEvent))
        {
            // StartIrWriteRegister 関数が確実に成功するため、本来必要ないが、
            // 完了のデータ取得を確認用に実装している。
            nn::os::ClearSystemEvent(&m_IrCommandCompletionEvent);
            nn::xcd::IrWriteRegisterState state;
            nn::xcd::GetIrWriteRegisterState(&state, handle);
            NN_LOG("[TeraIrTest] RegWrite done\n");
            break;
        }
        trialCounter++;
        NN_SDK_ASSERT_LESS(trialCounter, TrialRetryCountMax);
        ::nn::os::SleepThread(SamplingInterval);
    }

    // 書き込みが成功した場合は、キャプチャに戻る
    if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
    {
        nn::xcd::StartIrSampling(handle);
    }
}

/*
*  IR センサーのレジスタリードを行います
*/
void XcdIrSensorDriver::DoReadRegisterSequence() NN_NOEXCEPT
{
    auto handle = m_XcdDevice.handle;

    // モード実行中は一旦止める
    if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
    {
        nn::xcd::StopIrSampling(handle);
    }

    // 指しあたって読みだすサイズによってコストは変わらないので、最大領域を読みだすようにしています。
    nn::xcd::IrReadRegisterSetting setting;
    setting.bankId = 0;
    setting.startAddress = 0x00;
    setting.size = 128;
    nn::xcd::StartIrReadRegister(setting, handle);

    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::os::TryWaitSystemEvent(&m_IrCommandCompletionEvent))
        {
            NN_LOG("[TeraIrTest] Ir Register event!!\n");
            nn::os::ClearSystemEvent(&m_IrCommandCompletionEvent);
            nn::xcd::IrReadRegisterState state;
            nn::xcd::GetIrReadRegisterState(&state, handle);

            for (auto i = 0; i < 256; i++)
            {
                NN_LOG("%02X ", state.bankData[i]);
            }
            NN_LOG("\n");
            break;
        }
        trialCounter++;
        NN_SDK_ASSERT_LESS(trialCounter, TrialRetryCountMax);
        ::nn::os::SleepThread(SamplingInterval);
    }

    // 読み込みが完了したらキャプチャに戻る
    if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
    {
        nn::xcd::StartIrSampling(handle);
    }

}

/*
*  IR センサーのデータキャプチャを行います。
*/
void XcdIrSensorDriver::DoDataReadSequence() NN_NOEXCEPT
{
    auto handle = m_XcdDevice.handle;
    auto type = m_CurrentProcessorType;

    int maxPacketCnt = 1;
    int outPacketCnt = 0;
    if (nn::os::TryWaitSystemEvent(&m_IrCommandCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_IrCommandCompletionEvent);
        {
            nn::xcd::StartIrSampling(handle);
        }
    }
    // サンプリングイベントが起きていれば、読み込みを行う
    if (nn::os::TryWaitSystemEvent(&m_IrSamplingEvent))
    {
        nn::os::ClearSystemEvent(&m_IrSamplingEvent);
        if (type == nn::xcd::IrProcessorType::Moment)
        {
            m_Result = nn::xcd::GetIrMomentStates(&m_CommonData, &m_MomentData, &outPacketCnt, maxPacketCnt, handle);
            if (m_Result.IsSuccess())
            {
                NN_LOG("[TeraIrTest] Ir Moment Sampling!! samplingNumber:%d diffTimestamp:%d\n", m_MomentData.samplingNumber, m_MomentData.diffTimeStampCount);
            }
            else
            {
                NN_LOG("[TeraIrTest] Ir Moment Sampling Failed: %0X\n", m_Result);
            }
        }
        else if (type == nn::xcd::IrProcessorType::Clustering)
        {
            m_Result = nn::xcd::GetIrClusteringStates(&m_CommonData, &m_ClusteringData, &outPacketCnt, maxPacketCnt, handle);
            if (m_Result.IsSuccess())
            {
                NN_LOG("[TeraIrTest] Ir Clustering Sampling!! samplingNumber:%d diffTimestamp:%d\n", m_ClusteringData.samplingNumber, m_ClusteringData.diffTimeStampCount);
            }
            else
            {
                NN_LOG("[TeraIrTest] Ir Clustering Sampling Failed: %0X\n", m_Result);
            }
        }
        else if (type == nn::xcd::IrProcessorType::ImageTransfer)
        {
            m_Result = nn::xcd::GetIrImageTransferState(&m_CommonData, &m_ImageTransferData, handle);
            if (m_Result.IsSuccess())
            {
                NN_LOG("[TeraIrTest] Ir ImageTransfer Sampling!!\n");
            }
            else
            {
                NN_LOG("[TeraIrTest] Ir ImageTransfer Sampling Failed: %0X\n", m_Result);
            }
        }
        else if (type == nn::xcd::IrProcessorType::Dpd)
        {
            maxPacketCnt = nn::xcd::IrDpdProcessorStateCountMax;
            m_Result = nn::xcd::GetIrDpdStates(&m_CommonData, &m_DpdData[0], &outPacketCnt, maxPacketCnt, handle);
            if (m_Result.IsSuccess())
            {
                NN_LOG("[TeraIrTest] Ir Dpd Sampling!! samplingNumber:%d diffTimestamp:%d outCnt:%d\n", m_DpdData[0].samplingNumber, m_DpdData[0].diffTimeStampCount, outPacketCnt);
                m_OutPacketCnt = outPacketCnt;
            }
            else
            {
                NN_LOG("[TeraIrTest] Ir Dpd Sampling Failed: %0X\n", m_Result);
                m_OutPacketCnt = 0;
            }
        }
        else if (type == nn::xcd::IrProcessorType::TeraPlugin)
        {
            m_Result = nn::xcd::GetIrTeraPluginState(&m_CommonData, &m_HandAnalysisData, handle);
            if (m_Result.IsSuccess())
            {
                NN_LOG("[TeraIrTest] Ir HandAnalysis Sampling!! samplingNumber:%d diffTimestamp:%d\n", m_HandAnalysisData.samplingNumber, m_HandAnalysisData.diffTimeStampCount);
            }
            else
            {
                NN_LOG("[TeraIrTest] Ir HandAnalysis Sampling Failed: %0X\n", m_Result);
            }
        }
    }
}

/*
*  IR センサーのモード遷移を行います。
*/
void XcdIrSensorDriver::DoModeSettingSequence() NN_NOEXCEPT
{
    auto handle = m_XcdDevice.handle;
    auto type = m_NextProcessorType;

    if (m_NextProcessorType == m_CurrentProcessorType)
    {
        return;
    }

    if (type == nn::xcd::IrProcessorType::Moment)
    {
        if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
        {
            return;
        }
        nn::xcd::SetupIrMomentProcessor(&m_CommonWorkBuffer, &m_MomentWorkBuffer, handle);
    }
    else if (type == nn::xcd::IrProcessorType::Clustering)
    {
        if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
        {
            return;
        }
        nn::xcd::SetupIrClusteringProcessor(&m_CommonWorkBuffer, &m_ClusteringWorkBuffer, handle);
    }
    else if (type == nn::xcd::IrProcessorType::ImageTransfer)
    {
        if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
        {
            return;
        }
        auto size = nn::xcd::IrImageTransferProcessorFormat::ImageSize_320x240;
        //auto size = nn::xcd::IrImageTransferProcessorFormat::ImageSize_160x120;
        //auto size = nn::xcd::IrImageTransferProcessorFormat::ImageSize_80x60;
        nn::xcd::SetupIrImageTransferProcessor(&m_CommonWorkBuffer, &m_ImageTransferWorkBuffer, size, handle);
    }
    else if (type == nn::xcd::IrProcessorType::Dpd)
    {
        if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
        {
            return;
        }
        nn::xcd::SetupIrDpdProcessor(&m_CommonWorkBuffer, &m_DpdWorkBuffer[0], handle);
    }
    else if (type == nn::xcd::IrProcessorType::TeraPlugin)
    {
        if (m_CurrentProcessorType != nn::xcd::IrProcessorType::Ready)
        {
            return;
        }
        nn::xcd::SetupIrTeraPluginProcessor(&m_CommonWorkBuffer, &m_HandAnalysisWorkBuffer, handle);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetMode(type));
    m_CurrentProcessorType = m_NextProcessorType;
    m_NextProcessorType = nn::xcd::IrProcessorType::Ready;
}

/*
 * McuのIRへの状態遷移シーケンスを実行
 */
void XcdIrSensorDriver::DoInitializeMcuSequence() NN_NOEXCEPT
{
    // IR モードに遷移命令を出す
    NN_LOG("[TeraIrTest] Set state to IR\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetMcuState(nn::xcd::McuState_Ir));

    // イベントを登録
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::SetIrControlEvent(
        &m_IrSamplingEvent,
        &m_IrCommandCompletionEvent,
        m_XcdDevice.handle));
}

/*
 * Mcuの終了シーケンスを実行
 */
void XcdIrSensorDriver::DoFinalizeMcuSequence() NN_NOEXCEPT
{
    // リソースを破棄
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::TeardownIrProcessor(
        m_XcdDevice.handle));

    // IR モードに遷移命令を出す
    NN_LOG("[TeraIrTest] Set state to Standby\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetMcuState(nn::xcd::McuState_Standby));
}

nn::Result XcdIrSensorDriver::SetMcuState(nn::xcd::McuState state) NN_NOEXCEPT
{
    const auto TrialCountMax = 100;
    const auto SleepTime = ::nn::TimeSpan::FromMilliSeconds(15);
    const auto StandbySleepTime = ::nn::TimeSpan::FromMilliSeconds(100);

    auto handle = m_XcdDevice.handle;

    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_TRY(::nn::xcd::SetIrMcuState(state, handle))
            NN_RESULT_CATCH(::nn::xcd::ResultMcuBusy)
            {
                // McuBusy の場合はリトライする
                // リトライ上限を超えた場合はエラーを返す
                NN_RESULT_THROW_UNLESS(
                    trialCounter <TrialCountMax,
                    ::nn::xcd::ResultMcuBusy());

                ::nn::os::SleepThread(SleepTime);
                trialCounter++;
                continue;
            }
            NN_RESULT_CATCH_ALL
            {
                // その他のエラーの場合は、上に通知
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        // 成功した場合はループを抜けて、ベリファイに進む
        break;
    }

    for (auto i = 0; i < TrialCountMax; ++i)
    {
        auto current = ::nn::xcd::McuState_Idle;
        NN_RESULT_DO(::nn::xcd::GetIrMcuState(&current, handle));
        if (current == state)
        {
            if (current == ::nn::xcd::McuState_Standby)
            {
                ::nn::os::SleepThread(StandbySleepTime);
            }
            NN_RESULT_SUCCESS;
        }
        ::nn::os::SleepThread(SleepTime);
    }

    NN_RESULT_THROW(::nn::xcd::ResultMcuBusy());
}

::nn::Result XcdIrSensorDriver::SetMode(
    ::nn::xcd::IrProcessorType mode) NN_NOEXCEPT
{
    auto handle = m_XcdDevice.handle;
    auto modeOffset = 0;
    const auto SleepTime = ::nn::TimeSpan::FromMilliSeconds(50);
    const auto CommandExecutionTimeout = ::nn::TimeSpan::FromMilliSeconds(500);

    ::nn::os::ClearSystemEvent(&m_IrCommandCompletionEvent);
    nn::xcd::IrTeraPluginParameter param = {};
    NN_RESULT_DO(::nn::xcd::SetIrProcessorType(mode, modeOffset, param, requiredVersion, ::nn::xcd::IrCommandType::Normal, handle));
    if (!::nn::os::TimedWaitSystemEvent(
        &m_IrCommandCompletionEvent, CommandExecutionTimeout))
    {
        return ::nn::xcd::ResultIrCommandFailed();
    }
    ::nn::os::ClearSystemEvent(&m_IrCommandCompletionEvent);

    for (auto j = 0; j < TrialRetryCountMax; ++j)
    {
        ::nn::xcd::IrProcessorType current;
        ::nn::xcd::IrMcuVersion compatibleVersion;
        NN_RESULT_DO(::nn::xcd::GetIrProcessorType(&current, &compatibleVersion, handle));
        if (current == mode)
        {
            NN_RESULT_SUCCESS;
        }
        ::nn::os::SleepThread(SleepTime);
    }
    NN_RESULT_THROW(::nn::xcd::ResultIrProcessorNotReady());
}

/*
 * RegisterWrite の要求を発行
 */
bool XcdIrSensorDriver::RequestWriteRegister(nn::xcd::IrWriteRegisterSetting writeConfig) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        m_WriteRegisterSetting = writeConfig;
        m_DriverState = IrSensorDriverState::WriteRegister;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

/*
 * RegisterRead の要求を発行
 */
bool XcdIrSensorDriver::RequestReadRegister() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        m_DriverState = IrSensorDriverState::ReadRegister;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

/*
 * DataRead の要求を発行
 */
bool XcdIrSensorDriver::RequestDataRead() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        m_DriverState = IrSensorDriverState::DataRead;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

/*
 * McuInitializeの要求を発行
 */
bool XcdIrSensorDriver::RequestMcuInitialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        m_DriverState = IrSensorDriverState::InitializeMcu;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

/*
 * McuFinalize の要求を発行
 */
bool XcdIrSensorDriver::RequestMcuFinalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        nn::xcd::StopIrSampling(m_XcdDevice.handle);
        m_DriverState = IrSensorDriverState::FinalizeMcu;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

/*
 * ModeSet の要求を発行
 */
bool XcdIrSensorDriver::RequestModeSet(nn::xcd::IrProcessorType type) NN_NOEXCEPT
{
    bool isRequestEnabled = false;
    if (nn::os::TryWaitSystemEvent(&m_RequestCompletionEvent))
    {
        nn::os::ClearSystemEvent(&m_RequestCompletionEvent);

        m_DriverState = IrSensorDriverState::ModeSetting;
        m_NextProcessorType = type;
        isRequestEnabled = true;
    }
    return isRequestEnabled;
}

nn::Result XcdIrSensorDriver::GetMomentData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrMomentProcessorState* pOutData) NN_NOEXCEPT
{
    if (m_Result.IsSuccess())
    {
        std::memcpy(pOutCommonData, &m_CommonData, sizeof(nn::xcd::IrCommonData));
        std::memcpy(pOutData, &m_MomentData, sizeof(nn::xcd::IrMomentProcessorState));
    }
    return m_Result;
}

nn::Result XcdIrSensorDriver::GetClusteringData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrClusteringProcessorState* pOutData) NN_NOEXCEPT
{
    if (m_Result.IsSuccess())
    {
        std::memcpy(pOutCommonData, &m_CommonData, sizeof(nn::xcd::IrCommonData));
        std::memcpy(pOutData, &m_ClusteringData, sizeof(nn::xcd::IrClusteringProcessorState));
    }
    return m_Result;
}

nn::Result XcdIrSensorDriver::GetImageTransferData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrImageTransferProcessorState* pOutData) NN_NOEXCEPT
{
    if (m_Result.IsSuccess())
    {
        std::memcpy(pOutCommonData, &m_CommonData, sizeof(nn::xcd::IrCommonData));
        std::memcpy(pOutData->pImage, m_ImageTransferData.pImage, nn::xcd::IrImageSizeMax);
    }
    return m_Result;
}

nn::Result XcdIrSensorDriver::GetDpdData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrDpdProcessorState* pOutData, int* pOutPacketCnt) NN_NOEXCEPT
{
    if (m_Result.IsSuccess())
    {
        std::memcpy(pOutCommonData, &m_CommonData, sizeof(nn::xcd::IrCommonData));
        std::memcpy(pOutData, &m_DpdData[0], sizeof(nn::xcd::IrDpdProcessorState) * m_OutPacketCnt);
        *pOutPacketCnt = m_OutPacketCnt;
    }
    return m_Result;
}

nn::Result XcdIrSensorDriver::GetHandAnalysisData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrTeraPluginProcessorState* pOutData) NN_NOEXCEPT
{
    if (m_Result.IsSuccess())
    {
        std::memcpy(pOutCommonData, &m_CommonData, sizeof(nn::xcd::IrCommonData));
        std::memcpy(pOutData, &m_HandAnalysisData, sizeof(nn::xcd::IrTeraPluginProcessorState));
    }
    return m_Result;
}

#if defined (USE_OPENCV)
void XcdIrSensorDriver::GetPadData(nn::xcd::PadState* pOutPadData, nn::xcd::SensorCalibrationValue* pSensorCal) NN_NOEXCEPT
{
    std::memcpy(pOutPadData, &m_XcdDevice.pad, sizeof(nn::xcd::PadState));
    std::memcpy(pSensorCal, &m_XcdDevice.cal, sizeof(nn::xcd::SensorCalibrationValue));
}

void XcdIrSensorDriver::SetSixAxisSampling(bool isSixAxisSamplingEnabled) NN_NOEXCEPT
{
    // 6軸センサーの ON/OFF
    nn::xcd::SleepSensor(!isSixAxisSamplingEnabled, m_XcdDevice.handle);
}

void XcdIrSensorDriver::SetLraSending(bool isLraEnabled) NN_NOEXCEPT
{
    // LRA の ON/OFF
    m_IsLraEnabled = isLraEnabled;
    nn::xcd::SetVibratorEnabled(isLraEnabled, nn::xcd::VibratorPosition_Right, m_XcdDevice.handle);
}
#endif

}  // nnt

