﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/nn_TimeSpan.h>

#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

/*
 * このテストは hid プロセスが動作していない状態で実行する必要があります。
 */

namespace {

const char RomMountName[] = "rom";

const char FwImageFile[]               = "rom:/terafw_0401.bin";
const char FwImageFileFull[]           = "rom:/terafw_0401_with_iap.bin";
const char FwImageFileForFullKey[]     = "rom:/terafw_fullkey_0305.bin";
const char FwImageFileFullForFullKey[] = "rom:/terafw_fullkey_0305_with_iap.bin";

const uint8_t LedPattern[] = { 0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0d, 0x06 };

enum FirmwareImageType : uint8_t
{
    FirmwareImageType_Right   = 0xCC,
    FirmwareImageType_FullKey = 0xCD
};

struct XcdDeviceBlock
{
    nn::xcd::DeviceHandle       handle;
    nn::os::SystemEventType     event;
    nn::os::MultiWaitHolderType holder;
    nn::xcd::DeviceInfo         info;
    nn::xcd::PadState           pad;
    nn::xcd::DeviceStatus       status;
    nn::xcd::PadButtonSet       previousButton;
    nn::xcd::McuVersionData     mcuVersion;
    bool connected;
    bool hasValidMcuVersionInfo;
};
XcdDeviceBlock g_Devices[nn::xcd::DeviceCountMax];
XcdDeviceBlock* g_pUpdatingDevice;

nn::os::MultiWaitType       g_MultiWait;
nn::os::SystemEventType     g_LinkEvent;
nn::os::SystemEventType     g_UpdateFinishEvent;
nn::os::EventType           g_TerminateEvent;
nn::os::MultiWaitHolderType g_LinkHolder;
nn::os::MultiWaitHolderType g_UpdateFinishHolder;
nn::os::MultiWaitHolderType g_TerminateHolder;

bool                g_IsUpdating = false;
nn::fs::FileHandle  g_FwImageFile;
nn::os::Tick        g_StartTick;
nn::os::Tick        g_TimeoutTick;

// アップデート対象のファームウェアイメージ情報
struct FirmwareImageInfo
{
    uint8_t type;
    struct
    {
        uint8_t major;
        uint8_t minor;
    } version;
    NN_PADDING5;
} g_TargetImageInfo = {};

NN_STATIC_ASSERT(sizeof(FirmwareImageInfo) == 8);

// テストの状態
enum TestSequence
{
    StartingTest,   // テスト開始処理
    RunningTest,    // テスト実行中
    Verifying,      // 成否判定
    Verified,       // 成否判定完了
    Finished        // テスト終了
};

TestSequence g_CurrentSequence = TestSequence::StartingTest;

/*
* アップデート状況の名称を取得
*/
const char* GetUpdateStateName(nn::xcd::McuUpdateState state) NN_NOEXCEPT
{
    switch (state)
    {
    case nn::xcd::McuUpdateState_Boot:       return "Boot";
    case nn::xcd::McuUpdateState_RomErasing: return "RomErasing";
    case nn::xcd::McuUpdateState_Writing:    return "Writing";
    case nn::xcd::McuUpdateState_Reboot:     return "Reboot";
    case nn::xcd::McuUpdateState_End:        return "End";
    default:                                 return "???";
    }
}

/*
* Tera のバージョンを表示
*/
void PrintMcuVersion(const nn::xcd::McuVersionData& version) NN_NOEXCEPT
{
    NN_LOG("[MCU Version Data]\n");
    if (version.isCorrupted)
    {
        NN_LOG("  Corrupted!\n");
    }
    else
    {
        NN_LOG("  Version: %02X.%02X\n", version.major, version.minor);
    }
}

/*
* アップデート状況の表示
*/
void PrintUpdateState(const nn::xcd::McuUpdateStateInfo& updateInfo) NN_NOEXCEPT
{
    auto elapsedTime = (nn::os::GetSystemTick() - g_StartTick).ToTimeSpan();
    NN_LOG("[Updating %2d:%02d.%03d] %3d%% - %s\n",
        elapsedTime.GetMinutes(),
        elapsedTime.GetSeconds() % 60,
        elapsedTime.GetMilliSeconds() % 1000,
        updateInfo.progress,
        GetUpdateStateName(updateInfo.state));
}

/*
 * 対応しているデバイスか判定
 */
bool IsSupportedDevice(nn::xcd::DeviceType type) NN_NOEXCEPT
{
    return type == nn::xcd::DeviceType_Right || type == nn::xcd::DeviceType_FullKey;
}

void SetupEvents() NN_NOEXCEPT
{
    nn::os::InitializeMultiWait(&g_MultiWait);

    for (int i = 0; i < nn::xcd::DeviceCountMax; i++)
    {
        auto& device = g_Devices[i];
        device.connected = false;
        nn::os::CreateSystemEvent(&device.event, nn::os::EventClearMode_ManualClear, false);
        nn::os::InitializeMultiWaitHolder(&device.holder, &device.event);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &device.holder);
    }

    nn::os::CreateSystemEvent(&g_LinkEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&g_LinkHolder, &g_LinkEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_LinkHolder);

    nn::os::CreateSystemEvent(&g_UpdateFinishEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&g_UpdateFinishHolder, &g_UpdateFinishEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_UpdateFinishHolder);

    nn::os::InitializeEvent(&g_TerminateEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TerminateHolder, &g_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TerminateHolder);
}

void FinalizeEvents() NN_NOEXCEPT
{
    nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
    nn::os::FinalizeMultiWait(&g_MultiWait);

    nn::os::FinalizeMultiWaitHolder(&g_TerminateHolder);
    nn::os::FinalizeMultiWaitHolder(&g_UpdateFinishHolder);
    nn::os::FinalizeMultiWaitHolder(&g_LinkHolder);

    nn::os::FinalizeEvent(&g_TerminateEvent);
    nn::os::DestroySystemEvent(&g_UpdateFinishEvent);
    nn::os::DestroySystemEvent(&g_LinkEvent);

    for (int i = 0; i < nn::xcd::DeviceCountMax; i++)
    {
        auto& device = g_Devices[i];
        nn::os::FinalizeMultiWaitHolder(&device.holder);
        nn::os::DestroySystemEvent(&device.event);
    }
}

void StartUpdate(XcdDeviceBlock* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(!g_IsUpdating);
    NN_ASSERT(IsSupportedDevice(pDevice->info.deviceType));

    bool isFullKey = pDevice->info.deviceType == nn::xcd::DeviceType_FullKey;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::OpenFile(
            &g_FwImageFile,
            isFullKey ? FwImageFileForFullKey : FwImageFile,
            nn::fs::OpenMode_Read));

    // FW イメージのヘッダを取得
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::ReadFile(g_FwImageFile, 0, &g_TargetImageInfo, sizeof(g_TargetImageInfo)));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartMcuUpdate(pDevice->handle, g_FwImageFile, &g_UpdateFinishEvent));
    g_IsUpdating = true;
}

void StartFullUpdate(const XcdDeviceBlock* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(!g_IsUpdating);
    NN_ASSERT(IsSupportedDevice(pDevice->info.deviceType));

    bool isFullKey = pDevice->info.deviceType == nn::xcd::DeviceType_FullKey;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::OpenFile(
            &g_FwImageFile,
            isFullKey ? FwImageFileFullForFullKey : FwImageFileFull,
            nn::fs::OpenMode_Read));

    // FW イメージのヘッダを取得
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::ReadFile(g_FwImageFile, 0, &g_TargetImageInfo, sizeof(g_TargetImageInfo)));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartMcuUpdateFull(pDevice->handle, g_FwImageFile, &g_UpdateFinishEvent));
    g_IsUpdating = true;
}

void EndUpdate(const XcdDeviceBlock* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(g_IsUpdating);
    NN_UNUSED(pDevice);

    nn::fs::CloseFile(g_FwImageFile);
    g_IsUpdating = false;
}

void HandleLinkupEvent() NN_NOEXCEPT
{
    // nn::hid::GetDebugPadStates による状態取得
    nn::xcd::DeviceList list;
    if (nn::xcd::ListDevices(&list).IsFailure())
    {
        NN_LOG("No device Found\n");
        return;
    }

    // 接続されたデバイスの探索
    for (int i = 0; i < list.deviceCount; i++)
    {
        NN_LOG("New XCD Device Found\n");

        auto& device = g_Devices[i];
        device.handle = list.handleList[i];
        nn::xcd::SetSamplingEvent(&device.event, device.handle);
        nn::xcd::SleepSensor(false, device.handle);
        nn::xcd::SetPlayerIndicatorPattern(LedPattern[i], device.handle);
        device.connected = true;
        device.hasValidMcuVersionInfo = false;
        nn::xcd::GetDeviceInfo(&device.info, device.handle);

        if (IsSupportedDevice(device.info.deviceType))
        {
            NN_LOG(
                "New device is %s\n",
                device.info.deviceType == nn::xcd::DeviceType_Right ? "RightJoy" : "FullKey");

            // Tera のバージョンを取得
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::xcd::RequestMcuVersion(device.handle)
            );
        }
        else
        {
            NN_LOG("New device is not RightJoy / FullKey\n");
        }
    }
}

void HandleUpdateFinishEvent() NN_NOEXCEPT
{
    // アップデートが終了したので成否確認へ
    g_CurrentSequence = TestSequence::Verifying;
    g_TimeoutTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(10));
}

void ProceedUpdate(XcdDeviceBlock* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    static int s_DisplayCount = 0;

    nn::xcd::McuUpdateStateInfo updateInfo;
    nn::xcd::GetMcuUpdateState(&updateInfo, pDevice->handle);

    if (s_DisplayCount >= 12)
    {
        PrintUpdateState(updateInfo);
        s_DisplayCount = 0;
    }
    else
    {
        s_DisplayCount++;
    }

    switch (g_CurrentSequence)
    {
    case TestSequence::Verifying:
        {
            nn::xcd::McuUpdateStateInfo updateInfo;
            if (nn::xcd::GetMcuUpdateState(&updateInfo, pDevice->handle).IsSuccess() &&
                updateInfo.state == nn::xcd::McuUpdateState_End)
            {
                PrintUpdateState(updateInfo);

                nn::xcd::McuVersionData version;
                if (nn::xcd::GetMcuVersion(&version, pDevice->handle).IsSuccess())
                {
                    PrintMcuVersion(version);

                    if (version.isCorrupted)
                    {
                        NN_LOG("Update FAILED... FW may be broken.\n");
                    }
                    else if (version.major == g_TargetImageInfo.version.major &&
                             version.minor == g_TargetImageInfo.version.minor)
                    {
                        NN_LOG("Update successful!!!\n");
                    }
                    else
                    {
                        NN_LOG("Update FAILED... FW version is not matched:\n");
                        NN_LOG("  Image : %02X.%02X\n", g_TargetImageInfo.version.major, g_TargetImageInfo.version.minor);
                        NN_LOG("  Device: %02X.%02X\n", version.major, version.minor);
                    }
                }
                else
                {
                    NN_LOG("Failed to get FW version\n");
                }
                g_CurrentSequence = TestSequence::Verified;
                EndUpdate(pDevice);
            }
            else if (nn::os::GetSystemTick() > g_TimeoutTick)
            {
                NN_LOG("Verify timed out.\n");
                nn::xcd::McuState state;
                if (nn::xcd::GetMcuState(&state, pDevice->handle).IsSuccess())
                {
                    NN_LOG("State: %d\n", state);
                }

                g_CurrentSequence = TestSequence::Verified;
                EndUpdate(pDevice);
            }

            NN_LOG("Press R+ZR to exit.\n");
            return;
        }
    case TestSequence::RunningTest:
        {
            if (pDevice->pad.buttons.Test<nn::xcd::PadButton::X>() &&
                pDevice->pad.buttons.Test<nn::xcd::PadButton::Y>())
            {
                // X+Y で abort
                nn::xcd::AbortMcuUpdate(pDevice->handle);
            }
            return;
        }
    case TestSequence::Verified:
    case TestSequence::Finished:
        {
            // 何もしない
            return;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool NeedsIapUpdate(nn::xcd::DeviceType type, nn::xcd::McuVersionData& version) NN_NOEXCEPT
{
    if (version.isIapCorrupted)
    {
        // IAP が壊れているので要更新
        return true;
    }
    else if (version.isCorrupted)
    {
        // 通常の FW 壊れ判定ができていれば新 IAP が生きている
        return false;
    }

    uint8_t oldIapVersionMajor;
    uint8_t oldIapVersionMinor;
    switch (type)
    {
    case nn::xcd::DeviceType_Right:
    case nn::xcd::DeviceType_FullKey:
        {
            // CC/CD 2.3 以前は IAP が古い
            oldIapVersionMajor = 2;
            oldIapVersionMinor = 3;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (version.major < oldIapVersionMajor ||
        (version.major == oldIapVersionMajor && version.minor <= oldIapVersionMinor))
    {
        // 古い IAP なので要更新
        return true;
    }
    else
    {
        // IAP は更新済み
        return false;
    }
}

void HandleDeviceUpdate(XcdDeviceBlock* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    // 入力状態の取得
    nn::xcd::GetPadState(&pDevice->pad, pDevice->handle);
    nn::xcd::GetDeviceStatus(&pDevice->status, pDevice->handle);

    if (!IsSupportedDevice(pDevice->info.deviceType))
    {
        return;
    }

    if (!pDevice->hasValidMcuVersionInfo)
    {
        // バージョンの取得
        auto result = nn::xcd::GetMcuVersion(&pDevice->mcuVersion, pDevice->handle);
        if (result.IsFailure())
        {
            return;
        }

        pDevice->hasValidMcuVersionInfo = true;

        if (pDevice->mcuVersion.isIapCorrupted)
        {
            NN_LOG("IAP is broken.\n");
        }
        else if (pDevice->mcuVersion.isCorrupted)
        {
            NN_LOG("MCU FW is broken.\n");
        }
        else
        {
            NN_LOG("Current MCU FW version: %02X.%02X\n",
                pDevice->mcuVersion.major,
                pDevice->mcuVersion.minor);
        }

        NN_LOG("Press A button to update.\n");
    }

    if (g_IsUpdating)
    {
        ProceedUpdate(g_pUpdatingDevice);
    }
    else if (
        pDevice->hasValidMcuVersionInfo &&
        pDevice->pad.buttons.Test<nn::xcd::PadButton::A>())
    {
        if (NeedsIapUpdate(pDevice->info.deviceType, pDevice->mcuVersion))
        {
            // IAP の更新が必要
            // (開発機は IAP が古い個体があるので要更新の場合あり。製品では不要)
            NN_LOG("**** Tera FW is too old or IAP is broken. Starting full mode ****\n");
            StartFullUpdate(pDevice);
        }
        else
        {
            NN_LOG("**** Starting normal mode ****\n");
            StartUpdate(pDevice);
        }

        NN_LOG( "Target firmware version: %02X.%02X\n",
            g_TargetImageInfo.version.major,
            g_TargetImageInfo.version.minor);

        g_pUpdatingDevice = pDevice;
        g_CurrentSequence = TestSequence::RunningTest;
        g_StartTick = nn::os::GetSystemTick();
    }
    else if (
        pDevice->pad.buttons.Test<nn::xcd::PadButton::R>() &&
        pDevice->pad.buttons.Test<nn::xcd::PadButton::ZR>())
    {
        // R+ZR で終了
        nn::xcd::Detach(pDevice->handle);
        g_CurrentSequence = TestSequence::Finished;
    }
}

void HandleEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pHolder);

    if (nn::xcd::Proceed(pHolder) == true)
    {
        return;
    }
    else if (pHolder == &g_LinkHolder)
    {
        nn::os::ClearSystemEvent(&g_LinkEvent);
        HandleLinkupEvent();
    }
    else if (pHolder == &g_UpdateFinishHolder)
    {
        nn::os::ClearSystemEvent(&g_UpdateFinishEvent);
        HandleUpdateFinishEvent();
    }
    else
    {
        for (auto& device : g_Devices)
        {
            if (pHolder == &device.holder && device.connected)
            {
                nn::os::ClearSystemEvent(&device.event);
                HandleDeviceUpdate(&device);
            }
        }
    }
}

}  // anonymous


/**
 * XCD の Tera update 機能をテスト
 */
TEST(XcdTeraUpdate, Normal)
{
    SetupEvents();

    static const size_t MountRomCacheBufferSize = 4 * 1024;
    static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ABORT_UNLESS_LESS_EQUAL(mountRomCacheUseSize, MountRomCacheBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom(RomMountName, s_MountRomCacheBuffer, MountRomCacheBufferSize));

    // XCD の初期化
    nn::xcd::Initialize(&g_MultiWait);

    nn::xcd::BindLinkUpdateEvent(&g_LinkEvent);

    while (g_CurrentSequence != TestSequence::Finished)
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::WaitAny(&g_MultiWait);
        HandleEvent(pHolder);
    }

    nn::fs::Unmount(RomMountName);

    FinalizeEvents();
}
