﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

#include "testXcd_NfcAutoFixture.h"

// リトライ処理
#define NNT_XCD_RETRY(format, ...) \
    NN_LOG("%s: " format, NN_CURRENT_FUNCTION_NAME, ##__VA_ARGS__); \
    RetryAndWait(); \
    continue

void XcdNfcBase::SetUp() NN_NOEXCEPT
{
    ::testing::Test::SetUp();

    m_IsNfcInitialized = false;
    nn::os::InitializeMultiWait(&m_MultiWait);
    ::srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));

    nnt::xcd::SetupNpad();
}

void XcdNfcBase::TearDown() NN_NOEXCEPT
{
    nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    nn::os::FinalizeMultiWait(&m_MultiWait);

    if (m_IsNfcInitialized)
    {
        // NFC 関連の終了処理
        nn::os::FinalizeMultiWaitHolder(&m_GeneralEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_DetectEventHolder);
        nn::os::DestroySystemEvent(&m_GeneralEvent);
        nn::os::DestroySystemEvent(&m_DetectEvent);

        nn::xcd::DeviceHandle xcdHandle;
        EXPECT_TRUE(GetNfcDeviceHandle(&xcdHandle));

        EXPECT_TRUE(
            nn::xcd::SetMcuState(
                nn::xcd::McuState_Standby,
                xcdHandle).IsSuccess());

        // Standby が完了しそうな時間待つ
        // XXX: Standby -> NFC を短期間で連投すると動作しなくなる不具合のワークアラウンド
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));

        EXPECT_TRUE(
            nn::xcd::SetDataFormat(
                nn::xcd::PeriodicDataFormat_Basic,
                xcdHandle).IsSuccess());

        m_IsNfcInitialized = false;
    }

    ::testing::Test::TearDown();
}

bool XcdNfcBase::SetupTest() NN_NOEXCEPT
{
    // デバイスの接続検知
    if (!WaitDeviceConnection())
    {
        NN_LOG("%s: Device is not connected\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        NN_LOG("%s: Failed to get xcd handle\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    if (nn::xcd::SetDataFormat(
        nn::xcd::PeriodicDataFormat_MCU,
        xcdHandle).IsFailure())
    {
        NN_LOG("%s: Failed to set data format\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    if (nn::xcd::SetMcuState(
        nn::xcd::McuState_Nfc,
        xcdHandle).IsFailure())
    {
        NN_LOG("%s: Failed to set state\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    // NFC ステートへの遷移を待つ
    auto startTick = nn::os::GetSystemTick();
    nn::xcd::McuState state = nn::xcd::McuState_Initializing;
    while (state != nn::xcd::McuState_Nfc)
    {
        // 規定時間以内に NFC ステートにならない場合は失敗
        auto diffTick = nn::os::GetSystemTick() - startTick;
        if (diffTick.ToTimeSpan() >= nnt::xcd::StateTransitionTimeout)
        {
            NN_LOG("%s: Timed out\n", NN_CURRENT_FUNCTION_NAME);
            return false;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        if (nn::xcd::GetMcuState(
            &state,
            xcdHandle).IsFailure())
        {
            NN_LOG("%s: Failed to get state\n", NN_CURRENT_FUNCTION_NAME);
            return false;
        }
    }

    // NFC イベントの通知を受け取れるようにする
    if (nn::xcd::SetNfcEvent(
        &m_GeneralEvent,
        &m_DetectEvent,
        xcdHandle).IsFailure())
    {
        NN_LOG("%s: Failed to set events\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    nn::os::InitializeMultiWaitHolder(&m_GeneralEventHolder, &m_GeneralEvent);
    nn::os::InitializeMultiWaitHolder(&m_DetectEventHolder, &m_DetectEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_GeneralEventHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_DetectEventHolder);

    m_IsNfcInitialized = true;

    return true;
}

bool XcdNfcBase::DetectTag(const nn::xcd::NfcDiscoveryParameter& parameter) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        return false;
    }

    for (int i = 0; i < nnt::xcd::RetryCount; i++)
    {
        {
            auto result = nn::xcd::StartNfcDiscovery(parameter, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-1\n", i + 1);
            }
        }

        // タグの検出待ち
        if (WaitEvent() != EventType::Detect)
        {
            NNT_XCD_RETRY("Retry #%d-2\n", i + 1);
        }

        // タグ検出イベントであることを確認
        nn::xcd::NfcInfo info = {};
        {
            auto result = nn::xcd::GetNfcInfo(&info, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-3\n", i + 1);
            }
        }

        if (info.reason == nn::xcd::NfcEventReason_Detected)
        {
            return true;
        }

        NNT_XCD_RETRY("Retry #%d-4\n", i + 1);
    }

    return false;
}

bool XcdNfcBase::StopTagDetection() NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        return false;
    }

    return nn::xcd::StopNfcDiscovery(xcdHandle).IsSuccess();
}

bool XcdNfcBase::ReadNtagWithRetry(
    nn::xcd::NfcInfo* pOutInfo,
    const nn::xcd::NtagReadParameter readParameter) NN_NOEXCEPT
{
    return ReadNtagWithRetry(pOutInfo, readParameter, true, true);
}

bool XcdNfcBase::ReadNtagWithRetry(
    nn::xcd::NfcInfo* pOutInfo,
    const nn::xcd::NtagReadParameter readParameter,
    bool needsDetect,
    bool needsPrintTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutInfo);

    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        return false;
    }

    for (int i = 0; i < nnt::xcd::RetryCount; i++)
    {
        if (needsDetect && !DetectTagBasic())
        {
            NNT_XCD_RETRY("Retry #%d-1\n", i + 1);
        }

        auto startTick = nn::os::GetSystemTick();
        {
            auto result = nn::xcd::StartNtagRead(readParameter, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-2\n", i + 1);
            }
        }

        // タグの読み取り待ち
        if (WaitEvent() != EventType::General)
        {
            NNT_XCD_RETRY("Retry #%d-3\n", i + 1);
        }

        // 読み取り完了イベントであることを確認
        {
            auto result = nn::xcd::GetNfcInfo(pOutInfo, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-4\n", i + 1);
            }
        }

        if (pOutInfo->reason == nn::xcd::NfcEventReason_ReadFinish)
        {
            if (needsPrintTime)
            {
                auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
                NN_LOG("  Read time: %lld msec\n", diffTime.GetMilliSeconds());
            }
            return true;
        }

        NNT_XCD_RETRY("Retry #%d-5\n", i + 1);
    }

    return false;
}

bool XcdNfcBase::WriteNtagWithRetry(
    const nn::xcd::NtagWriteParameter& parameter) NN_NOEXCEPT
{
    // 誤ってシステム領域には書き込まないようにする
    NN_ABORT_UNLESS(
        !nnt::xcd::ContainsSystemArea(parameter.ntagWriteData));

    if (!ProcessDummyRead())
    {
        return false;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        return false;
    }

    for (int i = 0; i < nnt::xcd::RetryCount; i++)
    {
        auto startTick = nn::os::GetSystemTick();
        {
            auto result = nn::xcd::StartNtagWrite(parameter, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-1\n", i + 1);
            }
        }

        // タグへの書き込み待ち
        if (WaitEvent() != EventType::General)
        {
            NNT_XCD_RETRY("Retry #%d-2\n", i + 1);
        }

        // 書き込み完了イベントであることを確認
        nn::xcd::NfcInfo info = {};
        {
            auto result = nn::xcd::GetNfcInfo(&info, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-3\n", i + 1);
            }
        }

        if (info.reason == nn::xcd::NfcEventReason_WriteFinish)
        {
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            NN_LOG("  Write time: %lld msec\n", diffTime.GetMilliSeconds());
            return true;
        }
    }

    return false;
}

bool XcdNfcBase::VerifyWrittenData(
    const nn::xcd::NtagWriteParameter& writeParameter) NN_NOEXCEPT
{
    // Type2 タグの全域を読み込む
    nn::xcd::NtagReadParameter readParameter = {};
    readParameter.timeoutMsec        = 3000;
    readParameter.isPasswordRequired = false;
    readParameter.tagId.length       = 0;
    readParameter.blockCount         = 3;

    readParameter.addresses[0].startPage = 0x00;
    readParameter.addresses[0].endPage   = 0x39;
    readParameter.addresses[1].startPage = 0x3A;
    readParameter.addresses[1].endPage   = 0x73;
    readParameter.addresses[2].startPage = 0x74;
    readParameter.addresses[2].endPage   = nnt::xcd::NtagReadPageMax;

    nn::xcd::NfcInfo info = {};
    if (!ReadNtagWithRetry(&info, readParameter, false, false))
    {
        return false;
    }

    // 比較しやすくするため、読み込んだデータを連続したバッファに移す
    const size_t BufferSize = nnt::xcd::NtagReadPageCountMax * nn::xcd::Type2TagPageSize;
    char readData[BufferSize] = {};

    for (decltype(info.ntagData.blockCount) i = 0; i < info.ntagData.blockCount; i++)
    {
        auto&& block = info.ntagData.readDataBlocks[i];
        int startAddress = block.address.startPage * nn::xcd::Type2TagPageSize;
        int readBytes = (block.address.endPage - block.address.startPage + 1) * nn::xcd::Type2TagPageSize;
        std::memcpy(&readData[startAddress], block.data, readBytes);
    }

    // 読み込んだデータと、書き込んだデータの一致確認
    for (decltype(writeParameter.ntagWriteData.blockCount) i = 0; i < writeParameter.ntagWriteData.blockCount; i++)
    {
        auto&& block = writeParameter.ntagWriteData.dataBlocks[i];
        int offset = 0;
        int startPage = block.startPageAddress;
        int dataSize = block.dataSize;

        // Activation 領域は個別に比較
        // 書き込み開始位置は必ず 4 ページ以降であるという前提
        if (writeParameter.ntagWriteData.isActivationNeeded &&
            startPage == nnt::xcd::NtagActivationDataPage)
        {
            int activationAddress =
                nnt::xcd::NtagActivationDataPage * nn::xcd::Type2TagPageSize;
            if (std::memcmp(
                &readData[activationAddress],
                nnt::xcd::NtagActivationData,
                nn::xcd::NtagActivationAreaSize) != 0)
            {
                return false;
            }

            // Activation 情報分のアドレスをずらす
            offset = nn::xcd::Type2TagPageSize;
            startPage++;
            dataSize -= offset;

            if (dataSize <= 0)
            {
                continue;
            }
        }

        int startAddress = startPage * nn::xcd::Type2TagPageSize;
        if (std::memcmp(
            &readData[startAddress],
            &block.data[offset],
            dataSize) != 0)
        {
            return false;
        }
    }

    return true;
}

bool XcdNfcBase::WaitDeviceConnection() NN_NOEXCEPT
{
    auto startTick = nn::os::GetSystemTick();

    // NFC デバイスの接続待ち
    while (nn::hid::system::GetNpadsWithNfc(&m_DeviceNpadId, 1) == 0)
    {
        auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        if (diffTime >= nnt::xcd::DeviceAttachTimeout)
        {
            return false;
        }
        nn::os::SleepThread(nnt::xcd::DevicePollInterval);
    }

    if (nn::hid::system::ActivateNfc(m_DeviceNpadId).IsFailure())
    {
        return false;
    }

    // NFC 有効化待ち
    while (!nn::hid::system::IsNfcActivated(m_DeviceNpadId))
    {
        nn::os::SleepThread(nnt::xcd::DevicePollInterval);
    }

    return true;
}

bool XcdNfcBase::GetNfcDeviceHandle(nn::xcd::DeviceHandle* pOutHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);

    return nn::hid::system::GetXcdHandleForNpadWithNfc(pOutHandle, m_DeviceNpadId).IsSuccess();
}

XcdNfcBase::EventType XcdNfcBase::WaitEvent() NN_NOEXCEPT
{
    const auto Timeout = nn::TimeSpan::FromSeconds(5);
    auto* pHolder = nn::os::TimedWaitAny(&m_MultiWait, Timeout);
    if (pHolder == &m_GeneralEventHolder)
    {
        nn::os::ClearSystemEvent(&m_GeneralEvent);
        return EventType::General;
    }
    else if (pHolder == &m_DetectEventHolder)
    {
        nn::os::ClearSystemEvent(&m_DetectEvent);
        return EventType::Detect;
    }
    else
    {
        return EventType::None;
    }
}

void XcdNfcBase::RetryAndWait() NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    NN_ABORT_UNLESS(GetNfcDeviceHandle(&xcdHandle));

    nn::xcd::StopNfcDiscovery(xcdHandle);
    nn::os::SleepThread(nnt::xcd::RetryWait);
}

bool XcdNfcBase::DetectTagBasic() NN_NOEXCEPT
{
    nn::xcd::NfcDiscoveryParameter parameter = {};
    parameter.pollingMask       = nn::xcd::NfcPollingMask_All;
    parameter.activationTimeout = 3000;
    parameter.discoveryPeriod   = 300;

    return DetectTag(parameter);
}

bool XcdNfcBase::ProcessDummyRead() NN_NOEXCEPT
{
    nn::xcd::NtagReadParameter parameter = {};
    parameter.timeoutMsec        = 3000;    // タイムアウト: 3000 msec
    parameter.isPasswordRequired = false;   // パスワード認証なし
    parameter.tagId.length       = 0;       // UID 指定なし
    parameter.blockCount         = 1;

    // ダミーなので 1 ページだけ読む
    parameter.addresses[0].startPage = 0x00;
    parameter.addresses[0].endPage   = 0x00;

    nn::xcd::NfcInfo info = {};
    return ReadNtagWithRetry(&info, parameter);
}
