﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/htcs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_DetectApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include "TA_arg.h"
#include "TA_agentpage.h"
#include "TestAgent.h"

using namespace TestAgent;

namespace {
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    enum testCalcType
    {
        CALC_TYPE_UNKNOWN = 0,
        CALC_TYPE_ADDITION,
        CALC_TYPE_SUBTRACTION,
        CALC_TYPE_MULTIPLICATION,
        CALC_TYPE_DIVISION,
    };

    class AgentTest{
    public:
        bool m_WlanInitialized;
        int  m_wlanMode;

    private:

    public:
        AgentTest() {
            m_WlanInitialized = false;
            m_wlanMode = WLAN_MODE_UNKNOWN;
        }
        virtual ~AgentTest() {
            ;
        }
    };
}

const uint32_t STATE_BUFFER_MAX         = 256;
const uint32_t RESPONSE_SIZE_MAX        = 512;
const uint32_t COMMAND_NAME_MAX         = 32;
const uint32_t COMMAND_BUFFER_MAX       = 128;
const uint32_t SSID_LENGTH_MAX          = 32;
const uint32_t INTERFACE_NAME_MAX       = 16;
const uint32_t IP_ADDRESS_MAX           = 15;
const uint32_t IPERF_PARAMETER_CNT_MAX  = 32;
const uint32_t IPERF_PARAMETER_LEN_MAX  = 128;

int PORT_NUMBER_MAX                     = 65535;
int VALID_RETURN_MAX                    = 100000;

uint8_t tmpBuff[100]; // Avoid buffer flow
uint8_t bssidBuff[6];
uint8_t hashBuff[8];

const char* TestAgentVesrion = "1.0.5";
typedef int (*commandStrProcFuncPtr)(int, char *, char *);
AgentTest testPara;

typedef struct type_name_str_table
{
    char name[COMMAND_NAME_MAX];
    commandStrProcFuncPtr cmdProcFunc;
} typeNameStr_t;

void IncreaseHashValue(int idx);

int ExecAgentGetVersion(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;

    std::sprintf(respStr, "status,COMPLETE,version,%s\r\n", TestAgent::APP_VERSION);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecDeviceGetInfo(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    nn::settings::system::FirmwareVersion fwVer;
    nn::settings::system::SerialNumber serialNum;

    nn::settings::system::GetFirmwareVersion(&fwVer);
    nn::settings::system::GetSerialNumber(&serialNum);
    std::sprintf(respStr, "status,COMPLETE,model,NX,version,%s-%d.%d,serialNum,%s\r\n",
                 fwVer.displayVersion, fwVer.majorRelstep, fwVer.minorRelstep,serialNum.string);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanInitiation(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char *str;
    nn::wlan::MacAddress myMacAddress;
    char myMacString[nn::wlan::MacAddress::MacStringSize];

    if (testPara.m_WlanInitialized) {
        retval = -1;
        return retval;
    }

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        } else if (strcasecmp(str, "mode") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (strcasecmp(str, "infra") == 0) {
                testPara.m_wlanMode = WLAN_MODE_INFRA;
            } else if (strcasecmp(str, "master") == 0) {
                testPara.m_wlanMode = WLAN_MODE_MASTER;
            } else if (strcasecmp(str, "client") == 0) {
                testPara.m_wlanMode = WLAN_MODE_CLIENT;
            } else if (strcasecmp(str, "standalone") == 0) {
                testPara.m_wlanMode = WLAN_MODE_DETECT_SA;
            } else if (strcasecmp(str, "hostdriven") == 0) {
                testPara.m_wlanMode = WLAN_MODE_DETECT_HD;
            } else {
                testPara.m_wlanMode = WLAN_MODE_UNKNOWN;
            }
        }
    }

    switch (testPara.m_wlanMode) {
    case WLAN_MODE_INFRA:
        retval = TestAgent::WlanInfraInitialize();
          nn::wlan::Infra::GetMacAddress(&myMacAddress);
          std::sprintf(respStr, "status,COMPLETE,mac,%s\r\n",
                       myMacAddress.GetString(myMacString));
        break;
    case WLAN_MODE_MASTER:
    case WLAN_MODE_CLIENT:
        retval = TestAgent::WlanLocalInitialize(testPara.m_wlanMode);
        break;
    case WLAN_MODE_DETECT_SA:
    case WLAN_MODE_DETECT_HD:
        retval = TestAgent::WlanDetectInitialize();
        nn::wlan::Detect::GetMacAddress(&myMacAddress);
        std::sprintf(respStr, "status,COMPLETE,mac,%s\r\n",
                     myMacAddress.GetString(myMacString));
        break;
    default:
        retval = -1;
        break;
    }

    if (retval < 0) {
        return retval;
    }
    testPara.m_WlanInitialized = true;

    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanUninitiation(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char* str;
    int retval = -1;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        }
    }

    switch (testPara.m_wlanMode) {
    case WLAN_MODE_INFRA:
        retval = TestAgent::WlanInfraUninitialize(interfaceName);
        break;
    case WLAN_MODE_MASTER:
    case WLAN_MODE_CLIENT:
        retval = TestAgent::WlanLocalUninitialize(testPara.m_wlanMode);
        break;
    case WLAN_MODE_DETECT_SA:
    case WLAN_MODE_DETECT_HD:
        retval = TestAgent::WlanDetectUninitialize();
        break;
    default:
        retval = -1;
        break;
    }
    if (retval < 0) {return retval;}

    if (testPara.m_WlanInitialized) {
        testPara.m_WlanInitialized = false;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanConnect(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char* str;
    char ssid[SSID_LENGTH_MAX + 1];
    nn::wlan::Security security;

    if (testPara.m_WlanInitialized == false) {
        retval = -1;
        return retval;
    }

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "ssid") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > SSID_LENGTH_MAX) {
                retval = -1;
                break;
            }
            std::memset(ssid, 0, SSID_LENGTH_MAX);
            std::strcpy(ssid, str);
            ssid[SSID_LENGTH_MAX] = '\0';
        } else if (strcasecmp(str, "security") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (strcasecmp(str, "open") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wep64open") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wep64Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep64Open;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wep64shared") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wep64Shared;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep64Shared;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wep128open") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wep128Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Open;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wep128shared") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wep128Shared;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Shared;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wpatkip") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_WpaTkip;
                security.groupPrivacyMode = nn::wlan::SecurityMode_WpaTkip;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wpa2tkip") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wpa2Tkip;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Tkip;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wpaaes") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_WpaAes;
                security.groupPrivacyMode = nn::wlan::SecurityMode_WpaAes;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "wpa2aes") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_Wpa2Aes;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
                security.keyIdx           = 0;
            } else if (strcasecmp(str, "staticaes") == 0) {
                security.privacyMode      = nn::wlan::SecurityMode_StaticAes;
                security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
                security.keyIdx           = 0;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "password") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) >= 64) {
                retval = -1;
                break;
            }
            std::memset(security.key, 0, 64);
            std::strcpy((char*)security.key, str);
        }
    }

    if (retval < 0) {
        return retval;
    }

    NN_LOG("Connect to AP %s with %s(%d)\n", ssid, security.key,
           security.privacyMode);
    retval = TestAgent::WlanInfraConnect(ssid, &security);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanStart(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanIfconfig(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char ipAddress[IP_ADDRESS_MAX + 1];
    bool interfaceUp;
    bool dhcpEnabled;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        } else if (strcasecmp(str, "state") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "up") == 0) {
                interfaceUp = true;
            } else if (strcasecmp(str, "down") == 0) {
                interfaceUp = false;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "dhcp") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "1") == 0) {
                dhcpEnabled = true;
            } else if (strcasecmp(str, "0") == 0) {
                dhcpEnabled = false;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "ip") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > IP_ADDRESS_MAX) {
                retval = -1;
                break;
            }
            std::memset(ipAddress, 0, IP_ADDRESS_MAX);
            std::strcpy(ipAddress, str);
            ipAddress[IP_ADDRESS_MAX] = '\0';
        }
    }

    if (retval < 0) {
        return retval;
    }

    NN_LOG("Configure interface %s %s with dhcp %s(%s)\n", interfaceName,
           interfaceUp ? "up" : "down",
           dhcpEnabled ? "enable" : "disable",
           dhcpEnabled ? "dhcp" : ipAddress);
    retval = TestAgent::WlanIfconfig(interfaceName, interfaceUp, dhcpEnabled,
                                     ipAddress);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanRxAntenna(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char state[STATE_BUFFER_MAX];
    nn::wlan::RxAntennaPattern rxantenna;
    char* str;
    nn::Result result;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "rxantenna") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "0") == 0) {
                rxantenna = nn::wlan::RxAntennaPattern_0;
            } else if (strcasecmp(str, "1") == 0) {
                rxantenna = nn::wlan::RxAntennaPattern_1;
            } else if (strcasecmp(str, "2") == 0) {
                rxantenna = nn::wlan::RxAntennaPattern_Both;
            } else {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    std::memset(state, 0, STATE_BUFFER_MAX);
    result = nn::wlan::Infra::ChangeRxAntenna(rxantenna);
    if (result.IsFailure()) {
        retval = -1;
    }

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,%s\r\n", state);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecWlanScan(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0, i, outRssi = 100;
    char state[STATE_BUFFER_MAX];
    nn::wlan::ScanParameters ScanParam;
    nn::wlan::Ssid ScanSsid;
    uint32_t channelBitmap;
    const size_t  buffSize = 20 * 1024; // 20KB
    std::unique_ptr<uint8_t[]> g_scanBuffer(new uint8_t[buffSize]);
    char* str;
    nn::Result result;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "scantype") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "active") == 0) {
                ScanParam.scanType = nn::wlan::ScanType_Active;
            } else if (strcasecmp(str, "passive") == 0) {
                ScanParam.scanType = nn::wlan::ScanType_Passive;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "channel") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            ScanParam.channelCount = 0;
            channelBitmap = std::atoi(str);
            if (channelBitmap & 0x01) {
                ScanParam.channelList[ScanParam.channelCount] = 1;
                ScanParam.channelCount++;
            }
            if (channelBitmap & 0x20) {
                ScanParam.channelList[ScanParam.channelCount] = 6;
                ScanParam.channelCount++;
            }
            if (channelBitmap & 0x400) {
                ScanParam.channelList[ScanParam.channelCount] = 11;
                ScanParam.channelCount++;
            }
        } else if (strcasecmp(str, "scantime") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            ScanParam.channelScanTime = std::atoi(str);
        } else if (strcasecmp(str, "hometime") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            ScanParam.homeChannelTime = std::atoi(str);
        } else if (strcasecmp(str, "ssid") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "@all") == 0) {
                ScanParam.ssidList = nullptr;
                ScanParam.ssidCount = 0;
            } else {
                ScanSsid = nn::wlan::Ssid(str);
                ScanParam.ssidList = &ScanSsid;
                ScanParam.ssidCount = 1;
            }
            ScanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        }
    }

    if (retval < 0) {
        return retval;
    }

    NN_LOG("Scan Param:\n"
            "    Type: %s\n"
            "    ScanTime: %d\n"
            "    HomeTime: %d\n"
            "    SSIDCnt: %d%s%s%s"
            "    ChanCnt: %d\n",
            ScanParam.scanType == nn::wlan::ScanType_Active ? "Active" : "Passive",
            ScanParam.channelScanTime,
            ScanParam.homeChannelTime,
            ScanParam.ssidCount,
            ScanParam.ssidCount == 1 ? "(" : "",
            ScanParam.ssidCount == 1 ? (char*)ScanParam.ssidList[0].GetSsidData() : "",
            ScanParam.ssidCount == 1 ? ")\n" : "\n",
            ScanParam.channelCount);
    if (ScanParam.channelCount > 0) {
        NN_LOG("        Channel:");
        for (i = 0; i < ScanParam.channelCount; i++) {
            NN_LOG(" %d", ScanParam.channelList[i]);
        }
        NN_LOG("\n");
    }
    std::memset(state, 0, STATE_BUFFER_MAX);
    nn::wlan::Infra::StartScan(g_scanBuffer.get(), buffSize, ScanParam);
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer.get());

    uint32_t bssCount = resultReader.GetCount();
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

    NN_LOG("***** SCAN RESULT *****\n");
    NN_LOG("The number of found APs : %d\n", bssCount);

    for (i = 0; i < bssCount; i++) {
        // スキャン結果からさらにビーコン解析用のオブジェクトに切り出す
        nn::wlan::BeaconDescriptionReader beacon =
                resultReader.GetNextDescription();
        // 見つかったAPのSSIDとRSSI値を表示
        NN_LOG("[%d]%s:%d[dBm]ch%d\n", i,
               beacon.GetSsid().GetHexString(ssidStr), beacon.GetRssi(),
               beacon.GetChannel());

        // ssidが指定のものであれば、それを出力用変数に代入
        if (ScanParam.ssidCount == 1) {
            if (beacon.GetSsid() == ScanParam.ssidList[0]) {
                outRssi = beacon.GetRssi();
                NN_LOG("outRssi: %d\n", outRssi);
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,ap_bssid,%d\r\n", outRssi);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
} // NOLINT(impl/function_size)

int ExecWlanDisconnect(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;

    if (testPara.m_WlanInitialized == false) {
        retval = -1;
        return retval;
    }

    NN_LOG("Disconnect with AP\n");
    retval = TestAgent::WlanInfraDisconnect();

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWowlSetSocketInfo(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    int  socketId;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "socketid") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            socketId = std::atoi(str);
            if ((socketId <= 0) || (socketId >PORT_NUMBER_MAX)) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    retval = TestAgent::WowlSetSocketInfo(socketId);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecWlanIfstate(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char state[STATE_BUFFER_MAX];
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        }
    }

    if (retval < 0) {
        return retval;
    }

    std::memset(state, 0, STATE_BUFFER_MAX);
    retval = TestAgent::WlanIfstate(interfaceName, &state[0]);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,%s\r\n", state);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecSocketConnect(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char ipAddress[IP_ADDRESS_MAX + 1];
    int  serverPort;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "serverip") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > IP_ADDRESS_MAX) {
                retval = -1;
                break;
            }
            std::memset(ipAddress, 0, IP_ADDRESS_MAX);
            std::strcpy(ipAddress, str);
            ipAddress[IP_ADDRESS_MAX] = '\0';
        } else if (strcasecmp(str, "port") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            serverPort = std::atoi(str);
            if ((serverPort <= 0) || (serverPort >PORT_NUMBER_MAX)) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    retval = TestAgent::SocketConnect(&ipAddress[0], serverPort);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,result,%d\r\n", retval);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecSocketDisconnect(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    int  socketId;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "socketid") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            socketId = std::atoi(str);
            if ((socketId <= 0) || (socketId >PORT_NUMBER_MAX)) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    retval = TestAgent::SocketDisconnect(socketId);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecSendPing(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char ipAddress[IP_ADDRESS_MAX + 1];
    int  dataSize;
    int  sentInterval;
    int  sentCount;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "ip") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > IP_ADDRESS_MAX) {
                retval = -1;
                break;
            }
            std::memset(ipAddress, 0, IP_ADDRESS_MAX);
            std::strcpy(ipAddress, str);
            ipAddress[IP_ADDRESS_MAX] = '\0';
        } else if (strcasecmp(str, "pktsize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            dataSize = std::atoi(str);
            if ((dataSize <= 0) || (dataSize >4096)) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "interval") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            sentInterval = std::atoi(str);
            if (sentInterval < 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "sendcnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            sentCount = std::atoi(str);
            if (sentCount <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    retval = TestAgent::SendPing(&ipAddress[0], dataSize, sentInterval, sentCount);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,sendcnt,%d,recvcnt,%d\r\n", sentCount, retval);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecCalc_value(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    int valueFirst;
    int valueSecond;
    int valueTmp;
    testCalcType calcType;
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "calctype") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            valueTmp = std::atoi(str);
            switch (valueTmp) {
            case 1:
            case 2:
            case 3:
            case 4:
                calcType = (testCalcType) valueTmp;
                break;
            default:
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "value1") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            valueFirst = std::atoi(str);
        } else if (strcasecmp(str, "value2") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            valueSecond = std::atoi(str);
        }
    }

    switch (calcType) {
    case CALC_TYPE_ADDITION:
        valueTmp = valueFirst + valueSecond;
        break;
    case CALC_TYPE_SUBTRACTION:
        valueTmp = valueFirst - valueSecond;
        break;
    case CALC_TYPE_MULTIPLICATION:
        valueTmp = valueFirst * valueSecond;
        break;
    case CALC_TYPE_DIVISION:
        valueTmp = valueFirst / valueSecond;
        break;
    default:
        retval = -1;
        break;
    }

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,result,%d\r\n", valueTmp);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecStartIperf(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char state[STATE_BUFFER_MAX];
    char iperfParameter[IPERF_PARAMETER_CNT_MAX][IPERF_PARAMETER_LEN_MAX];
    bool isServer;
    int  displayInterval;
    char ipAddress[IP_ADDRESS_MAX + 1];
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "mode") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "server") == 0) {
                isServer = true;
            } else if (strcasecmp(str, "client") == 0) {
                isServer = false;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "interval") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            displayInterval = std::atoi(str);
            if (displayInterval <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "ip") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > IP_ADDRESS_MAX) {
                retval = -1;
                break;
            }
            std::memset(ipAddress, 0, IP_ADDRESS_MAX);
            std::strcpy(ipAddress, str);
            ipAddress[IP_ADDRESS_MAX] = '\0';
        }
    }

    if (retval < 0) {
        return retval;
    }

    std::memset(iperfParameter, 0, IPERF_PARAMETER_LEN_MAX);
    std::sprintf(respStr, "%s%s -i %d", isServer ? "-s" : "-c",
                 isServer ? "" : ipAddress, displayInterval);
    retval = TestAgent::WlanStartIperf(interfaceName, &state[0]);

    if (retval < 0) {
        return retval;
    }

    std::sprintf(respStr, "status,COMPLETE,%s\r\n", state);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecDetectSaStart(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    int  channel = 1;
    int  tx_interval = 500;
    int  txCount = 10;
    int  idleCount = 10;
    int  rxStart = 1;
    int  rxCount = 2;
    int  data_size = 500;
    char tmp_str[3];
    char *str;

    memset(&hashBuff[0], 0x00, 8);
    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        } else if (strcasecmp(str, "channel") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            channel = std::atoi(str);
            if (channel <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "interval") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            tx_interval = std::atoi(str);
            if (tx_interval <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "txcnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            txCount = std::atoi(str);
        } else if (strcasecmp(str, "idlecnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            idleCount = std::atoi(str);
        } else if (strcasecmp(str, "rxstart") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            rxStart = std::atoi(str);
        } else if (strcasecmp(str, "rxcnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            rxCount = std::atoi(str);
        } else if (strcasecmp(str, "hash") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
        } else if (strcasecmp(str, "datasize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            data_size = std::atoi(str);
            if (data_size <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    TestAgent::DetectStandAloneStart(tx_interval, txCount, idleCount, rxStart,
                                   rxCount, &hashBuff[0], data_size, channel);

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecDetectSaStop(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    int  channel = 1;
    char *str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "channel") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            channel = std::atoi(str);
            if (channel <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    TestAgent::DetectStandAloneStop();

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}
int ExecDetectSaFetchFrame(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    bool isStart;
    char *str;

    isStart = false;
    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "command") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "start") == 0) {
                isStart = true;
            } else if (strcasecmp(str, "stop") == 0) {
                isStart = false;
            } else {
                retval = -1;
                break;
            }
            }
        }

    if (retval < 0) {
        return retval;
    }

    if (isStart)
        TestAgent::DetectStandAloneStartFetchFrame();
    else
        TestAgent::DetectStandAloneStopFetchFrame();

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecDetectSaSetHash(int sock, char *pcmdStr, char *respStr)
{
    int retval = 0;
    int sentBytes;
    int  repeat_cnt = 1;
    char tmp_str[3];
    char* str;

    memset(&hashBuff[0], 0x00, 8);
    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "hash_start") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
        } else if (strcasecmp(str, "count") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            repeat_cnt = std::atoi(str);
            if (repeat_cnt <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    ClearHashFilter();
    SetHashFilter(&hashBuff[0]);
    repeat_cnt--;
    while (true) {
        if (repeat_cnt <= 0)
            break;
        IncreaseHashValue(7);
        SetHashFilter(&hashBuff[0]);
        repeat_cnt--;
    }
    DetectStandAloneSetHashList();

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecDetectHdStart(int sock, char *pcmdStr, char *respStr)
{
    int retval = 0;
    int sentBytes;
    bool isSend;
    int  channel = 1;
    int  tx_interval = 500;
    int  txCount = 10;
    int  idleCount = 10;
    int  rxStart = 1;
    int  rxCount = 2;
    int  data_size = 500;
    char tmp_str[3];
    char* str;

    memset(&hashBuff[0], 0x00, 8);
    isSend = true;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "send") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            if (strcasecmp(str, "enable") == 0) {
                isSend = true;
            } else if (strcasecmp(str, "disable") == 0) {
                isSend = false;
            } else {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "channel") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            channel = std::atoi(str);
            if (channel <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "interval") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            tx_interval = std::atoi(str);
            if (tx_interval <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "txcnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            txCount = std::atoi(str);
        } else if (strcasecmp(str, "idlecnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            idleCount = std::atoi(str);
        } else if (strcasecmp(str, "rxstart") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            rxStart = std::atoi(str);
        } else if (strcasecmp(str, "rxcnt") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            rxCount = std::atoi(str);
        } else if (strcasecmp(str, "hash") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
        } else if (strcasecmp(str, "datasize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            data_size = std::atoi(str);
            if (data_size <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    TestAgent::DetectHostDrivenStart(tx_interval, txCount, idleCount,
                                   rxStart, rxCount, &hashBuff[0],
                                   data_size, isSend, channel);

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
} // NOLINT(impl/function_size)

int ExecDetectHdStop(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;

    TestAgent::DetectHostDrivenStop();

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

void IncreaseHashValue(int idx)
{
    if (idx < 0) return;
    if (hashBuff[idx] == 255) {
        hashBuff[idx] = 0;
        IncreaseHashValue(idx - 1);
    } else {
        hashBuff[idx]++;
    }
}

int ExecDetectHdOneShot(int sock, char *pcmdStr, char *respStr)
{
    int retval = 0;
    int sentBytes;
    bool isSetMac;
    bool isHashChange;
    int  interval = 10;
    int  repeat_cnt = 1;
    int  seq = 0;
    int  data_size;
    int  cmd;
    char tmp_str[3];
    nn::os::Tick sysTick;
    nn::wlan::MacAddress myMacAddress;
    char* str;

    memset(&bssidBuff[0], 0x00, 6);
    memset(&hashBuff[0], 0x00, 8);
    isSetMac = false;
    isHashChange = false;
    cmd = 2;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "cmd") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            cmd = std::atoi(str);
        } else if (strcasecmp(str, "hash") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
            isHashChange = false;
        } else if (strcasecmp(str, "hash_start") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
            isHashChange = true;
        } else if (strcasecmp(str, "datasize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            data_size = std::atoi(str);
            if (data_size <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "interval") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            interval = std::atoi(str);
            if (interval <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "count") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            repeat_cnt = std::atoi(str);
            if (repeat_cnt <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "seq_start") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            seq = std::atoi(str);
            if (seq < 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "macaddr") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != nn::wlan::MacAddress::MacStringSize - 1) {
                retval = -1;
                break;
            }

            for (int i = 0; i < 6; i++) {
                tmp_str[0] = str[3 * i];
                tmp_str[1] = str[3 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&bssidBuff[0], &tmpBuff[0], 6);
            isSetMac = true;
        }
    }

    if (retval < 0) {
        return retval;
    }

    nn::wlan::Detect::GetMacAddress(&myMacAddress);
    sysTick = nn::os::GetSystemTick();
    if (isSetMac)
        TestAgent::DetectHostDrivenOneShot(&hashBuff[0], data_size, cmd,
                                         &bssidBuff[0], seq);
    else
        TestAgent::DetectHostDrivenOneShot(&hashBuff[0], data_size, cmd,
                                         myMacAddress.GetMacAddressData(), seq);
    repeat_cnt--;
    while (true) {
        if (repeat_cnt <= 0)
            break;
        if((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > interval) {
            sysTick = nn::os::GetSystemTick();
            seq++;
            if (isHashChange) IncreaseHashValue(7);
            TestAgent::DetectHostDrivenOneShot(&hashBuff[0], data_size, cmd,
                                             myMacAddress.GetMacAddressData(), seq);
            repeat_cnt--;
        } else {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
} // NOLINT(impl/function_size)

int ExecDetectSimuPeerTx(int sock, char *pcmdStr, char *respStr)
{
    int retval = 0;
    int sentBytes;
    bool isSetMac;
    int  peer_cnt = 1;
    int  data_size;
    char tmp_str[3];
    nn::os::Tick sysTick;
    nn::wlan::MacAddress myMacAddress;
    char* str;

    memset(&bssidBuff[0], 0x00, 6);
    memset(&hashBuff[0], 0x00, 8);
    isSetMac = false;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "hash") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
        } else if (strcasecmp(str, "datasize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            data_size = std::atoi(str);
            if (data_size <= 0) {
                retval = -1;
                break;
            }
        } else if (strcasecmp(str, "macstart") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != nn::wlan::MacAddress::MacStringSize - 1) {
                retval = -1;
                break;
            }

            for (int i = 0; i < 6; i++) {
                tmp_str[0] = str[3 * i];
                tmp_str[1] = str[3 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&bssidBuff[0], &tmpBuff[0], 6);
            isSetMac = true;
        } else if (strcasecmp(str, "peercount") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            peer_cnt = std::atoi(str);
            if (peer_cnt <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    TestAgent::DetectSimulatePeerTx(&hashBuff[0], &bssidBuff[0], peer_cnt, data_size);

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
} // NOLINT(impl/function_size)

int ExecDetectClearBuf(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;

    TestAgent::ClearRecvDetectFrameBuff();

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecDetectCheckBuf(int sock, char *pcmdStr, char *respStr)
{
    int retval = 0;
    int sentBytes;
    int  data_size;
    int  recv_cnt;
    char tmp_str[3];
    char* str;

    memset(&bssidBuff[0], 0x00, 6);
    memset(&hashBuff[0], 0x00, 8);

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "macaddr") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != nn::wlan::MacAddress::MacStringSize - 1) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 6; i++) {
                tmp_str[0] = str[3 * i];
                tmp_str[1] = str[3 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&bssidBuff[0], &tmpBuff[0], 6);
        } else if (strcasecmp(str, "hash") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) != 16) {
                retval = -1;
                break;
            }
            for (int i = 0; i < 8; i++) {
                tmp_str[0] = str[2 * i];
                tmp_str[1] = str[2 * i + 1];
                tmp_str[2] = 0x00;
                sscanf(tmp_str, "%02x", (unsigned int *)&tmpBuff[i]);
            }
            std::memcpy(&hashBuff[0], &tmpBuff[0], 8);
        } else if (strcasecmp(str, "datasize") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);
            data_size = std::atoi(str);
            if (data_size <= 0) {
                retval = -1;
                break;
            }
        }
    }

    if (retval < 0) {
        return retval;
    }

    recv_cnt = TestAgent::FindDetectFrameByHash(&bssidBuff[0], &hashBuff[0],
                                              data_size);

    std::sprintf(respStr, "status,COMPLETE,recvcnt,%d\r\n", recv_cnt);
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return 0;
}

int ExecWlanReset(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;
    int retval = 0;
    char interfaceName[INTERFACE_NAME_MAX + 1];
    char* str;

    while (true) {
        str = strtok_r(NULL, ",", &pcmdStr);
        if (str == NULL || str[0] == '\0')
            break;

        if (strcasecmp(str, "interface") == 0) {
            str = strtok_r(NULL, ",", &pcmdStr);

            if (std::strlen(str) > INTERFACE_NAME_MAX) {
                retval = -1;
                break;
            }
            std::memset(interfaceName, 0, INTERFACE_NAME_MAX);
            std::strcpy(interfaceName, str);
            interfaceName[INTERFACE_NAME_MAX] = '\0';
        }
    }

    if (retval < 0) {
        return retval;
    }

    if (testPara.m_WlanInitialized) {
        WlanInfraUninitialize(interfaceName);
        testPara.m_WlanInitialized = false;
    }

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        retval = -1;
    }

    return retval;
}

int ExecAgentClose(int sock, char *pcmdStr, char *respStr)
{
    int sentBytes;

    std::sprintf(respStr, "status,COMPLETE\r\n");
    sentBytes =nn::htcs::Send(sock, respStr, std::strlen(respStr), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return -1;
    }

    return (VALID_RETURN_MAX + 1);
}

typeNameStr_t nameStr[] =
{
    {"ca_get_version",         ExecAgentGetVersion},
    {"device_get_info",        ExecDeviceGetInfo},
    {"calc_value",             ExecCalc_value},
    {"ca_close",               ExecAgentClose},

    {"wlan_reset",             ExecWlanReset},
    {"wlan_initiation",        ExecWlanInitiation},
    {"wlan_uninitiation",      ExecWlanUninitiation},
    {"wlan_ifconfig",          ExecWlanIfconfig},
    {"wlan_ifstate",           ExecWlanIfstate},
    {"send_ping",              ExecSendPing},
    {"start_iperf",            ExecStartIperf},

    {"wlan_connect",           ExecWlanConnect},
    {"wlan_start",             ExecWlanStart},
    {"wlan_scan",              ExecWlanScan},
    {"wlan_rxantenna",         ExecWlanRxAntenna},
    {"wlan_disconnect",        ExecWlanDisconnect},

    {"wowl_setinfo",           ExecWowlSetSocketInfo},
    {"socket_connect",         ExecSocketConnect},
    {"socket_disconnect",      ExecSocketDisconnect},

    {"detect_sa_start",        ExecDetectSaStart},
    {"detect_sa_stop",         ExecDetectSaStop},
    {"detect_sa_fetch_frame",  ExecDetectSaFetchFrame},
    {"detect_sa_set_hash",     ExecDetectSaSetHash},
    {"detect_hd_start",        ExecDetectHdStart},
    {"detect_hd_stop",         ExecDetectHdStop},
    {"detect_hd_oneshot",      ExecDetectHdOneShot},
    {"detect_simulate_peer_tx",ExecDetectSimuPeerTx},
    {"clear_detect_recv_buf",  ExecDetectClearBuf},
    {"check_detect_recv_buf",  ExecDetectCheckBuf},

    {"", NULL},
};

// 送られてきたデータをそのまま相手に送り返す
int ProcessCmd(int sock, int gui)
{
    int ret;
    char respBuffer[RESPONSE_SIZE_MAX];
    char cmdName[COMMAND_NAME_MAX];
    char cmdBuffer[COMMAND_BUFFER_MAX];
    char *pcmdStr = NULL;
    nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, cmdBuffer, sizeof(cmdBuffer) - 1, 0);

    if (receivedBytes < 0) {
        NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
        return 0;
    } else if (receivedBytes == 0) {
        return -1;
    }

    cmdBuffer[receivedBytes - 3] = '\0';

    std::memset(respBuffer, 0, RESPONSE_SIZE_MAX);
    std::sprintf(respBuffer, "status,RUNNING\r\n");
    nn::htcs::ssize_t sentBytes =nn::htcs::Send(sock, respBuffer, std::strlen(respBuffer), 0);
    if (sentBytes < 0) {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
    }
    if (gui) TestAgent::SetCmdList(cmdBuffer);

    std::memcpy(cmdName, strtok_r((char *)cmdBuffer, ",", (char **)&pcmdStr), COMMAND_NAME_MAX);
    int i = 0, isFound = 0;
    while (nameStr[i].cmdProcFunc != NULL) {
        if (std::strcmp(nameStr[i].name, cmdName) == 0) {
            isFound = 1;
            break;
        }
        i++;
    }

    if (isFound == 0) {
        std::memset(respBuffer, 0, RESPONSE_SIZE_MAX);
        std::sprintf(respBuffer, "status,INVALID\r\n");
        nn::htcs::ssize_t sentBytes =nn::htcs::Send(sock, respBuffer, std::strlen(respBuffer), 0);
        if (sentBytes < 0) {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
        }
        return 0;
    }

    std::memset(respBuffer, 0, RESPONSE_SIZE_MAX);
    NN_LOG("Execute %s\n", nameStr[i].name);
    ret = nameStr[i].cmdProcFunc(sock, pcmdStr, respBuffer);
    if (gui) {
        respBuffer[std::strlen(respBuffer) - 2] = '\0';
        for (i = 0; i < 7; i++)
            respBuffer[i] = ' ';
        TestAgent::SetCmdList(respBuffer);
    }
    if (ret < 0) {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        std::memset(respBuffer, 0, RESPONSE_SIZE_MAX);
        std::sprintf(respBuffer, "status,INVALID\r\n");
        nn::htcs::Send(sock, respBuffer, std::strlen(respBuffer), 0);
        NN_LOG("Failed to execute %s command\n", nameStr[i].name);
        return 0;
    }

    return ret;
}

NN_ALIGNAS(4096) char  g_AgentThreadStack[40960];
nn::os::ThreadType  g_AgentThread;
static bool g_ExitFlagAgent = false;
int serviceSocket;

void AgentThreadFunc(void* arg)
{
    int result;
    int sock;
    g_ExitFlagAgent = true;
    char newtest = '\r';

    testPara.m_WlanInitialized = false;

    // 接続を待ち受ける
    while (g_ExitFlagAgent) {
        NN_LOG("Waiting for connection from host.\n");
        TestAgent::SetAgentStatus(0);
        sock = nn::htcs::Accept(serviceSocket, nullptr);
        TestAgent::SetCmdList(&newtest);
        TestAgent::SetAgentStatus(1);
        NN_LOG("Accepted client on host.\n");

        while (g_ExitFlagAgent) {
            result = ProcessCmd(sock, 1);
            if ((result < 0) || (result > VALID_RETURN_MAX)) {
                break;
            }
        }

        /* Receive close command */
        if (result > VALID_RETURN_MAX) {
            break;
        } else {
            nn::htcs::Close(sock);
        }
    }
}

void StartAgentThread() NN_NOEXCEPT
{
    // サービスを登録
    nn::htcs::SockAddrHtcs serviceAddr;
    nn::Result result;

    nn::htcs::Initialize(&Allocate, &Deallocate);
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serviceAddr.portName.name, "NCLTestAgent");
    serviceSocket = nn::htcs::Socket();
    nn::htcs::Bind(serviceSocket, &serviceAddr);
    nn::htcs::Listen(serviceSocket, 1);

    /* AgentThreadスレッド作成 */
    result = nn::os::CreateThread( &g_AgentThread, AgentThreadFunc, NULL, g_AgentThreadStack, sizeof(g_AgentThreadStack), nn::os::DefaultThreadPriority );

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread( &g_AgentThread );
}

void StopAgentThread() NN_NOEXCEPT
{
    g_ExitFlagAgent = false;

    nn::htcs::Close(serviceSocket);
    NN_LOG("Socket Closed.\n");

    nn::os::WaitThread( &g_AgentThread );
    nn::os::DestroyThread( &g_AgentThread );

    nn::htcs::Finalize();
}

void TestAgentStart()
{
    // サービスを登録
    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serviceAddr.portName.name, "NCLTestAgent");
    int result;
    int sock;

    testPara.m_WlanInitialized = false;
    int serviceSocket = nn::htcs::Socket();
    nn::htcs::Bind(serviceSocket, &serviceAddr);
    nn::htcs::Listen(serviceSocket, 1);

    // 接続を待ち受ける
    while (true) {
        NN_LOG("Waiting for connection from host.\n");
        sock = nn::htcs::Accept(serviceSocket, nullptr);
        NN_LOG("Accepted client on host.\n");

        while (true) {
            result = ProcessCmd(sock, 0);
            if ((result < 0) || (result > VALID_RETURN_MAX)) {
                break;
            }
        }

        /* Receive close command */
        if (result > VALID_RETURN_MAX) {
            break;
        } else {
            nn::htcs::Close(sock);
        }
    }

    // 切断
    nn::htcs::Close(serviceSocket);
    NN_LOG("Socket Closed.\n");
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    TestAgent::Initialize();

    Display::EnableDisplayOutput();

    Arg testAgentArg;

    testAgentArg.Initialize();

    if (testAgentArg.Parse()) {
        int modeId = testAgentArg.GetModeId();

        if(modeId == MODE_ID_TEST_AGENT) {
            NN_LOG("Enter test agent mode\n");
            AgentPage agentModePage;
            agentModePage.SetAgentMode(true);
            agentModePage.ApplicationImpl();
            nn::htcs::Initialize(&Allocate, &Deallocate);

            TestAgentStart();

            // 終了
            nn::htcs::Finalize();

            NPad::Finalize();
        } else {
            NN_LOG("Enter manual mode\n");
            AgentPage agentModePage;
            agentModePage.SetAgentMode(false);
            agentModePage.Execute();
            NPad::Finalize();
        }
    }
    else
    {
        NN_LOG("Exit for error...\n");
    }

    TestAgent::Uninitialize();
    return;
}
