﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sstream>
#include <iostream>
#include <iomanip>
#include <string>
#include <vector>

namespace TestAgent {

using namespace std;

/*!--------------------------------------------------------------------------*
 @brief        選択可能値のインタフェース
 *---------------------------------------------------------------------------*/
class ISelectableValue {
public:

    virtual void Next(bool loop = false) = 0;
    virtual void Back(bool loop = false) = 0;

    virtual bool Select(const int& index) = 0;
    virtual bool Deselect(const int& index) = 0;

    // 選択可能なアイテム数
    virtual size_t GetSelectableItemCount() = 0;
    // 選択済みアイテム数
    virtual size_t GetSelectingItemCount() = 0;
    virtual size_t GetSelectingItemCountMax() = 0;
    virtual size_t GetCurrentIndex() = 0;

    virtual string ToStringSelectableItem() = 0;
    virtual string ToStringSelectingItem() = 0;
};

inline ostream& operator<<(ostream& stream, ISelectableValue* svalue) {
    stream << svalue->ToStringSelectableItem();
    return stream;
}

/*!--------------------------------------------------------------------------*
 @brief        汎用な選択可能値
 *---------------------------------------------------------------------------*/
template<typename T>
class SelectableValue: public ISelectableValue {
public:

    SelectableValue() {
        currentIndex = 0;
    }

    virtual ~SelectableValue() {
        Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value) {
        return Find(value, values);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value) {
        return -1;
    }

    virtual bool Select(const int& index) {
        if (index < 0 || index >= GetSelectableItemCount()) {
            return false;
        }

        currentIndex = index;
        return true;
    }

    virtual bool Deselect(const int& index) {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() {
        return values.size();
    }

    virtual size_t GetSelectingItemCount() {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() {
        return 1;
    }

    virtual size_t GetCurrentIndex() {
        return currentIndex;
    }

    virtual void Add(T value, bool isDefault = false) {
        if (isDefault) {
            currentIndex = values.size();
        }

        values.push_back(value);
    }

    virtual void Clear() {
        currentIndex = 0;
        values.clear();
    }

    virtual T GetValue() {
        NN_ASSERT(values.size() != 0)
        return values[currentIndex];
    }

    virtual bool GetValue(T& v) {
        if (values.size() == 0) {
            return false;
        }

        v = values[currentIndex];
        return true;
    }

    virtual void Next(bool loop = false) {
        currentIndex++;

        if (currentIndex == values.size()) {
            if (loop) {
                currentIndex = 0;
            } else {
                currentIndex--;
            }
        }
    }

    virtual void Back(bool loop = false) {
        currentIndex--;

        if (currentIndex < 0) {
            if (loop) {
                currentIndex = values.size() - 1;
            } else {
                currentIndex++;
            }
        }
    }

    virtual string ToStringSelectableItem() {
        ostringstream oss;
        oss << values[currentIndex];
        return oss.str();
    }

    virtual string ToStringSelectingItem() {
        return ToStringSelectableItem();
    }

    size_t Size() {
        return values.size();
    }

    // 暗黙の型変換演算子をオーバーライド
    NN_IMPLICIT operator T() {
        return GetValue();
    }

protected:

    vector<T> values;
    int currentIndex;

    int Find(const T& value, const vector<T>& list) {
        for (int i = 0; i < list.size(); ++i) {
            if (list[i] == value) {
                return i;
            }
        }

        return -1;
    }
};

/*!--------------------------------------------------------------------------*
 @brief        名前つき値
 *---------------------------------------------------------------------------*/
template<typename T>
class NamedValue {
public:

    string Name;
    T Value;

    NamedValue(string name, T value) {
        Name = name;
        Value = value;
    }

    NamedValue(const NamedValue<T>& nv) {
        Name = nv.Name;
        Value = nv.Value;
    }

    virtual ~NamedValue() {
    }

    virtual NamedValue operator=(const NamedValue<T>& nv) {
        Name = nv.Name;
        Value = nv.Value;

        return *this;
    }

    NN_IMPLICIT virtual operator T() {
        return Value;
    }
};

template<typename T>
ostream& operator<<(ostream& stream, NamedValue<T>& nv) {
    stream << nv.Name;
    return stream;
}

/*!--------------------------------------------------------------------------*
 @brief        NamedValue<T>専用の選択可能値
 必要なのはoperator Tのオーバーロードのみ。
 ただ、コンパイラがテンプレートの部分特殊化に未対応なので、
 全体を特殊化している。

 *---------------------------------------------------------------------------*/
template<typename T>
class SelectableValue<NamedValue<T> > : public ISelectableValue {
public:

    SelectableValue() {
        currentIndex = 0;
    }

    virtual ~SelectableValue() {
        Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value) {
        return Find(value, values);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value) {
        return -1;
    }

    virtual size_t GetCurrentIndex() {
        return currentIndex;
    }

    virtual bool Select(const int& index) {
        if (index < 0 || index >= GetSelectableItemCount()) {
            return false;
        }

        currentIndex = index;
        return true;
    }

    virtual bool Deselect(const int& index) {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() {
        return values.size();
    }

    virtual size_t GetSelectingItemCount() {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() {
        return 1;
    }

    virtual void Add(NamedValue<T> value, bool isDefault = false) {
        if (isDefault) {
            currentIndex = values.size();
        }

        values.push_back(value);
    }

    virtual void Clear() {
        currentIndex = 0;
        values.clear();
    }

    virtual NamedValue<T> GetValue() {
        NN_ASSERT(values.size() != 0)
        return values[currentIndex];
    }

    virtual void Next(bool loop = false) {
        currentIndex++;

        if (currentIndex == values.size()) {
            if (loop) {
                currentIndex = 0;
            } else {
                currentIndex--;
            }
        }
    }

    virtual void Back(bool loop = false) {
        currentIndex--;

        if (currentIndex < 0) {
            if (loop) {
                currentIndex = values.size() - 1;
            } else {
                currentIndex++;
            }
        }
    }

    virtual string ToStringSelectableItem() {
        ostringstream oss;
        oss << values[currentIndex];
        return oss.str();
    }

    virtual string ToStringSelectingItem() {
        return ToStringSelectableItem();
    }

    size_t Size() {
        return values.size();
    }

    NN_IMPLICIT operator T() {
        return (T) (NamedValue<T> ) GetValue();
    } // 暗黙の型変換演算子をオーバーライド

protected:

    vector<NamedValue<T> > values;
    int currentIndex;

    int Find(const T& value, const vector<NamedValue<T> >& list) {
        for (int i = 0; i < list.size(); ++i) {
            if (list[i].Value == value) {
                return i;
            }
        }

        return -1;
    }

};
}

