﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {
    struct TestUnitJoinParams
    {
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        nn::wlan::BeaconIndication indication;
        int beaconLostTimeout;
    };

    const char WlanTesStaticAesSSID[] = "WlanTest";
    const size_t WlanTestBufferSize = 100 * 1024;
//    const size_t ThreadStackSize = nn::os::ThreadStackAlignment * 3;
    nn::Result result;
    bool isCheck;

//    const uint64_t WlanTestJoinTimeOut = 180;

    const TestUnitJoinParams NormalJoinParam = {
        nn::wlan::Ssid(WlanTesStaticAesSSID), nn::wlan::MacAddress(), -1,
        { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::BeaconIndication_Enable, 10
    };

    const uint8_t LocalVendorIe[] = {
        0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x78
    };
}

namespace WlanTest {

    class LocalSpectatorNormalInitUnitTest : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

        nn::wlan::MacAddress macAddress;
        nn::wlan::ConnectionStatus joinStatus;

        nn::os::ThreadType testCancelJoinThread;

        LocalApiClass localSpectator;

        nn::wlan::DisconnectClient disClient;

        nn::os::SystemEventType joinEvent;

        nn::wlan::ConnectionStatus connectStatus;
        nn::os::SystemEventType connectEvent;
        nn::wlan::MacAddress putMacAddress;
        uint32_t rxId;
        int32_t rssi;
    };

    class LocalSpectatorNormalUnitTest : public LocalSpectatorNormalInitUnitTest
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalSpectatorNormalInitUnitTest::SetUp();
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenSpectatorMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseSpectatorMode());
            LocalSpectatorNormalInitUnitTest::TearDown();
        }

    };

    void TestApiCancelJoinThread(void* arg) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(arg);
        for (int i = 0; i < 50; i++)
        {
            result = nn::wlan::Local::CancelJoin();
            if (result.IsSuccess())
            {
                isCheck = true;
            }
        }
    }


    TEST_F(LocalSpectatorNormalInitUnitTest, SpectatorMode_No9_No10)
    {
        // 正常値テストNo7 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenSpectatorMode());
        // 正常値テストNo8 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseSpectatorMode());
    }

    //TEST_F(LocalSpectatorNormalUnitTest, StaticAesSecurityJoin_No74_No75)
    //{
    //    // 正常値テストNo74 - No75 CLIENT(Target MASTER)
    //    nn::wlan::Ssid ssid(WlanTesStaticAesSSID);
    //    nn::wlan::Ssid dummySsid;
    //    nn::wlan::MacAddress dummyMacAddress;

    //    macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    //    const TestUnitJoinParams JoinParams[] = {
    //        // 正常値テストNo74 - No75 StaticAes SSID/MacAddress
    //        { ssid, dummyMacAddress, -1,{ nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, nn::wlan::BeaconIndication_Enable, 10 },
    //        { dummySsid, macAddress, -1,{ nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, nn::wlan::BeaconIndication_Enable, 10 },
    //    };

    //    for (auto joinSet : JoinParams)
    //    {
    //        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(joinSet.ssid,
    //            joinSet.bssid, joinSet.channel, joinSet.security, joinSet.indication, joinSet.beaconLostTimeout));

    //        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&joinStatus));
    //        WLANTEST_ASSERT_TRUE(joinStatus.state == nn::wlan::ConnectionState_Joined);
    //    }
    //}

    //TEST_F(LocalSpectatorNormalUnitTest, CancelJion_No76)
    //{
    //    // 正常値テストNo76 CLIENT(Target MASTER)
    //    isCheck = false;

    //    macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    //    static NN_ALIGNAS(4096) uint8_t testApiJoinStack[ThreadStackSize];

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&testCancelJoinThread, TestApiCancelJoinThread, this, testApiJoinStack, ThreadStackSize, nn::os::DefaultThreadPriority));
    //    nn::os::StartThread(&testCancelJoinThread);
    //    nn::os::WaitThread(&testCancelJoinThread);

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid,
    //        NormalJoinParam.bssid, NormalJoinParam.channel, NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));

    //    // 正常値テストNo88
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&joinStatus));
    //    WLANTEST_ASSERT_TRUE(joinStatus.state != nn::wlan::ConnectionState_Joined);

    //    nn::os::DestroyThread(&testCancelJoinThread);
    //}

    //TEST_F(LocalSpectatorNormalUnitTest, Disconnect_No79)
    //{
    //    // 正常値テストNo79
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, &disClient));
    //}

    //TEST_F(LocalSpectatorNormalUnitTest, GetConnectionEvent_No82_No85)
    //{
    //    // 正常値テストNo82
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&joinEvent));

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));
    //    WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&joinEvent, nn::TimeSpan::FromSeconds(WlanTestJoinTimeOut)));

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, &disClient));

    //    // 正常値テストNo85
    //    WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&joinEvent, nn::TimeSpan::FromSeconds(WlanTestJoinTimeOut)));
    //}

    TEST_F(LocalSpectatorNormalUnitTest, GetState_No92)
    {
        nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
        WLANTEST_ASSERT_TRUE(wlanState >= nn::wlan::WlanState_LocalSpectatorIdle && wlanState <= nn::wlan::WlanState_LocalSpectatorScan);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    TEST_F(LocalSpectatorNormalUnitTest, AddDeleteIe_No96_No99)
    {
        uint32_t ieId;

        // 正常値テストNo96 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieId, nn::wlan::ManagementFrameType_Beacon, LocalVendorIe, sizeof(LocalVendorIe)));

        // 正常値テストNo99 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteIe(ieId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, GetMacAddress_No100)
    {
        // 正常値テストNo100 MASTER
        nn::wlan::MacAddress macAddress;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
    }

    //TEST_F(LocalSpectatorNormalUnitTest, GetRssi_No102)
    //{
    //    // 正常値テストNo102
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetRssi(&rssi));
    //    WLANTEST_ASSERT_TRUE(rssi != -128);

    //    // 正常値テストNo105 SPECTATOR
    //    nn::wlan::LinkLevel link;
    //    link = nn::wlan::Local::ConvertRssiToLinkLevel(rssi);
    //    WLANTEST_ASSERT_TRUE(nn::wlan::LinkLevel_0 >= link && nn::wlan::LinkLevel_3 <= link);
    //}

    //TEST_F(LocalSpectatorNormalUnitTest, GetLinkLevel_No104)
    //{
    //    // 正常値テストNo104 SPECTATOR
    //    nn::wlan::LinkLevel link;

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));

    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetLinkLevel(&link));
    //    WLANTEST_ASSERT_TRUE(nn::wlan::LinkLevel_0 >= link && nn::wlan::LinkLevel_3 <= link);
    //}

    TEST_F(LocalSpectatorNormalUnitTest, GetFrame_No110)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        size_t outSize = 0;

        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        // 正常値テストNo110 SPECTATOR(Target MASTER)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, &outSize, localSpectator.m_entRxId));

        WLANTEST_ASSERT_TRUE(std::memcmp(getBuffer.get(), TestMatchInfo[0].matchData, TestMatchInfo[0].matchLength) == 0);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, &disClient));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, CancelGetFrame_No113)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        size_t outSize = 0;

        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));

        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, &outSize, localSpectator.m_entRxId));

        // 正常値テストNo113 SPECTATOR(Target MASTER)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelGetFrame(localSpectator.m_entRxId));

        WLANTEST_ASSERT_TRUE(std::memcmp(getBuffer.get(), TestMatchInfo[0].matchData, TestMatchInfo[0].matchLength) == 0);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, &disClient));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, CreateDeleteRxEntry_No116_No119)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };

        // 正常値テストNo116 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));

        // 正常値テストNo119 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, AddDeleteEtherType_No122_No125)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };
        uint32_t outRxId;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));

        // 正常値テストNo122 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(rxId, Ethertypes[0]));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, Ethertypes[0]));

        // 正常値テストNo125 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, AddRemoveMatchData_No128_No131)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));
        // 正常値テストNo128 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, TestMatchInfo[0]));
        // 正常値テストNo131 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, RxEntryForActionFrame_No145_No148)
    {
        uint32_t actionRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };

        // 正常値テストNo145 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        // 正常値テストNo148 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalSpectatorNormalUnitTest, AddDeleteSubtypeToRxEntryForActionFrame_No151_No154)
    {
        uint32_t actionRxId, outActionRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        // 正常値テストNo151 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actionRxId, nn::wlan::ActionFrameType_Local));

        // 正常値テストNo154 SPECTATOR
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, nn::wlan::ActionFrameType_Local));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest
