﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {
    struct TestUnitJoinParams
    {
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        nn::wlan::BeaconIndication indication;
        int beaconLostTimeout;
    };
    const char WlanTesStaticAesSSID[] = "WlanTest";
    const TestUnitJoinParams NormalJoinParam = {
        nn::wlan::Ssid(WlanTesStaticAesSSID), nn::wlan::MacAddress(), -1,
        { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::BeaconIndication_Enable, 10
    };

} // namespace

namespace WlanTest {

    /********************************************/
    // Class 定義
    /********************************************/
    class LocalSpectoarExceptionStateUnitTest : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    };

    class LocalSpectoarExceptionArgmentUnitTest : public LocalSpectoarExceptionStateUnitTest
    {
    protected:
        nn::Result Result;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

    };

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, OpenCloseMasterMode_No2_No4)
    {
        Result = nn::wlan::Local::OpenMasterMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseMasterMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, OpenCloseClientMode_No5_No7)
    {
        Result = nn::wlan::Local::OpenClientMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseClientMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, CreateDestroyBss_No14_No16)
    {
        nn::wlan::Ssid ssid;
        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };
        nn::wlan::MasterBssParameters bssParameters = {
            -1,
            true,
            30,
            false,
            nn::wlan::RateSetLegacy_11gMask,
            nn::wlan::RateSetLegacy_11gMask,
            security,
            ssid,
            0
        };
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::DestroyBss();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, Connect_No21)
    {
        nn::wlan::Ssid ssid;
        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };
        Result = nn::wlan::Local::Connect(ssid, nn::wlan::MacAddress(), -1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, Connect_No25)
    {
        Result = nn::wlan::Local::CancelConnect();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    //TEST_F(LocalSpectoarExceptionArgmentUnitTest, Join_No29)
    //{
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));
    //    Result = nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout);
    //    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    //}

    //TEST_F(LocalSpectoarExceptionArgmentUnitTest, CancelJoin_No32)
    //{
    //    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Join(NormalJoinParam.ssid, NormalJoinParam.bssid, NormalJoinParam.channel,
    //        NormalJoinParam.security, NormalJoinParam.indication, NormalJoinParam.beaconLostTimeout));

    //    // Sleep
    //    Result = nn::wlan::Local::CancelJoin();
    //    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    //}

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, GetClientStatus_No41)
    {
        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        nn::Bit32 clientStatusBitMap = 0;
        Result = nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, PutFrameRaw_No60)
    {
        const size_t WlanTestBufferSize = 100 * 1024;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        Result = nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(putBuffer.get()), WlanTestBufferSize);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, SetActionFrameWithBeacon_No98)
    {
        uint8_t  txBuffer[Setactionsizenormal] = { 0x40,0xD2,0x8A };
        Result = nn::wlan::Local::SetActionFrameWithBeacon(&txBuffer[0], Setactionsizenormal);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, CancelActionFrameWithBeacon_No100)
    {
        Result = nn::wlan::Local::CancelActionFrameWithBeacon();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalSpectoarExceptionArgmentUnitTest, SetMaxAssociationNumber_No125)
    {
        Result = nn::wlan::Local::SetMaxAssociationNumber(-1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest

