﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define WLANTEST_GTESTLIB  // gtest有効/無効切り替え

#include <memory>
#include "../../Common/testWlan_UnitTest.h"
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace {

// 仕様確定後に有効
//#define EXCEPTION_NOT_TEST

// Bufer
const int  Buffer_size_100k = 100 * 1024;

// SSID
const int ssid_size_min = 1;

// スレッドタイムアウト
const size_t threadStackSize = 20480;

nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;
nn::os::ThreadType wlanStateThread;

NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack[threadStackSize];
std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[Buffer_size_100k]);

// TestName
#define EXCEPTION_TESTNAME                 "Exception"
#define EXCEPTION_STATEREADY_TESTNAME      "Exception_InvalidStateReady"
#define EXCEPTION_STATEIDLE_TESTNAME       "Exception_InvalidStateInfraIdle"
#define EXCEPTION_STATESTA_TESTNAME        "Exception_InvalidStateInfraSta"
#define EXCEPTION_STATESTASCAN_TESTNAME    "Exception_InvalidStateInfraStaScan"
#define EXCEPTION_STATEIDLESCAN_TESTNAME   "Exception_InvalidStateInfraIdleScan"
#define EXCEPTION_STATESLEEP_TESTNAME      "Exception_InvalidStateSleep"
int32_t ExceptionTestNo = 1;

// Assertとなるため、試験無効にするオプションを作成する
// 用途により試験無効にして続きの試験を実施する
#define GETCONNECTIONEVENT_ASSERT
#define GETCONNECTIONSTATUS_ASSERT
#define GETSTATE_ASSERT
#define GETMACADDRESS_ASSERT
#define GETRSSI_ASSERT
#define GETLINKLEVEL_ASSERT

#define WLANTEST_ACTIVESCAN         nn::wlan::ScanType::ScanType_Active
#define WLANTEST_PASSIVESCAN        nn::wlan::ScanType::ScanType_Passive
#define WLANTEST_BROADCASTADDRESS   nn::wlan::MacAddress::CreateBroadcastMacAddress()

//#define WLANTEST_NO34
//#define WLANTEST_NO5_6

struct ScanTestPattern
{
    uint32_t scanSize;
    nn::wlan::ScanParameters scanParam;
};

ScanTestPattern scanTestParam[] = {

    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 1,2,3,4,5 }, 5, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_PASSIVESCAN,{ 36 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_PASSIVESCAN,{ 13 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 1, 6, 11 }, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 1 }, 1, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 1 }, 0, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },

#ifdef SDEVEDEV
    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 48 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
#else
    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 140 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
#endif
    // N49
    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13 }, 13, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
    { Buffer_size_100k,
    { WLANTEST_ACTIVESCAN,{ 36, 40, 44, 48 }, 4, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
};

nn::Result result; //test
} // namespace

namespace WlanTest {

    nn::wlan::ConnectionStatus connectionStatus;
    nn::wlan::Ssid connectSsid(WtestSsid[1]);
    nn::Result Result;

    // ScanBufferサイズ
    const size_t TestScanMinSize = 50 * 1024;

    bool beacon_flg = true;
    size_t     size = 1;


static void InvalidStateReady() NN_NOEXCEPT
{
    // ***************************************************/
    //         StartScan試験
    // ***************************************************/
    // 不正ステート：例外テスト7
    ExceptionTestNo = 7;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[TestScanMinSize]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::StartScan(pScanBuffer.get(), TestScanMinSize,
                    { nn::wlan::ScanType::ScanType_Active,{ 1 }, 1, 120, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() })));

    // ***************************************************/
    //         Connect試験
    // ***************************************************/
    // 不正ステート：例外テスト11
    ExceptionTestNo = 11;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, beacon_flg)));

    // 不正ステート：例外テスト24
    int rssiValue = 0;
    ExceptionTestNo = 24;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetRssi(&rssiValue)));

    // 不正ステート：例外テスト27
    nn::wlan::LinkLevel LinkLevel;
    ExceptionTestNo = 27;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetLinkLevel(&LinkLevel)));

    // ***************************************************/
    //         ChangeRxAntenna試験
    // ***************************************************/
    // 不正ステート：例外テスト29
    ExceptionTestNo = 29;
#ifdef EXCEPTION_NOT_TEST
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_0)));
#else
    ExceptionTestNo++;
#endif

    // ***************************************************/
    //         ConnectWithWps試験
    // ***************************************************/
    // 不正ステート：例外テスト34
    ExceptionTestNo = 34;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEREADY_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No34は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO34
    char sPin[(nn::wlan::WpsPinLength + 1)];
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GenerateWpsPin(&sPin[0], sizeof(sPin)));
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 10)));
#endif
}

static void InvalidStateInfraIdle() NN_NOEXCEPT
{
    // 不正ステート：例外テスト24
    int rssiValue = 0;
    ExceptionTestNo = 24;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLE_TESTNAME, &ExceptionTestNo);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetRssi(&rssiValue)));

    // 不正ステート：例外テスト27
    nn::wlan::LinkLevel LinkLevel;
    ExceptionTestNo = 27;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLE_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetLinkLevel(&LinkLevel)));
}

static void InvalidStateInfraSta() NN_NOEXCEPT
{
    nn::wlan::WlanState State;
    nn::wlan::Infra::GetState(&State);
    NN_LOG("             %s\n", WlanStateStr[State]);

    // ***************************************************/
    //         OpenMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト5
    ExceptionTestNo = 5;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTA_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::OpenMode()));
#endif

    // ***************************************************/
    //         CloseMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト6
    ExceptionTestNo = 6;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTA_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CloseMode()));
#endif

    // ***************************************************/
    //         Connect試験
    // ***************************************************/
    // 不正ステート：例外テスト11
    ExceptionTestNo = 11;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTA_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
    { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, beacon_flg)));

    // ***************************************************/
    //         CancelConnect試験
    // ***************************************************/
    // 不正ステート：例外テスト13
    ExceptionTestNo = 13;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTA_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CancelConnect()));

    // ***************************************************/
    //         ConnectWithWps試験
    // ***************************************************/
    // 不正ステート：例外テスト34
    ExceptionTestNo = 34;
    NN_LOG("             No34は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO34
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTA_TESTNAME, &ExceptionTestNo);
    char sPin[(nn::wlan::WpsPinLength + 1)];
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GenerateWpsPin(&sPin[0], sizeof(sPin)));
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 10)));
#endif
}

static void InvalidStateInfraStaScan() NN_NOEXCEPT
{
    nn::wlan::WlanState State;
    nn::wlan::Infra::GetState(&State);
    NN_LOG("             %s\n", WlanStateStr[State]);

    // ***************************************************/
    //         OpenMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト5
    ExceptionTestNo = 5;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::OpenMode()));
#endif

    // ***************************************************/
    //         CloseMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト6
    ExceptionTestNo = 6;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CloseMode()));
#endif

    // ***************************************************/
    //         StartScan試験
    // ***************************************************/
    // 不正ステート：例外テスト7
    ExceptionTestNo = 7;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[TestScanMinSize]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::StartScan(pScanBuffer.get(),
        TestScanMinSize, { nn::wlan::ScanType::ScanType_Active,{ 1 }, 1, 120, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() })));


    // ***************************************************/
    //         Connect試験
    // ***************************************************/
    // 不正ステート：例外テスト11
    ExceptionTestNo = 11;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
    { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, beacon_flg)));

    // ***************************************************/
    //         Disconnect試験
    // ***************************************************/
    // 不正ステート：例外テスト12
    ExceptionTestNo = 12;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Disconnect()));


    // ***************************************************/
    //         CancelConnect試験
    // ***************************************************/
    // 不正ステート：例外テスト13
    ExceptionTestNo = 13;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CancelConnect()));

    // ***************************************************/
    //         ChangeRxAntenna試験
    // ***************************************************/
    // 不正ステート：例外テスト29
    ExceptionTestNo = 29;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_0)));

    // ***************************************************/
    //         RequestSleep試験
    // ***************************************************/
    // 不正ステート：例外テスト33
    ExceptionTestNo = 33;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESTASCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::RequestSleep()));
}

static void InvalidStateInfraIdleScan() NN_NOEXCEPT
{
    nn::wlan::WlanState State;
    nn::wlan::Infra::GetState(&State);
    NN_LOG("             %s\n", WlanStateStr[State]);
    // ***************************************************/
    //         StartScan試験
    // ***************************************************/
    // 不正ステート：例外テスト7
    ExceptionTestNo = 7;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[TestScanMinSize]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::StartScan(pScanBuffer.get(),
        TestScanMinSize, { nn::wlan::ScanType::ScanType_Active,{ 1 }, 1, 120, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() })));


    // ***************************************************/
    //         Connect試験
    // ***************************************************/
    // 不正ステート：例外テスト11
    ExceptionTestNo = 11;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
    { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, beacon_flg)));

    // ***************************************************/
    //         Disconnect試験
    // ***************************************************/
    // 不正ステート：例外テスト12
    ExceptionTestNo = 12;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

    // ***************************************************/
    //         CancelConnect試験
    // ***************************************************/
    // 不正ステート：例外テスト13
    ExceptionTestNo = 13;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CancelConnect()));

    // 不正ステート：例外テスト24
    int rssiValue = 0;
    ExceptionTestNo = 24;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetRssi(&rssiValue)));

    // 不正ステート：例外テスト27
    nn::wlan::LinkLevel LinkLevel;
    ExceptionTestNo = 27;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetLinkLevel(&LinkLevel)));

    // ***************************************************/
    //         ChangeRxAntenna試験
    // ***************************************************/
    // 不正ステート：例外テスト29
    ExceptionTestNo = 29;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_0)));

    // ***************************************************/
    //         RequestSleep試験
    // ***************************************************/
    // 不正ステート：例外テスト33
    ExceptionTestNo = 33;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::RequestSleep()));

    // ***************************************************/
    //         ConnectWithWps試験
    // ***************************************************/
    // 不正ステート：例外テスト34
    ExceptionTestNo = 34;
    NN_LOG("             No34は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO34
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    char sPin[(nn::wlan::WpsPinLength + 1)];
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GenerateWpsPin(&sPin[0], sizeof(sPin)));
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 10)));
#endif

    // ***************************************************/
    //         OpenMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト5
    ExceptionTestNo = 5;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
// NGのため詳細表示
//    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::OpenMode()));
    result = nn::wlan::Infra::OpenMode();
    NN_LOG("%s %d Line Expect Result Failler code(0x%08x)\n",__FUNCTION__, __LINE__, result.GetInnerValueForDebug());
#endif

    // ***************************************************/
    //         CloseMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト6
    ExceptionTestNo = 6;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATEIDLESCAN_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
// NGのため詳細表示
//    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CloseMode()));
    result = nn::wlan::Infra::CloseMode();
    NN_LOG("%s %d Line Expect Result Failler code(0x%08x)\n",__FUNCTION__, __LINE__, result.GetInnerValueForDebug());
#endif
}

static void InvalidStateSleep() NN_NOEXCEPT
{
    // ***************************************************/
    //         OpenMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト5
    ExceptionTestNo = 5;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::OpenMode()));
#endif

    // ***************************************************/
    //         CloseMode試験
    // ***************************************************/
    // 不正ステート ：例外テスト6
    ExceptionTestNo = 6;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No5,6は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO5_6
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CloseMode()));
#endif

    // ***************************************************/
    //         StartScan試験
    // ***************************************************/
    // 不正ステート：例外テスト7
    ExceptionTestNo = 7;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[TestScanMinSize]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::StartScan(pScanBuffer.get(),
        TestScanMinSize, { nn::wlan::ScanType::ScanType_Active,{ 1 }, 1, 120, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() })));

    // ***************************************************/
    //         StopScan試験
    // ***************************************************/
    // 不正ステート：例外テスト10
    ExceptionTestNo = 10;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::StopScan()));

    // ***************************************************/
    //         Connect試験
    // ***************************************************/
    // 不正ステート：例外テスト11
    ExceptionTestNo = 11;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Connect(connectSsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
    { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, beacon_flg)));

    // ***************************************************/
    //         Disconnect試験
    // ***************************************************/
    // 不正ステート：例外テスト12
    ExceptionTestNo = 12;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::Disconnect()));


    // ***************************************************/
    //         CancelConnect試験
    // ***************************************************/
    // 不正ステート：例外テスト13
    ExceptionTestNo = 13;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::CancelConnect()));

    // ***************************************************/
    //         GetConnectionEvent試験
    // ***************************************************/
    // 不正ステート：例外テスト14
    ExceptionTestNo = 14;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    nn::os::SystemEventType connectEvent;
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectEvent));


    // ***************************************************/
    //         GetConnectionStatus試験
    // ***************************************************/
    // 不正ステート：例外テスト16
    ExceptionTestNo = 16;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));

    // 不正ステート：例外テスト24
    int rssiValue = 0;
    ExceptionTestNo = 24;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetRssi(&rssiValue)));

    // 不正ステート：例外テスト27
    nn::wlan::LinkLevel LinkLevel;
    ExceptionTestNo = 27;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::GetLinkLevel(&LinkLevel)));

    // ***************************************************/
    //         ChangeRxAntenna試験
    // ***************************************************/
    // 不正ステート：例外テスト29
    ExceptionTestNo = 29;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern::RxAntennaPattern_0)));

    // ***************************************************/
    //         RequestSleep試験
    // ***************************************************/
    // 不正ステート：例外テスト33
    ExceptionTestNo = 33;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::RequestSleep()));

    // ***************************************************/
    //         ConnectWithWps試験
    // ***************************************************/
    // 不正ステート：例外テスト34
    ExceptionTestNo = 34;
    WLANTEST_SET_TESTNAME(EXCEPTION_STATESLEEP_TESTNAME, &ExceptionTestNo);
    NN_LOG("             No34は個別実行させる。NGとなり続きの項目に影響がでるため\n");
#ifdef WLANTEST_NO34
    char sPin[(nn::wlan::WpsPinLength + 1)];
    //WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GenerateWpsPin(&sPin[0], sizeof(sPin)));
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 10)));
#endif
}

static void ScanStateThread(void* arg) NN_NOEXCEPT
{
    NN_LOG("             StartStan\n");
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[0].scanSize, scanTestParam[0].scanParam));
    NN_LOG("             StartStan End\n");
}

static void WlanStartScan() NN_NOEXCEPT
{
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&wlanStateThread, ScanStateThread, nullptr, ThreadStack, threadStackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&wlanStateThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    nn::wlan::Infra::GetState(&wlanState);
    if (nn::wlan::WlanState_InfraIdleScan == wlanState)
    {
        InvalidStateInfraIdleScan();
    }
    else if(nn::wlan::WlanState_InfraStaScan == wlanState)
    {
        InvalidStateInfraStaScan();
    }

    nn::os::WaitThread(&wlanStateThread);
    nn::os::DestroyThread(&wlanStateThread);
}

TEST(InfraExceptionUnitTest, ExceptionUnitTest) NN_NOEXCEPT
{


    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Active,
            {1, 6, 11},
            0,
            -1,
            -1,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    // ***************************************************/
    //         InitializeInfraManager試験
    // ***************************************************/
    // 二重呼び ：例外テスト1
    ExceptionTestNo = 1;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::InitializeInfraManager());

    // 不正ステート：例外テスト2
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 2;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_EXPECT_RESULT_NOTSUCCESS( nn::wlan::InitializeInfraManager() );
#else
    ExceptionTestNo ++;
#endif

    // 不正ステート　WlanState_Ready
    nn::wlan::Infra::GetState(&wlanState);

    if (nn::wlan::WlanState_Ready == wlanState)
    {
        InvalidStateReady();
    }

    // ***************************************************/
    //         FinalizeInfraManager試験
    // ***************************************************/
    // 二重呼び ：例外テスト3
    ExceptionTestNo = 3;
#ifdef EXCEPTION_NOT_TEST
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    nn::wlan::FinalizeInfraManager();
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::FinalizeInfraManager() );
    nn::wlan::InitializeInfraManager();
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
#else
    ExceptionTestNo++;
#endif

    // 不正ステート ：例外テスト4
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 4;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         StartScanスレッド
    // ***************************************************/
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

    //WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[0].scanSize, scanTestParam[0].scanParam));

    NN_LOG("             *****************************\n");
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    WlanStartScan();

    // ***************************************************/
    //         Connect状態
    // ***************************************************/

    nn::wlan::Ssid Ssid(WtestSsid[1]);
    nn::os::SystemEventType connectionEvent;
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    NN_LOG("             Connect()成功まで待つ \n");
    while (1)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
                                                                 { nn::wlan::SecurityMode_Wep64Open, nn::wlan::SecurityMode_Wep64Open, 0, "3031323334" }, true));
        nn::os::WaitSystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            break;
        }
        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);

    InvalidStateInfraSta();
    WlanStartScan();
    NN_LOG("             *****************************\n");

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

    // ***************************************************/
    //         不正ステート　WlanState_Sleep　スタート
    // ***************************************************/
    nn::wlan::Infra::RequestSleep();
    nn::wlan::Infra::GetState(&wlanState);

    if (nn::wlan::WlanState_Sleep == wlanState)
    {
        InvalidStateSleep();
    }
    nn::wlan::Infra::RequestWakeUp();
    nn::wlan::Infra::OpenMode();

    // ***************************************************/
    //         不正ステート　WlanState_InfraIdle
    // ***************************************************/
    nn::wlan::Infra::GetState(&wlanState);
    if (nn::wlan::WlanState_InfraIdle == wlanState)
    {
        InvalidStateInfraIdle();
    }

    // ***************************************************/
    //         StartScan試験
    // ***************************************************/
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    std::unique_ptr<uint8_t[]>pScanBuffer(new uint8_t[TestScanMinSize]);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pScanBuffer.get(),
        TestScanMinSize, { nn::wlan::ScanType::ScanType_Active,{ 1 }, 1, 120, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() }));

    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);


    // 第一引数のアドレス不正：例外テスト8
    ExceptionTestNo = 8;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);

    nn::wlan::Ssid ssid;
    nn::Bit8 ssdid_test[32];
    memset(ssdid_test, 0, sizeof(ssdid_test));
    strcmp((char *)ssdid_test, "a");
    ssid.Set( ssdid_test , ssid_size_min - 1 );
    scanParam.ssidList = &ssid;
    scanParam.ssidCount = 1;
    Result = nn::wlan::Infra::StartScan(NULL, size, scanParam);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );

    // 第三引数 ssidListが不正アドレス：例外テスト9
    ExceptionTestNo = 9;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    size = Buffer_size_100k;
    scanParam.ssidList = NULL;
    scanParam.ssidCount = 1;
    std::unique_ptr<char[]>pTestMem(new char[size]);
    WLANTEST_EXPECT_TRUE(pTestMem.get() != nullptr);
    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );
    // ***************************************************/
    //         GetConnectionEvent試験
    // ***************************************************/
    // アドレス不正：例外テスト15
#ifdef GETCONNECTIONEVENT_ASSERT
    ExceptionTestNo = 15;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::Infra::GetState(&wlanState);
    NN_LOG("             GetState(): %s \n", WlanStateStr[wlanState]);
    Result = nn::wlan::Infra::GetConnectionEvent(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetConnectionStatus試験
    // ***************************************************/

    // アドレス不正：例外テスト17
#ifdef GETCONNECTIONSTATUS_ASSERT
    ExceptionTestNo = 17;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    Result = nn::wlan::Infra::GetConnectionStatus(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetState試験
    // ***************************************************/
    // 不正ステート：例外テスト18 実施不可（全ステータスで実行可能なため）
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 18;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
#else
    ExceptionTestNo ++;
#endif

    // アドレス不正：例外テスト19
#ifdef GETSTATE_ASSERT
    ExceptionTestNo = 19;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    Result = nn::wlan::Infra::GetState(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetAllowedChannels試験
    // ***************************************************/
    // 関数未実装：例外テスト20 未実装
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 20;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetMacAddress試験
    // ***************************************************/
    // 不正ステート：例外テスト21 実施不可（全ステータスで実行可能なため）
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 21;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
#else
    ExceptionTestNo ++;
#endif

    // アドレス不正：例外テスト22
#ifdef GETMACADDRESS_ASSERT
    ExceptionTestNo = 22;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    Result = nn::wlan::Infra::GetMacAddress(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetRssi試験
    // ***************************************************/
    // RSSIの取得に失敗：例外テスト23
    ExceptionTestNo = 23;
#ifdef EXCEPTION_NOT_TEST
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    int rssiValue = 0;
    NN_LOG("             Connect()成功まで待つ \n");
    while (1)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
                                                                 { nn::wlan::SecurityMode_Wep64Open, nn::wlan::SecurityMode_Wep64Open, 0, "3031323334" }, true));
        nn::os::WaitSystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            break;
        }
        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }

    NN_LOG("\nConnect():ここで引き抜く\n");
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    Result = nn::wlan::Infra::GetRssi(&rssiValue);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultCommandFailure::Includes(Result) );
    NN_LOG("Result:%x\n", Result);

    nn::wlan::Infra::Disconnect();
    NN_LOG("\nここでテスト26に備えて再度つなぐ\n");
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(60000));
    NN_LOG("             Connect()成功まで待つ \n");
    while (1)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
                                                                 { nn::wlan::SecurityMode_Wep64Open, nn::wlan::SecurityMode_Wep64Open, 0, "3031323334" }, true));
        nn::os::WaitSystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            break;
        }
        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }
#else
    ExceptionTestNo++;
#endif

    // アドレス不正：例外テスト25
    ExceptionTestNo = 25;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    Result = nn::wlan::Infra::GetRssi(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );

    // ***************************************************/
    //         GetLinkLevel試験
    // ***************************************************/
    // RSSIの取得に失敗：例外テスト26
    // GetLinkLevel()内でWaitを実行してその間にアクセスポイントの電源を抜く

    ExceptionTestNo = 26;
#ifdef EXCEPTION_NOT_TEST
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    nn::wlan::LinkLevel LinkLevel;

    NN_LOG("\nConnect():ここで引き抜く\n");
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    Result = nn::wlan::Infra::GetLinkLevel(&LinkLevel);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultCommandFailure::Includes(Result) );
    NN_LOG("Result:%x\n", Result);

    nn::wlan::Infra::Disconnect();
    NN_LOG("\nDisonnect()\n");
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));
#else
    ExceptionTestNo++;
#endif

    // アドレス不正：例外テスト28
#ifdef GETLINKLEVEL_ASSERT
    ExceptionTestNo = 28;
    WLANTEST_SET_TESTNAME( EXCEPTION_TESTNAME, &ExceptionTestNo );
    Result = nn::wlan::Infra::GetLinkLevel(NULL);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );
#else
    ExceptionTestNo ++;
#endif

    // ***************************************************/
    //         GetFwVersion試験
    // ***************************************************/
    // 不正ステート：例外テスト30 実施不可（WlanState_Stop、WlanState_Exitを除く全ステータスで実行可能なため）
#ifdef EXCEPTION_NOT_TEST
    ExceptionTestNo = 30;
    WLANTEST_SET_TESTNAME(EXCEPTION_TESTNAME, &ExceptionTestNo);
#else
    ExceptionTestNo++;
#endif

    // アドレス不正：例外テスト32
    ExceptionTestNo = 32;
    WLANTEST_SET_TESTNAME(EXCEPTION_TESTNAME, &ExceptionTestNo);
    Result = nn::wlan::Infra::GetFwVersion(NULL, 0);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        NN_LOG("argv[%d]=%s\n", argc, argv[i]);
    }

    const bool isEnabled = true;

    WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    ::testing::InitGoogleTest(&argc, argv);

    WLANTEST_STATE_SET(WlanTestState_Infra);
    // 初期化処理
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());

    int result = RUN_ALL_TESTS();

    // 終了処理
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());

    nnt::Exit(result);
}

} // namespace WlanTest

