﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.h"

namespace WlanTest {

void SystemInitialize() NN_NOEXCEPT
{
    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
}

void SystemFinalize() NN_NOEXCEPT
{
}

void ElapsedTime(char* pOutTimeStr, nn::os::Tick srcTick, int64_t tickFreq) NN_NOEXCEPT
{
    if( pOutTimeStr == NULL )
    {
        NN_LOG("%s: null pointer is given\n");
        return;
    }
    int64_t t = ( nn::os::GetSystemTick().GetInt64Value() - srcTick.GetInt64Value() ) / tickFreq;
    int s = static_cast<int>(t % 60);
    int m = static_cast<int>((t / 60) % 60);
    int h = static_cast<int>(t / (60 * 60));
    nn::util::SNPrintf( pOutTimeStr, sizeof("00:00:00"), "%02d:%02d:%02d", h, m, s );
}

void DumpScanResultWithReader(void* pBuf) NN_NOEXCEPT
{
    if( pBuf == NULL )
    {
        return;
    }

    nn::wlan::BeaconScanResultReader resultReader(pBuf);
    uint32_t bssCount = resultReader.GetCount();
    char macStr[nn::wlan::MacAddress::MacStringSize];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    const nn::wlan::VendorInfoElementReader* vie;
    nn::Bit8 matchingOui1[nn::wlan::MacAddress::OuiSize] = {0x00, 0x50, 0xf2};
    nn::Bit8 matchingOui2[nn::wlan::MacAddress::OuiSize] = {0x00, 0x07, 0x40};
    uint8_t matchType = 1;

    NN_LOG("\n\n\n\n ***** SCAN RESULT *****");
    NN_LOG("BSS count : %d\n", bssCount);
    NN_LOG("Scan Result Total Length : %d\n", resultReader.GetTotalResultLength());

    for(uint32_t i = 0; i < bssCount; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();

        NN_LOG("-----------------------%d BSS------------------------\n", i + 1);
        NN_LOG("BSS size : %d\n", beacon.GetLength());
        NN_LOG("IE count : %d\n", beacon.GetIeCount());
        NN_LOG("Channel : %d\n", beacon.GetChannel());
        NN_LOG("RSSI : %d[dBm]\n", beacon.GetRssi());
        NN_LOG("LINKLEVEL : %d\n", beacon.GetLinkLevel());
        NN_LOG("Beacon Period : %d[ms]\n", beacon.GetInterval());
        NN_LOG("BSSID : %s\n", beacon.GetBssid().GetString(macStr));
        NN_LOG("SSID : %s\n", beacon.GetSsid().GetHexString(ssidStr));

        // WPA/WPA2
        if( beacon.IsWpaSupported() == true )
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::WifiInfoElementReader>* pWpa
                = beacon.GetWpaIe();

            NN_LOG("    == WPA: version[%d]\n", pWpa->GetVersion());
            for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                NN_LOG("       PairCipher(%d) %s\n", k, cipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
            }
            NN_LOG("       GroupCipher %s\n", cipherTypeString[pWpa->GetGroupCipherSuite()]);
            NN_LOG("       AKM %s\n", akmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
        }
        if (beacon.IsWpa2Supported())
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::InfoElementReader>* pWpa
                = beacon.GetWpa2Ie();

            NN_LOG("    == WPA2: version[%d]\n", pWpa->GetVersion());
            for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                NN_LOG("       PairCipher(%d) %s\n", k, cipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
            }
            NN_LOG("       GroupCipher %s\n", cipherTypeString[pWpa->GetGroupCipherSuite()]);
            NN_LOG("       AKM %s\n", akmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
        }

        // VIE matching
        vie = beacon.GetVendorIeWithOui(matchingOui1);
        if( vie != NULL )
        {
            NN_LOG("--- OUI matched!\n");
            NN_LOG("IE element id : %d\n", vie->GetElementId());
            NN_LOG("IE length : %d\n", vie->GetLength());
            NN_LOG("IE OUI : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
        }
        else
        {
            NN_LOG("--- OUI not matched...\n");
        }
        vie = beacon.GetVendorIeWithOuiAndType(matchingOui2, matchType);
        if( vie != NULL )
        {
            NN_LOG("--- OUI and type matched!\n");
            NN_LOG("IE element id : %d\n", vie->GetElementId());
            NN_LOG("IE length : %d\n", vie->GetLength());
            NN_LOG("IE OUI : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
            NN_LOG("IE type : %d\n", vie->GetVendorData()[0]);
        }
        else
        {
            NN_LOG("--- OUI and type not matched...\n");
        }
    }
}

void PrintConnectionStatus(nn::wlan::ConnectionStatus* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    char macStr[nn::wlan::MacAddress::MacStringSize];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

    NN_LOG("+++ Connection Status +++\n");
    NN_LOG("STATE : %s\n", wlanConnectionStateStr[pStatus->state]);
    NN_LOG("CAUSE : %d\n", pStatus->cause);
    NN_LOG("CHANNEL : %d\n", pStatus->channel);
    NN_LOG("SSID : %s\n", pStatus->ssid.GetHexString(ssidStr));
    NN_LOG("BSSID : %s\n", pStatus->bssid.GetString(macStr));
    NN_LOG("AID : %d\n", pStatus->aid);
    NN_LOG("REASON CODE : %d\n", pStatus->statusReasonCode);
    NN_LOG("CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
    NN_LOG("BEACON INTERVAL : %d\n", pStatus->beaconInterval);
    NN_LOG("+++++++++++++++++++++++++\n");
}

void PrintClientStatus(nn::wlan::ClientStatus* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    char macStr[nn::wlan::MacAddress::MacStringSize];

    NN_LOG("+++++ Client Status +++++\n");
    NN_LOG("STATE : %s\n", wlanConnectionStateStr[pStatus->state]);
    NN_LOG("CAUSE : %d\n", pStatus->cause);
    NN_LOG("BSSID : %s\n", pStatus->clientMacAddress.GetString(macStr));
    NN_LOG("REASON CODE : %d\n", pStatus->statusReasonCode);
    NN_LOG("CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
    NN_LOG("RSSI  : %d[dbm]\n", pStatus->rssi);
    NN_LOG("+++++++++++++++++++++++++\n");
}

}
