﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstdlib>
#include "../common.h"

namespace {

// Scan buffer
const size_t  buffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];
const int ScanPatternCount = 5;
nn::wlan::ScanParameters g_scanPattern[ScanPatternCount];

// Timer
char g_TimeStr[sizeof("00:00:00")];
int64_t g_TickFreq;
nn::os::Tick g_StartTick;
}

void InitializeScanPatterns()
{
    /* Scan pattern 0 */
    g_scanPattern[0].scanType = nn::wlan::ScanType_Passive;
    g_scanPattern[0].channelCount = 0;
    g_scanPattern[0].channelScanTime = 120;
    g_scanPattern[0].homeChannelTime = 0;
    g_scanPattern[0].ssidList = nullptr;
    g_scanPattern[0].ssidCount = 0;
    g_scanPattern[0].bssid.SetDirect(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF);

    /* Scan pattern 1 */
    g_scanPattern[1].scanType = nn::wlan::ScanType_Active;
    g_scanPattern[1].channelCount = 0;
    g_scanPattern[1].channelScanTime = 40;
    g_scanPattern[1].homeChannelTime = 0;
    g_scanPattern[1].ssidList = nullptr;
    g_scanPattern[1].ssidCount = 0;
    g_scanPattern[1].bssid.SetDirect(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF);

    /* Scan pattern 2 */
    g_scanPattern[2].scanType = nn::wlan::ScanType_Passive;
    g_scanPattern[2].channelList[0] = nn::wlan::WirelessChannel_1ch;
    g_scanPattern[2].channelList[1] = nn::wlan::WirelessChannel_6ch;
    g_scanPattern[2].channelList[2] = nn::wlan::WirelessChannel_11ch;
    g_scanPattern[2].channelCount = 3;
    g_scanPattern[2].channelScanTime = 120;
    g_scanPattern[2].homeChannelTime = 0;
    g_scanPattern[2].ssidList = nullptr;
    g_scanPattern[2].ssidCount = 0;
    g_scanPattern[2].bssid.SetDirect(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF);

    /* Scan pattern 3 */
    g_scanPattern[3].scanType = nn::wlan::ScanType_Active;
    g_scanPattern[3].channelList[0] = nn::wlan::WirelessChannel_1ch;
    g_scanPattern[3].channelList[1] = nn::wlan::WirelessChannel_6ch;
    g_scanPattern[3].channelList[2] = nn::wlan::WirelessChannel_11ch;
    g_scanPattern[3].channelCount = 3;
    g_scanPattern[3].channelScanTime = 40;
    g_scanPattern[3].homeChannelTime = 0;
    g_scanPattern[3].ssidList = nullptr;
    g_scanPattern[3].ssidCount = 0;
    g_scanPattern[3].bssid.SetDirect(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF);

    /* Scan pattern 4 */
    g_scanPattern[4].scanType = nn::wlan::ScanType_Passive;
    g_scanPattern[4].channelList[0] = nn::wlan::WirelessChannel_1ch;
    g_scanPattern[4].channelList[1] = nn::wlan::WirelessChannel_9ch;
    g_scanPattern[4].channelList[2] = nn::wlan::WirelessChannel_36ch;
    g_scanPattern[4].channelList[3] = nn::wlan::WirelessChannel_48ch;
    g_scanPattern[4].channelList[4] = nn::wlan::WirelessChannel_64ch;
    g_scanPattern[4].channelCount = 5;
    g_scanPattern[4].channelScanTime = 40;
    g_scanPattern[4].homeChannelTime = 0;
    g_scanPattern[4].ssidList = nullptr;
    g_scanPattern[4].ssidCount = 0;
    g_scanPattern[4].bssid.SetDirect(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF);
}

nn::Result InfraScan(int pattern)
{
    nn::Result result;

    WlanTest::ElapsedTime(g_TimeStr, g_StartTick, g_TickFreq);
    NN_LOG("[%s]Do scan with pattern%d\n", g_TimeStr, pattern);
    result = nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, g_scanPattern[pattern]);
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
    WlanTest::ElapsedTime(g_TimeStr, g_StartTick, g_TickFreq);
    NN_LOG("[%s]Found %d APs\n", g_TimeStr, resultReader.GetCount());

    return result;
}

void InitializeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
}

void FinalizeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start InfraScanRepeat Test\n\n");

    WlanTest::SystemInitialize();
    InitializeWlanInfra();
    InitializeScanPatterns();

    g_TickFreq = nn::os::GetSystemTickFrequency();
    g_StartTick = nn::os::GetSystemTick();

    nn::Result result;
    int64_t AttemptCount = 0;
    while( 1 )
    {
        for( int i = 0; i < ScanPatternCount; i++ )
        {
            AttemptCount++;
            NN_LOG("Scan times[%lld]\n", AttemptCount);
            result = InfraScan(i);
            if( result.IsFailure() )
            {
                NN_LOG("[ERROR]StartScan failed!!!\n");
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

    FinalizeWlanInfra();
    WlanTest::SystemFinalize();

    NN_LOG("\n\n End InfraScanRepeat Test \n\n");
} //NOLINT(impl/function_size)

