﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <vector>
#include <nn/util/util_ResDic.h>
#include <nn/utilTool/utilTool_BinarizerUtil.h>
#include <nn/utilTool/utilTool_BinarizerContext.h>

namespace nn { namespace utilTool {

struct SampleDicNode
{
    util::string_view key;
};

struct SampleFileHeader
{
    util::BinaryFileHeader header;
    char dummyData[8];
};

TEST( BinarizerUtilTest, BinarizerUtil )
{
    typedef std::vector< SampleDicNode* > NodeVector;

    const int monthCount = 12;
    const char* months[] = {
        "January", "February", "March", "April", "May", "June",
        "July", "August", "September", "October", "November", "December"
    };

    // 下準備
    BinarizerContext context;
    context.Initialize( 1 );

    NodeVector nodes;
    for (int i = 0; i < monthCount; ++i)
    {
        SampleDicNode* pNode = new SampleDicNode();
        pNode->key = months[ i ];
        nodes.push_back( pNode );
        context.InsertString( months[ i ] );
    }

    util::MemorySplitter::MemoryBlock headerBlock;
    headerBlock.SetSizeBy< SampleFileHeader >();
    context.AddMemoryBlock( 0, &headerBlock );

    context.AddMemoryBlock( 0, context.BuildStringPool() );

    util::MemorySplitter::MemoryBlock resDicBlock;
    resDicBlock.SetSize( util::ResDic::CalculateSize( monthCount ) );
    context.AddMemoryBlock( 0, &resDicBlock );

    context.AllocateBase();
    context.ConvertStringPool();

    // ResDic を構築
    bool buildResult = BinarizerUtil::BuildResDic(
        &context,
        &resDicBlock,
        nodes.begin(),
        nodes.end(),
        []( const SampleDicNode& node ){ return node.key; } );

    // 検証
    {
        EXPECT_TRUE( buildResult );
        util::ResDic* pDic = resDicBlock.Get< util::ResDic >( context.GetBasePtr() );

        // ResDic のアクセスはオフセット→ポインタ変換後なので Relocate しておく
        for (int i = 0; i <= monthCount; ++i)
        {
            pDic->ToData().entries[i].pKey.Relocate( context.GetBasePtr() );
        }

        EXPECT_EQ( util::ResDic::Npos, pDic->FindIndex( "Month" ) );
        EXPECT_EQ( 0, pDic->FindIndex( "January" ) );
        EXPECT_EQ( 1, pDic->FindIndex( "February" ) );
        EXPECT_EQ( 2, pDic->FindIndex( "March" ) );
        EXPECT_EQ( 3, pDic->FindIndex( "April" ) );
        EXPECT_EQ( 4, pDic->FindIndex( "May" ) );
        EXPECT_EQ( 5, pDic->FindIndex( "June" ) );
        EXPECT_EQ( 6, pDic->FindIndex( "July" ) );
        EXPECT_EQ( 7, pDic->FindIndex( "August" ) );
        EXPECT_EQ( 8, pDic->FindIndex( "September" ) );
        EXPECT_EQ( 9, pDic->FindIndex( "October" ) );
        EXPECT_EQ( 10, pDic->FindIndex( "November" ) );
        EXPECT_EQ( 11, pDic->FindIndex( "December" ) );
    }

    for (NodeVector::iterator iter = nodes.begin(), end = nodes.end(); iter != end; ++iter)
    {
        delete( *iter );
    }
}

}} // nn::utilTool
