﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <malloc.h>
#include <nnt/nntest.h>
#include <nn/util/util_BinaryFormat.h>
#include <nn/util/util_ResDic.h>
#include <nn/utilTool/utilTool_BinarizerContext2.h>

namespace {

const int SampleDataBlockDataLength = 32;
const int PtrArrayLength = 300;

// サンプルのファイルヘッダです
struct SampleFileHeader
{
    // ファイル全体のヘッダです。
    nn::util::BinaryFileHeader header;

    // 任意のデータです。
    char dummyData[8];
};

// サンプルのデータブロックです。
struct SampleDataBlock
{
    // ブロックごとのヘッダです。
    nn::util::BinaryBlockHeader header;

    nn::util::BinPtrToString pSampleString;         // セクション0, 文字列プール内の文字列を指す
    nn::util::BinTPtr<char> pFileHeaderDummyData;   // セクション0, SampleFileHeader の dummyData を指す
    nn::util::BinTPtr<int64_t> pDataBlockDummyData; // セクション1, 直後の dummyData を指す

    // 任意のデータです。
    int64_t dummyData[SampleDataBlockDataLength];

    // 256個 を超える BinPtr の配列です。
    nn::util::BinPtr binPtrArray[PtrArrayLength];
};

TEST( BinarizerContext2Test, BinarizerContext2 )
{
    // データ出力先のポインタです。
    void* pDst;

    // コンテキストを作成します。
    nn::utilTool::BinarizerContext2 context;

    // バイナリファイルを構築します。
    {
        // ---------------------------------------------------------------
        // コンテキストを初期化・設定します
        // ---------------------------------------------------------------

        // コンテキストを初期化します（仮にセクション数は2とします）
        context.Initialize( 2 );

        // バイナリファイル名を設定します。
        // （内部で文字列プールへの登録を行います）
        context.SetName( "SampleFile" );

        // ---------------------------------------------------------------
        // 使用するメモリ領域を計算するため、メモリブロックを登録します。
        // ---------------------------------------------------------------

        // セクション 0 にファイルヘッダのメモリブロックを登録します。

        // メモリブロックを作成します。
        nn::util::MemorySplitter::MemoryBlock headerMemoryBlock;

        // メモリブロックにファイルヘッダに合わせたサイズを設定します。
        headerMemoryBlock.SetSizeBy<SampleFileHeader>();

        // セクション 0 に登録します。
        context.AddMemoryBlock( 0, &headerMemoryBlock);

        // セクション 1 に登録するデータ用メモリブロックを作成します。
        nn::util::MemorySplitter::MemoryBlock dataMemoryBlock[2];

        // 配置されるべきアライメントを設定します。
        dataMemoryBlock[0].SetAlignment( 64 );

        // ブロックヘッダにデータを登録します。
        for (int blockIndex = 0; blockIndex < 2; blockIndex++)
        {
            nn::util::MemorySplitter::MemoryBlock* pMemoryBlock = &dataMemoryBlock[blockIndex];

            // サイズを設定します。
            pMemoryBlock->SetSizeBy<SampleDataBlock>();

            // セクション 1 に登録します。
            context.AddMemoryBlock(1, pMemoryBlock);

            // ポインタの位置と参照先の位置を書き込みます。
            context.RegisterBinPtr(
                pMemoryBlock,
                static_cast<ptrdiff_t>(offsetof(SampleDataBlock, pFileHeaderDummyData)),
                &headerMemoryBlock,
                static_cast<ptrdiff_t>(offsetof(SampleFileHeader, dummyData)));

            context.RegisterBinPtr(
                pMemoryBlock,
                static_cast<ptrdiff_t>(offsetof(SampleDataBlock, pDataBlockDummyData)),
                pMemoryBlock,
                static_cast<ptrdiff_t>(offsetof(SampleDataBlock, dummyData)));

            context.RegisterBinPtrToString(
                pMemoryBlock,
                static_cast<ptrdiff_t>(offsetof(SampleDataBlock, pSampleString)),
                "SampleString");

            for (int i = 0; i < PtrArrayLength; i++)
            {
                context.RegisterBinPtr(
                    pMemoryBlock,
                    static_cast<ptrdiff_t>(offsetof(SampleDataBlock, binPtrArray) + (i * sizeof(nn::util::BinPtr))),
                    pMemoryBlock);
            }
        }

        // 全ての文字列の登録が終わった後で文字列プールブロックを構築する必要があります。
        nn::util::MemorySplitter::MemoryBlock* stringBlock = context.CalculateStringPool();
        context.AddMemoryBlock(0, stringBlock);

        // ---------------------------------------------------------------
        // バッファを確保し、設定します。
        // ---------------------------------------------------------------

        // 登録したメモリブロックからサイズを計算して全体のバッファを確保します。
        context.Calculate();
        pDst = _aligned_malloc(context.GetSize(), context.GetAlignment());
        context.SetPtr(pDst, context.GetSize());

        // ---------------------------------------------------------------
        // 各種コンバート処理を実行します。
        // ---------------------------------------------------------------

        // 確保したバッファにデータを設定します
        {
            // ファイルヘッダにデータを登録します。
            // 領域の計算に使用したメモリブロックを用いて、ブロックが登録された位置を取得します。
            SampleFileHeader* pFileHeader = headerMemoryBlock.Get<SampleFileHeader>( context.GetPtr() );
            for (char i = 0; i < 8; i++)
            {
                pFileHeader->dummyData[i] = i;
            }

            // ブロックヘッダにデータを登録します。
            for (int blockIndex = 0; blockIndex < 2; blockIndex++)
            {
                SampleDataBlock* pDataBlock = dataMemoryBlock[blockIndex].Get<SampleDataBlock>( context.GetPtr() );
                for (int i = 0; i < SampleDataBlockDataLength; i++)
                {
                    pDataBlock->dummyData[i] = blockIndex * SampleDataBlockDataLength + i;
                }
                // データブロックにシグネチャを設定します。
                pDataBlock->header.signature.SetPacked( NN_UTIL_CREATE_SIGNATURE_4( 'D', 'B', 'L', 'K') );

                // コンテクストにブロックヘッダを登録します。
                context.AddHeader( &pDataBlock->header );
            }
        }

        // 登録済みの各データを変換します。
        nn::util::BinaryFileHeader* pBinaryFileHeader = context.Convert();

        // ---------------------------------------------------------------
        // ファイルヘッダのうち、自動で設定されない項目を設定します。
        // ---------------------------------------------------------------

        // ファイルヘッダのシグネチャとバージョンは各自で設定します。
        pBinaryFileHeader->signature.SetPacked( NN_UTIL_CREATE_SIGNATURE_8( 'b', 'i', 'n', 's', 'm', 'p', 'l', 0 ) );
        pBinaryFileHeader->version.major = 1;
        pBinaryFileHeader->version.minor = 0;
        pBinaryFileHeader->version.micro = 0;

        // バイトオーダーマークは任意で設定します。（初期値は 0xFEFF です）
        pBinaryFileHeader->SetByteOrderMark( nn::util::ByteOrderMark_Normal );
    }

    // バイナリファイルを検証します。
    {
        SampleFileHeader* pFile = static_cast< SampleFileHeader* >( pDst );
        nn::util::BinaryFileHeader& fileHeader = pFile->header;

        // SampleFileHeader の検証
        {
            int64_t signature = NN_UTIL_CREATE_SIGNATURE_8( 'b', 'i', 'n', 's', 'm', 'p', 'l', 0 );

            EXPECT_EQ( signature, fileHeader.signature.GetPacked() );
            EXPECT_TRUE( fileHeader.signature.IsValid( signature ) );
            EXPECT_TRUE( fileHeader.IsSignatureValid( signature ) );

            EXPECT_EQ( 1, fileHeader.version.major );
            EXPECT_EQ( 0, fileHeader.version.minor );
            EXPECT_EQ( 0, fileHeader.version.micro );
            EXPECT_TRUE( fileHeader.version.IsValid( 1, 0, 0 ) );
            EXPECT_TRUE( fileHeader.IsVersionValid( 1, 0, 0 ) );

            EXPECT_TRUE( fileHeader.IsAlignmentValid() );
            EXPECT_TRUE( fileHeader.IsEndianValid() );

            EXPECT_TRUE( fileHeader.IsValid( signature, 1, 0, 0 ) );

            EXPECT_EQ( "SampleFile", fileHeader.GetFileName() );
            EXPECT_EQ( context.GetSize(), fileHeader.GetFileSize() );

            // リロケーション
            EXPECT_FALSE( fileHeader.IsRelocated() );

            fileHeader.GetRelocationTable()->Relocate();

            EXPECT_TRUE( fileHeader.IsRelocated() );

            for (int i = 0, end = sizeof( pFile->dummyData ); i < end; i++)
            {
                EXPECT_EQ( i, pFile->dummyData[ i ] );
            }
        }

        // RelocationTable の検証
        {
            nn::util::RelocationTable& relTable = *fileHeader.GetRelocationTable();
            EXPECT_EQ( relTable._signature.GetPacked(), nn::util::RelocationTable::PackedSignature );

            EXPECT_EQ( fileHeader._offsetToRelTable, relTable._position );
            EXPECT_EQ( 2, relTable._sectionCount );

            nn::util::RelocationTable::Entry* pEntries = reinterpret_cast< nn::util::RelocationTable::Entry* >( &relTable._sections[relTable._sectionCount] );

            nn::util::RelocationTable::Section& first = *relTable.GetSection( 0 );

            // セクション0を指す BinTPtr は計 4 個あり、圧縮されて 2 個になる
            EXPECT_EQ( 2, first._entryCount );
            EXPECT_EQ( 0, first._entryIndex );

            int firstPtrCount = 0;
            for (int i = first._entryIndex; i < first._entryIndex + first._entryCount; i++)
            {
                nn::util::RelocationTable::Entry& entry = pEntries[i];
                firstPtrCount += entry._offsetCount * entry._structCount;
            }
            EXPECT_EQ( 4, firstPtrCount );

            nn::util::RelocationTable::Section& second = *relTable.GetSection( 1 );

            // セクション1を指す BinTPtr は計 2 + PtrArrayLength * 2 個あり、圧縮されて 2 個になる。
            EXPECT_EQ( 5, second._entryCount );
            EXPECT_EQ( 2, second._entryIndex );

            int secondPtrCount = 0;
            for (int i = second._entryIndex; i < second._entryIndex + second._entryCount; i++)
            {
                nn::util::RelocationTable::Entry& entry = pEntries[i];
                secondPtrCount += entry._offsetCount * entry._structCount;
            }
            EXPECT_EQ( 2 + PtrArrayLength * 2, secondPtrCount );
        }

        // SampleDataBlock の検証
        {
            SampleDataBlock& firstDataBlock = *reinterpret_cast< SampleDataBlock* >( pFile->header.FindFirstBlock( NN_UTIL_CREATE_SIGNATURE_4( 'D', 'B', 'L', 'K') ) );
            EXPECT_FALSE( NULL == firstDataBlock.header.GetNextBlock() );

            SampleDataBlock& secondDataBlock = *reinterpret_cast< SampleDataBlock* >( firstDataBlock.header.GetNextBlock() );
            EXPECT_TRUE( NULL == secondDataBlock.header.GetNextBlock() );

            for (int blockIndex = 0; blockIndex < 2; blockIndex++)
            {
                SampleDataBlock& dataBlock =  blockIndex == 0 ? firstDataBlock : secondDataBlock;

                EXPECT_EQ( sizeof( SampleDataBlock ), dataBlock.header.GetBlockSize() );
                for (int i = 0; i < SampleDataBlockDataLength; i++)
                {
                    EXPECT_EQ( blockIndex * SampleDataBlockDataLength + i, dataBlock.dummyData[ i ] );
                }

                // リロケート済みのポインタをチェック
                EXPECT_EQ( "SampleString", dataBlock.pSampleString.Get()->Get() );
                EXPECT_EQ( pFile->dummyData, dataBlock.pFileHeaderDummyData.Get() );
                EXPECT_EQ( dataBlock.dummyData, dataBlock.pDataBlockDummyData.Get() );

                for (int i = 0; i < PtrArrayLength; i++)
                {
                    EXPECT_EQ( &dataBlock, dataBlock.binPtrArray[i].Get() );
                }

                // 再度 Unrelocate → Relocate をチェック
                fileHeader.GetRelocationTable()->Unrelocate();

                EXPECT_FALSE( fileHeader.IsRelocated() );

                // リロケート前のオフセットをチェック
                // pSampleString は場所がわからないので、リロケート後に文字列が正しければよしとする
                EXPECT_EQ( nn::util::BytePtr( pFile ).Distance( pFile->dummyData ), dataBlock.pFileHeaderDummyData.GetOffset() );
                EXPECT_EQ( nn::util::BytePtr( pFile ).Distance( dataBlock.dummyData ), dataBlock.pDataBlockDummyData.GetOffset() );

                fileHeader.GetRelocationTable()->Relocate();
                EXPECT_TRUE( fileHeader.IsRelocated() );

                // リロケート済みのポインタをチェック
                EXPECT_EQ( "SampleString", dataBlock.pSampleString.Get()->Get() );
                EXPECT_EQ( pFile->dummyData, dataBlock.pFileHeaderDummyData.Get() );
                EXPECT_EQ( dataBlock.dummyData, dataBlock.pDataBlockDummyData.Get() );

                for (int i = 0; i < PtrArrayLength; i++)
                {
                    EXPECT_EQ( &dataBlock, dataBlock.binPtrArray[i].Get() );
                }
            }
        }
    }
} // NOLINT(impl/function_size)

// BinarizerContext2 による ResDic の構築をテストします。
TEST(BuildResDicTest, BinarizerContext2)
{
    const int monthCount = 12;
    const nn::util::string_view months[] = {
        "January", "February", "March", "April", "May", "June",
        "July", "August", "September", "October", "November", "December"
    };

    // データ出力先のポインタです。
    void* pDst;

    nn::utilTool::BinarizerContext2 context;
    context.Initialize(1);
    context.SetName("SampleFile");

    nn::util::MemorySplitter::MemoryBlock headerMemoryBlock;
    headerMemoryBlock.SetSizeBy<SampleFileHeader>();
    context.AddMemoryBlock(0, &headerMemoryBlock);

    nn::util::MemorySplitter::MemoryBlock resDicMemoryBlock;
    resDicMemoryBlock.SetSize(nn::util::ResDic::CalculateSize(monthCount));
    context.AddResDicMemoryBlock(0, &resDicMemoryBlock, months, monthCount);

    context.AddMemoryBlock(0, context.CalculateStringPool());

    context.Calculate();
    pDst = _aligned_malloc(context.GetSize(), context.GetAlignment());
    context.SetPtr(pDst, context.GetSize());

    nn::util::BinaryFileHeader* pHeader = context.Convert();
    pHeader->GetRelocationTable()->Relocate();

    // ResDic の検証
    {
        const nn::util::ResDic* pResDic = resDicMemoryBlock.Get<nn::util::ResDic>(pDst);

        EXPECT_EQ(monthCount, pResDic->GetCount());
        EXPECT_EQ(nn::util::ResDic::Npos, pResDic->FindIndex("Month"));

        for (int i = 0; i < monthCount; i++)
        {
            EXPECT_EQ(i, pResDic->FindIndex(months[i]));
        }
    }
}

} // anonymous namespace
