﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Execution.h>

#include <cstddef>
#include <cstdlib>
#include <cstring>
#include <functional>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#define NNT_UTIL_ENABLE_EXECUTOR_BASE
#define NNT_UTIL_ENABLE_EXECUTOR_PARALLEL_1_1
#define NNT_UTIL_ENABLE_EXECUTOR_PARALLEL_10_15
#define NNT_UTIL_ENABLE_EXECUTOR_USECASE

#define NNT_UTIL_ASSERT_RESULT_SUCCESS(expression) \
    do \
    { \
        auto _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("\"" #expression "\" failed with %03d-%04d(0x%08x)\n", _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ASSERT_TRUE(_r.IsSuccess()); \
        } \
    } \
    while (NN_STATIC_CONDITION(false))

#define NNT_UTIL_EXPECT_RESULT_SUCCESS(expression) \
    do \
    { \
        auto _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("\"" #expression "\" failed with %03d-%04d(0x%08x)\n", _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            EXPECT_TRUE(_r.IsSuccess()); \
        } \
    } \
    while (NN_STATIC_CONDITION(false))

#define NNT_UTIL_ASSERT_RESULT_INCLUDED(expect, expression) \
    do \
    { \
        auto _r = (expression); \
        if (!expect::Includes(_r)) \
        { \
            NN_LOG("\"" #expression "\" returned %03d-%04d(0x%08x), which not included in " #expect "\n", _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ASSERT_TRUE(_r.IsSuccess()); \
        } \
    } \
    while (NN_STATIC_CONDITION(false))

#define NNT_UTIL_EXPECT_RESULT_INCLUDED(expect, expression) \
    do \
    { \
        auto _r = (expression); \
        if (!expect::Includes(_r)) \
        { \
            NN_LOG("\"" #expression "\" returned %03d-%04d(0x%08x), which not included in " #expect "\n", _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            EXPECT_TRUE(_r.IsSuccess()); \
        } \
    } \
    while (NN_STATIC_CONDITION(false))

namespace
{
NN_DEFINE_ERROR_RANGE_RESULT(ResultCanceled, 511, 1, 2);
NN_DEFINE_ERROR_RANGE_RESULT(ResultOutOfQueueCapacity, 511, 2, 3);
template <int Capacity>
struct Config
{
    static const int QueueCapacity = Capacity;
    class ResultOutOfQueueCapacity
        : public ::ResultOutOfQueueCapacity
    {
    };
};

class Thread
{
    NN_DISALLOW_COPY(Thread);
    NN_DISALLOW_MOVE(Thread);

private:
    bool m_Initialized {false};
    nn::os::ThreadType m_Thread;

    std::function<void(void)> m_Function;

    static void ThreadFunction(void* p)
    {
        reinterpret_cast<Thread*>(p)->Run();
    }

    void Run() NN_NOEXCEPT
    {
        m_Function();
    }
    void Finalize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Initialized);
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
        m_Initialized = false;
    }

public:
    Thread() NN_NOEXCEPT = default;
    ~Thread() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            Finalize();
        }
    }
    void Initialize(std::function<void(void)> f, void* stack, size_t stackSize, int priority, const char* name) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_Initialized);
        m_Function = f;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunction, this, stack, stackSize, priority));
        nn::os::SetThreadNamePointer(&m_Thread, name);
        m_Initialized = true;
    }
    void Start() NN_NOEXCEPT
    {
        nn::os::StartThread(&m_Thread);
    }
};

typedef char StackType[4096 * 2];
NN_ALIGNAS(4096) StackType g_Stacks[5];

class EmptyTask
    : public nn::util::Executable
{
public:
    virtual nn::Result ExecuteImpl(void*, size_t) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
};

template <int TimeoutMilliSeconds, int WaitInterval>
class EmptyTaskWithWait
    : public nn::util::Executable
{
private:
    nn::os::Event m_Event;
public:
    EmptyTaskWithWait() NN_NOEXCEPT
        : m_Event(nn::os::EventClearMode_ManualClear)
    {
    }
    virtual nn::Result ExecuteImpl(void*, size_t) NN_NOEXCEPT final NN_OVERRIDE
    {
        m_Event.Signal();
        int ct = 0;
        while (true
            && !IsCanceled()
            && ct ++ < TimeoutMilliSeconds / WaitInterval)
        {
            // this->Cancel() が呼ばれるか、タイムアウトするまで待機
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitInterval));
        }
        if (IsCanceled())
        {
            NN_RESULT_THROW(ResultCanceled());
        }
        NN_RESULT_SUCCESS;
    }
    void Wait() NN_NOEXCEPT
    {
        m_Event.Wait();
    }
};

template <int TimeoutMilliSeconds, int WaitInterval>
class DeathTask
    : public nn::util::Executable
{
private:
    nn::os::Event m_Event;
    bool m_IsCanceledInExecuteImpl {false};
public:
    DeathTask() NN_NOEXCEPT
        : m_Event(nn::os::EventClearMode_ManualClear)
    {
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_IsCanceledInExecuteImpl;
    }
    virtual nn::Result ExecuteImpl(void*, size_t) NN_NOEXCEPT final NN_OVERRIDE
    {
        m_Event.Signal();
        int ct = 0;
        do
        {
            // this->Cancel() が呼ばれるまで待機
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitInterval));
            NN_ABORT_UNLESS(++ ct < TimeoutMilliSeconds / WaitInterval);
        } while (!IsCanceled());
        m_IsCanceledInExecuteImpl = true;
        return ResultCanceled();
    }
    void Wait() NN_NOEXCEPT
    {
        m_Event.Wait();
    }
};

template <int TimeoutMilliSeconds, int WaitInterval>
class DeathTask2
    : public nn::util::Executable
{
private:
    nn::os::Event m_Event;
    bool m_IsCanceledInExecuteImpl {false};
    class Subtask
        : public nn::util::CancelInjectable
    {
    public:
        explicit Subtask(const nn::util::Cancelable* pCancelable = nullptr) NN_NOEXCEPT
            : CancelInjectable(pCancelable)
        {
        }
        void DoSomething() NN_NOEXCEPT
        {
            int ct = 0;
            do
            {
                // this->Cancel() が呼ばれるまで待機
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitInterval));
                NN_ABORT_UNLESS(++ ct < TimeoutMilliSeconds / WaitInterval);
            } while (!IsCanceled());
        }
    };
public:
    DeathTask2() NN_NOEXCEPT
        : m_Event(nn::os::EventClearMode_ManualClear)
    {
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_IsCanceledInExecuteImpl;
    }
    virtual nn::Result ExecuteImpl(void*, size_t) NN_NOEXCEPT final NN_OVERRIDE
    {
        m_Event.Signal();
        Subtask sub(this);     // Cancelable の注入
        sub.DoSomething();    // this->Cancel() が呼ばれるまで待機
        m_IsCanceledInExecuteImpl = true;
        return ResultCanceled();
    }
    void Wait() NN_NOEXCEPT
    {
        m_Event.Wait();
    }
};

template <typename ExecutorType>
void DefaultExecutionThreadFunction(ExecutorType& executor, nn::os::Event& terminator) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto index = nn::os::WaitAny(terminator.GetBase(), executor);
        switch (index)
        {
        case 0:
            return;
        case 1:
            executor.ClearSignal();
            executor.Execute(nullptr, 0);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

template <typename TaskType>
void RunTaskExecutionTest() NN_NOEXCEPT
{
    {
        TaskType task;
        NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());
        nn::os::SystemEvent e;
        e.AttachReadableHandle(task.GetReadableHandle(), false, nn::os::EventClearMode_ManualClear);

        // 初期状態
        ASSERT_FALSE(e.TryWait());
        ASSERT_FALSE(task.HasDone());

        // タスクの実行
        ASSERT_TRUE(task.TryLock());
        task.Execute(nullptr, 0);
        task.Unlock();
        ASSERT_TRUE(e.TryWait());
        ASSERT_TRUE(task.HasDone());
        NNT_UTIL_ASSERT_RESULT_SUCCESS(*task.TryGetResult());

        // シグナル解除
        e.Clear();
        ASSERT_FALSE(e.TryWait());
        ASSERT_TRUE(task.HasDone());
    }

    {
        TaskType task;
        NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());
        nn::os::SystemEvent e;
        e.AttachReadableHandle(task.GetReadableHandle(), false, nn::os::EventClearMode_ManualClear);

        // 初期状態
        ASSERT_FALSE(e.TryWait());
        ASSERT_FALSE(task.HasDone());

        // キャンセル
        task.Cancel();
        ASSERT_TRUE(task.IsCanceled());
        ASSERT_FALSE(e.TryWait());
        ASSERT_FALSE(task.HasDone());

        // キャンセル後の実行
        ASSERT_TRUE(task.TryLock());
        task.Execute(nullptr, 0);
        task.Unlock();
        ASSERT_TRUE(e.TryWait());
        ASSERT_TRUE(task.HasDone());
        ASSERT_FALSE(task.TryGetResult());

        // シグナル解除
        e.Clear();
        ASSERT_FALSE(e.TryWait());
        ASSERT_TRUE(task.HasDone());
    }
}

template <typename TaskType>
void RunSerialExecutionTest() NN_NOEXCEPT
{
    TaskType task;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());
    TaskType task2;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task2.Initialize());
    TaskType task3;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task3.Initialize());

    {
        // Executor 初期状態
        nn::util::Executor<Config<2>> executor;
        ASSERT_FALSE(executor.TryWaitSignal());
        ASSERT_EQ(-1, nn::os::TryWaitAny(executor));

        // Executor タスク登録
        auto r = executor.Register(&task);
        NNT_UTIL_ASSERT_RESULT_SUCCESS(r);
        ASSERT_FALSE(task.HasDone());
        ASSERT_TRUE(executor.TryWaitSignal());
        ASSERT_TRUE(executor.TryWaitSignal()); // ManualClear 性
        ASSERT_EQ(0, nn::os::TryWaitAny(executor));
        executor.WaitSignal();

        // Executor イベントクリア
        executor.ClearSignal();
        ASSERT_FALSE(executor.TryWaitSignal());

        // Executor 複数タスク登録
        r = executor.Register(&task2);
        NNT_UTIL_ASSERT_RESULT_SUCCESS(r);
        executor.Unregister(&task);
        r = executor.Register(&task);
        NNT_UTIL_ASSERT_RESULT_SUCCESS(r); // 一度解除できる
        ASSERT_FALSE(task2.HasDone());
        ASSERT_TRUE(executor.TryWaitSignal());
        executor.ClearSignal();

        // Executor タスク登録上限
        r = executor.Register(&task3);
        NNT_UTIL_ASSERT_RESULT_INCLUDED(ResultOutOfQueueCapacity, r);
        ASSERT_FALSE(task3.HasDone());
        ASSERT_FALSE(executor.TryWaitSignal());

        // Executor 実行
        executor.Execute(nullptr, 0);
        ASSERT_FALSE(executor.TryWaitSignal());
        ASSERT_EQ(-1, nn::os::TryWaitAny(executor));

        // タスクの事後状態 (全タスク完了)
        ASSERT_TRUE(task.HasDone());
        NNT_UTIL_EXPECT_RESULT_SUCCESS(*task.TryGetResult());
        ASSERT_TRUE(task2.HasDone());
        NNT_UTIL_EXPECT_RESULT_SUCCESS(*task2.TryGetResult());

        // 実行後に解除されている
        r = executor.Register(&task3);
        NNT_UTIL_ASSERT_RESULT_SUCCESS(r);
        ASSERT_FALSE(task3.HasDone());
        ASSERT_TRUE(executor.TryWaitSignal());
        executor.ClearSignal();

        // 暗黙的に解除されてても解除できる
        executor.Unregister(&task);
        executor.Unregister(&task2);

        // 明示的な解除
        executor.Unregister(&task3);
    }

    // 未実行のタスクは実行されておらず、実行できる
    ASSERT_FALSE(task3.HasDone());
    ASSERT_TRUE(task3.TryLock());
    task3.Execute(nullptr, 0);
    task3.Unlock();
    NNT_UTIL_ASSERT_RESULT_SUCCESS(*task3.TryGetResult());
}

template <typename TaskType>
void RunNestedCancelTest() NN_NOEXCEPT
{
    nn::util::Executor<Config<2>> executor;
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    TaskType task;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());
    executor.Register(&task);

    // スレッドの作成、開始
    Thread thread;
    auto execute = [&]() -> void
    {
        DefaultExecutionThreadFunction<>(executor, terminator);
    };
    thread.Initialize(execute, &g_Stacks[0], sizeof(g_Stacks[0]), nn::os::DefaultThreadPriority, "testAccountExecutor:RunNestedCancelTest");
    thread.Start();

    // キャンセル
    task.Wait(); // タスクが実行されるまで待つ
    task.Cancel();
    EXPECT_TRUE(task.IsCanceled());
    {
        auto begin = nn::os::GetSystemTick();
        executor.Unregister(&task);
        auto end = nn::os::GetSystemTick();
        auto duration = (end - begin).ToTimeSpan();
        NN_LOG("duration.GetMilliSeconds() = %lld\n", duration.GetMilliSeconds());
    }
    ASSERT_TRUE(task.HasDone());
    NNT_UTIL_EXPECT_RESULT_INCLUDED(ResultCanceled, *task.TryGetResult());

    // タスク内部でキャンセルされたかの検査
    EXPECT_TRUE(task);

    // スレッドの終了待ちなど。
    terminator.Signal();
}

template <typename TaskType>
void RunDoubleUnregistrationTest() NN_NOEXCEPT
{
    nn::util::Executor<Config<2>> executor;
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    TaskType task;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());
    executor.Register(&task);

    // スレッドの作成、開始
    Thread thread;
    auto execute = [&]() -> void
    {
        DefaultExecutionThreadFunction<>(executor, terminator);
    };
    thread.Initialize(execute, &g_Stacks[0], sizeof(g_Stacks[0]), nn::os::DefaultThreadPriority, "testAccountExecutor:RunDoubleUnregistrationTest");
    thread.Start();

    // タスクが完了するまで待つ
    task.Wait();
    {
        auto begin = nn::os::GetSystemTick();
        executor.Unregister(&task);
        auto end = nn::os::GetSystemTick();
        auto duration = (end - begin).ToTimeSpan();
        NN_LOG("duration.GetMilliSeconds() = %lld\n", duration.GetMilliSeconds());
    }
    ASSERT_TRUE(task.HasDone());
    NNT_UTIL_EXPECT_RESULT_SUCCESS(*task.TryGetResult());

    // スレッドの終了待ちなど。
    terminator.Signal();
}

template <int Capacity, int NumTask, int LoopCount, typename TaskType>
void RunParallelExecutionTest() NN_NOEXCEPT
{
    nn::util::Executor<Config<Capacity>> executor;
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    auto dispatch = [&]() -> void
    {
        TaskType* tasks = new TaskType[NumTask];
        NN_UTIL_SCOPE_EXIT { delete[] tasks; };

        for (int i = 0; i < NumTask; ++ i)
        {
            NNT_UTIL_ASSERT_RESULT_SUCCESS(tasks[i].Initialize());
        }

        int ct = 0;
        while (ct < NumTask)
        {
            auto r = executor.Register(&tasks[ct]);
            if (!r.IsSuccess())
            {
                NNT_UTIL_EXPECT_RESULT_INCLUDED(ResultOutOfQueueCapacity, r);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                continue;
            }
            ++ ct;
        }

        for (int i = 0; i < NumTask; ++ i)
        {
            nn::os::SystemEvent e;
            e.AttachReadableHandle(tasks[i].GetReadableHandle(), false, nn::os::EventClearMode_ManualClear);
            e.Wait();
            executor.Unregister(&tasks[i]);
        }
    };
    auto dispatchAndCancel = [&]() -> void
    {
        TaskType* tasks = new TaskType[NumTask];
        NN_UTIL_SCOPE_EXIT { delete[] tasks; };

        for (int i = 0; i < NumTask; ++ i)
        {
            NNT_UTIL_ASSERT_RESULT_SUCCESS(tasks[i].Initialize());
        }

        int ct = 0;
        while (ct < NumTask)
        {
            auto r = executor.Register(&tasks[ct]);
            if (!r.IsSuccess())
            {
                NNT_UTIL_EXPECT_RESULT_INCLUDED(ResultOutOfQueueCapacity, r);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                continue;
            }
            ++ ct;
        }

        for (int i = 0; i < NumTask; ++ i)
        {
            tasks[i].Cancel();
            nn::os::SystemEvent e;
            e.AttachReadableHandle(tasks[i].GetReadableHandle(), false, nn::os::EventClearMode_ManualClear);
            e.Wait();
            executor.Unregister(&tasks[i]);
        }
    };

    // Executor スレッドの開始
    Thread threadExecutor;
    NN_UTIL_SCOPE_EXIT {terminator.Signal();};
    threadExecutor.Initialize(
        [&]() -> void
        {
            DefaultExecutionThreadFunction<>(executor, terminator);
        },
        &g_Stacks[0], sizeof(g_Stacks[0]),
        nn::os::DefaultThreadPriority + 3,
        "testAccountExecutor:RunParallelExecutionTest:Executor");
    threadExecutor.Start();

    // Dispatcher x1
    for (auto s = 0; s < LoopCount; ++ s)
    {
        Thread threadDispatcher;
        threadDispatcher.Initialize(dispatch, &g_Stacks[1], sizeof(g_Stacks[1]), nn::os::DefaultThreadPriority + 1, "testAccountExecutor:RunParallelExecutionTest:Dispatcher");

        // 開始と終了
        threadDispatcher.Start();

    }

    // Dispatcher x3, Dispach&Cancel x1
    for (auto s = 0; s < LoopCount; ++ s)
    {
        Thread threadLow;
        threadLow.Initialize(dispatchAndCancel, &g_Stacks[1], sizeof(g_Stacks[1]), nn::os::DefaultThreadPriority + 2, "testAccountExecutor:RunParallelExecutionTest:Jammer");

        Thread threadDispatchers[3];
        for (int i = 0; i < sizeof(threadDispatchers) / sizeof(threadDispatchers[0]); ++i)
        {
            auto& t = threadDispatchers[i];
            t.Initialize(dispatch, &g_Stacks[2 + i], sizeof(g_Stacks[2 + i]), nn::os::DefaultThreadPriority + 1, "testAccountExecutor:RunParallelExecutionTest:Dispatcher");
        }

        for (auto& t : threadDispatchers)
        {
            t.Start();
        }
        threadLow.Start();
    }
} // NOLINT(readability/fn_size)

} // ~namespace <anonymous>

#if defined(NNT_UTIL_ENABLE_EXECUTOR_BASE)
TEST(UtilExecution, Death)
{
    {
        DeathTask<100, 10> t;
        NNT_UTIL_ASSERT_RESULT_SUCCESS(t.Initialize());
        ASSERT_DEATH_IF_SUPPORTED({
            t.Lock();
            t.Execute(nullptr, 0);
            t.Unlock();
        }, ".*");
    }

    {
        DeathTask2<100, 10> t;
        NNT_UTIL_ASSERT_RESULT_SUCCESS(t.Initialize());
        ASSERT_DEATH_IF_SUPPORTED({
            t.Lock();
            t.Execute(nullptr, 0);
            t.Unlock();
        }, ".*");
    }
}

TEST(UtilExecution, Base)
{
    RunTaskExecutionTest<EmptyTask>();     // タスク単体の状態のテスト
    RunSerialExecutionTest<EmptyTask>();   // Executor 単体の状態のテスト
    RunDoubleUnregistrationTest<EmptyTaskWithWait<100, 10>>(); // execute 中の Unregister のテスト
    RunNestedCancelTest<DeathTask<300, 100>>();  // タスク内部で Cancel 可能かのテスト
    RunNestedCancelTest<DeathTask2<300, 100>>(); // タスク内部の別の要素内で Cancel 可能かのテスト
    for (int i = 0; i < 3; ++ i)
    {
        RunDoubleUnregistrationTest<EmptyTaskWithWait<20, 1>>(); // execute 中の Unregister のテスト
        RunNestedCancelTest<DeathTask<20, 1>>();  // タスク内部で Cancel 可能かのテスト
        RunNestedCancelTest<DeathTask2<20, 1>>(); // タスク内部の別の要素内で Cancel 可能かのテスト
    }
}
#endif

#if defined(NNT_UTIL_ENABLE_EXECUTOR_PARALLEL_1_1)
TEST(UtilExecution, Parallel_1_1)
{
    RunParallelExecutionTest<1, 1, 3, EmptyTaskWithWait<60, 1>>();
}
#endif
#if defined(NNT_UTIL_ENABLE_EXECUTOR_PARALLEL_10_15)
TEST(UtilExecution, Parallel_10_15)
{
    RunParallelExecutionTest<10, 15, 3, EmptyTaskWithWait<60, 1>>();
}
#endif

/* ------------------------------------------------------------
    それっぽいユースケース
 */
#if defined(NNT_UTIL_ENABLE_EXECUTOR_USECASE)
namespace
{
// タスクが必要とする実行用のリソースの定義
struct IoBuffer
{
    void* pointer;
    size_t size;
};
// タスク中でさらに実行されるサブタスクの定義
// (処理内容に意味はない)
class SubTask
    : public nn::util::CancelInjectable
{
public:
    explicit SubTask(nn::util::Cancelable* pC = nullptr) NN_NOEXCEPT
        : CancelInjectable(pC)
    {
    }
    nn::Result DoSomething() const NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(int counter, = 0);
        for (int i = 0; i < 10; ++ i)
        {
            if (IsCanceled())
            {
                NN_LOG(">> SubTask Canceled: counter=%d\n", counter);
                NN_RESULT_THROW(ResultCanceled());
            }
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
        }
        ++ counter;
        NN_RESULT_SUCCESS;
    }
};
// 実行されるタスク
// (処理内容に意味はない)
class Task
    : public nn::util::Executable
{
private:
    SubTask m_SubTask;
    SubTask m_SubTask1;
    SubTask m_SubTask2;

protected:
    virtual nn::Result ExecuteImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_SDK_ASSERT(bufferSize == sizeof(IoBuffer));
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<IoBuffer>::value == 0);
        const IoBuffer& resource = *reinterpret_cast<IoBuffer*>(buffer);

        // 本当は通信とか IO とか、時間のかかる処理をする。
        NN_LOG("> Task start\n");
        std::memset(resource.pointer, 0x00, resource.size);

        NN_LOG("> SubTask start\n");
        for (int i = 0; i < 100; ++ i)
        {
            if (IsCanceled())
            {
                NN_LOG("> Task Canceled: counter=%d\n", i);
                NN_RESULT_THROW(ResultCanceled());
            }
            NN_RESULT_DO(m_SubTask.DoSomething());
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
        }
        NN_RESULT_SUCCESS;
    }
public:
    Task() NN_NOEXCEPT
        : m_SubTask(this)
        , m_SubTask1(this)
        , m_SubTask2() // SubTask2 はキャンセル不可能な処理
    {
    }
};
// タスクを実行するスレッド。メインの処理スレッドより優先度低く動作すべき。
void ExecutorThreadFunction(
    nn::util::AbstractExecutor& executor,
    IoBuffer& resource,
    nn::os::Event& terminator) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto index = nn::os::WaitAny(terminator.GetBase(), executor);
        switch (index)
        {
        case 0:
            terminator.Clear();
            return;
        case 1:
            executor.ClearSignal();
            executor.Execute(&resource, sizeof(resource));
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}
// タスクを実行してほしい人
// DoHogehogeAsync() の内部処理と、その呼び出し元を混ぜたようなイメージ
template <int TimeOutMilliSeconds>
void DispatcherThreadFunction(nn::util::AbstractExecutor& executor) NN_NOEXCEPT
{
    // 実行したいタスク (IPC なら、サーバーで保持)
    Task task;
    NNT_UTIL_ASSERT_RESULT_SUCCESS(task.Initialize());

    // タスクの完了検出用イベント (IPC なら、セッションに紐づけてクライアントに渡す)
    nn::os::SystemEvent taskEvent;
    taskEvent.AttachReadableHandle(task.GetReadableHandle(), false, nn::os::EventClearMode_ManualClear);

    // タイムアウト検出用のタイマー
    nn::os::TimerEvent timer(nn::os::EventClearMode_ManualClear);

    // タスクの実行
    NN_LOG("[Start task with timeout= %d [msec]]\n", TimeOutMilliSeconds);

    // タスクの登録 (DoHogehogeAsync() の内部実装)
    auto start = nn::os::GetSystemTick();
    executor.Register(&task);
    timer.StartOneShot(nn::TimeSpan::FromMilliSeconds(TimeOutMilliSeconds));

    // タスクの完了 or タイムアウト待ち
    auto index = nn::os::WaitAny(timer.GetBase(), taskEvent.GetBase());
    switch (index)
    {
    case 0:
        // タイムアウト
        NN_LOG("[Task timed out]\n");
        task.Cancel(); // タイムアウト時はキャンセルする。
        break;
    case 1:
        taskEvent.Clear();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // タスクの削除
    executor.Unregister(&task);
    auto end = nn::os::GetSystemTick();

    // タスクの実行結果
    NN_LOG("[Task Result]\n");
    NN_LOG(" - Duration: %lld [msec]\n", (end - start).ToTimeSpan().GetMilliSeconds());
    if (task.HasDone())
    {
        auto pR = task.TryGetResult();
        if (!pR)
        {
            NN_LOG(" - Task state: Canceled before execution\n");
        }
        else
        {
            if (pR->IsSuccess())
                NN_LOG(" - Task state: success\n");
            else if (ResultCanceled::Includes(*pR))
                NN_LOG(" - Task state: canceled during execution\n");
            else
                NN_LOG(" - Task state: error(%d, %d)\n", pR->GetModule(), pR->GetDescription());
        }
    }
    else
    {
        NN_LOG(" - Task state: not-started\n");
    }
}

template <int TimeOutMilliSeconds>
void RunUsecaseTest()
{
    // 実行時に使用するリソースの初期化
    IoBuffer resource = {std::malloc(1024), 1024};
    NN_UTIL_SCOPE_EXIT
    {
        std::free(resource.pointer);
    };

    // Executor と停止用のイベントの初期化
    nn::util::Executor<Config<1>> executor;
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    // Executor スレッドの開始
    Thread threadExecutor;
    threadExecutor.Initialize(
        [&]() -> void
        {
            ExecutorThreadFunction(executor, resource, terminator);
        },
        &g_Stacks[0], sizeof(g_Stacks[0]), nn::os::LowestThreadPriority, "testAccountExecutor:RunUsecaseTest");
    threadExecutor.Start();

    // Dispatcher の実行
    DispatcherThreadFunction<TimeOutMilliSeconds>(executor);

    // Executor スレッドの終了
    terminator.Signal();
}
} // ~namespace <anonymous>

TEST(UtilExecution, Usecase)
{
    RunUsecaseTest<20>();   // (だいたい) タイムアウト前に完了する
    RunUsecaseTest<3>();    // (だいたい) タイムアウトしてキャンセルされる
    RunUsecaseTest<0>();    // (だいたい) いきなりキャンセルされる
}
#endif
