﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <random>
#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_CharacterEncodingResult.h>
#include "SjisEncoding.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#include <nn/nn_Windows.h>
#endif

// Shift_JIS と UTF-16 を相互変換
TEST(EncSjisTest, Simple)
{
    // あいうえお (Shift_JIS)
    unsigned char pSjisSrc[11] = { 0x82u, 0xA0u,
        0x82u, 0xA2u,
        0x82u, 0xA4u,
        0x82u, 0xA6u,
        0x82u, 0xA8u,
        '\0' };

    // あいうえお (UTF-16)
    uint16_t pUtf16Src[6] = { 0x3042,
        0x3044,
        0x3046,
        0x3048,
        0x304A,
        '\0' };

    unsigned char pSjisDst[11];
    uint16_t pUtf16Dst[6];

    auto result = ConvertStringSjisToUtf16(pUtf16Dst, 6, reinterpret_cast<char*>(pSjisSrc));
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    for (int i = 0; i < 5; ++i)
    {
        ASSERT_EQ(pUtf16Src[i], pUtf16Dst[i]);
    }

    pUtf16Dst[5] = '\0';
    result = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 11, pUtf16Dst);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    for (int i = 0; i < 10; ++i)
    {
        ASSERT_EQ(pSjisSrc[i], pSjisDst[i]);
    }
}

// 配列の不足
TEST(EncSjisTest, InsufficientLength)
{
    // あいうえお (Shift_JIS)
    unsigned char pSjisSrc[11] = { 0x82u, 0xA0u,
                                   0x82u, 0xA2u,
                                   0x82u, 0xA4u,
                                   0x82u, 0xA6u,
                                   0x82u, 0xA8u,
                                   '\0' };

    // あいうえお (UTF-16)
    uint16_t pUtf16Src[6] = { 0x3042,
                              0x3044,
                              0x3046,
                              0x3048,
                              0x304A,
                              '\0' };

    unsigned char pSjisDst[10];
    uint16_t pUtf16Dst[5];

    // 配列の長さとして渡す数が 1 足りない
    auto result = ConvertStringSjisToUtf16(pUtf16Dst, 4, reinterpret_cast<char*>(pSjisSrc));
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);
    result = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 9, pUtf16Src);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InsufficientLength);

    // 配列の長さがぎりぎり足りる
    result = ConvertStringSjisToUtf16(pUtf16Dst, 5, reinterpret_cast<char*>(pSjisSrc));
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);
    result = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 10, pUtf16Src);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_Success);

}

// 不正文字
TEST(EncSjisTest, InvalidFormat)
{
    // 不正文字（ 1 文字）
    unsigned char pInvalidSjisChar[3] = { 0x81, 0xB2, '\0' };
    uint16_t pInvalidUtf16Char[2] = { 0xD801, '\0' };

    // 不正文字列
    unsigned char pInvalidSjisString[11] = { 0x82u, 0xA0u,
                                             0x82u, 0xA2u,
                                             0x82u, 0xA4u,
                                             0x82u, 0x5Cu,    // ここだけ不正
                                             0x82u, 0xA8u,
                                             '\0' };

    uint16_t pInvalidUtf16String[6] = { 0x3042,
                                        0x3044,
                                        0x3046,
                                        0xFBD7,               // ここだけ不正
                                        0x304A,
                                        '\0' };

    unsigned char pSjisDst[11];
    uint16_t pUtf16Dst[6];

    // Shift_JIS -> UTF-16
    auto result = ConvertStringSjisToUtf16(pUtf16Dst, 6, reinterpret_cast<char*>(pInvalidSjisChar));
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
    result = ConvertStringSjisToUtf16(pUtf16Dst, 6, reinterpret_cast<char*>(pInvalidSjisString));
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);

    // UTF-16 -> Shift_JIS
    result = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 11, pInvalidUtf16Char);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
    result = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pSjisDst), 11, pInvalidUtf16String);
    ASSERT_EQ(result, nn::util::CharacterEncodingResult_InvalidFormat);
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// MultiByteToWideChar と処理結果を比較（ 1 文字）
TEST(EncSjisTest, CompareCharWithMultiByteToWideChar)
{
    for (int i = 1; i <= 0xFF; ++i)
    {
        unsigned char pSrc[3];
        uint16_t encDst, winDst;

        pSrc[0] = static_cast<unsigned char>(i);
        pSrc[1] = '\0';
        auto encResult = ConvertStringSjisToUtf16(&encDst, 1, reinterpret_cast<char*>(pSrc));
        int winResult = MultiByteToWideChar(932, 0, reinterpret_cast<char*>(pSrc), 1, reinterpret_cast<wchar_t*>(&winDst), 1);
        if (encResult == nn::util::CharacterEncodingResult_Success)
        {
            ASSERT_NE(winResult, 0);    // MultiByteToWideChar も成功しているはず
            ASSERT_EQ(encDst, winDst);
        }
        else
        {
            // Windows の MultiByteToWideChar はうまく変換できない文字を「・」(0x30FB)に
            // 変換して成功したことにする場合がある
            // それ以外の失敗の場合のみ FAIL とする
            if (winDst != 0x30FB || (winDst == 0x30FB && pSrc[0] == 0x81 && pSrc[1] == 0x45))
            {
                ASSERT_EQ(winResult, 0);    // MultiByteToWideChar も失敗しているはず
            }
        }

        // 2 バイト文字の確認
        for (int j = 1; j <= 0xFF; ++j)
        {
            pSrc[0] = static_cast<unsigned char>(i);
            pSrc[1] = static_cast<unsigned char>(j);
            pSrc[2] = '\0';

            encResult = ConvertStringSjisToUtf16(&encDst, 1, reinterpret_cast<char*>(pSrc));
            winResult = MultiByteToWideChar(932, 0, reinterpret_cast<char*>(pSrc), 2, reinterpret_cast<wchar_t*>(&winDst), 1);
            if (encResult == nn::util::CharacterEncodingResult_Success)
            {
                ASSERT_NE(winResult, 0);
                ASSERT_EQ(encDst, winDst);
            }
            else
            {
                if (winDst != 0x30FB || (winDst == 0x30FB && pSrc[0] == 0x81 && pSrc[1] == 0x45))
                {
                    ASSERT_EQ(winResult, 0);    // MultiByteToWideChar も失敗しているはず
                }
            }
        }
    }
}

// WideCharToMultiByte と処理結果を比較（ 1 文字）
TEST(EncSjisTest, CompareCharWithWideCharToMultiByte)
{
    for (int i = 1; i <= 0xFFFF; ++i)
    {
        uint16_t pSrc[2];
        unsigned char pEncDst[2], pWinDst[2];
        char alter = 0x3F;   // WideCharToMultiByte でマップできない文字の既定値

        pSrc[0] = static_cast<uint16_t>(i);
        pSrc[1] = '\0';
        auto encResult = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pEncDst), 2, pSrc);
        int winResult = WideCharToMultiByte(932, 0, reinterpret_cast<wchar_t*>(pSrc), 1, reinterpret_cast<char*>(pWinDst), 2, &alter, NULL);
        if (encResult == nn::util::CharacterEncodingResult_Success)
        {
            ASSERT_NE(winResult, 0);    // MultiByteToWideChar も成功しているはず
            ASSERT_EQ(pEncDst[0], pWinDst[0]);
            if(winResult > 1)
            {
                ASSERT_EQ(pEncDst[1], pWinDst[1]);
            }
        }
        else
        {
            // Windows の MultiByteToWideChar はうまく変換できない文字をあらかじめ
            // 指定した既定値に変換して成功したことにする場合がある
            // それ以外の失敗の場合のみ FAIL とする
            if (pWinDst[0] != 0x3F || (pWinDst[0] == 0x3F && pSrc[0] == 0x003F))
            {
                if ((pSrc[0] == 0x3099 && pWinDst[0] == 0x81 && pWinDst[1] == 0x4A)
                    || (pSrc[0] == 0x309A && pWinDst[0] == 0x81 && pWinDst[1] == 0x4B))
                {
                    // 0x3099 (combining katakana-hiragana voiced sound mark) と
                    // 0x309A (combining katakana-hiragana semi-voiced sound mark) は
                    // Windows8/8.1 でのみ変換が成功するので、この場合は winResult が 2 になる
                    ASSERT_TRUE(winResult == 0 || winResult == 2);
                }
                else
                {
                    ASSERT_EQ(winResult, 0);    // MultiByteToWideChar も失敗しているはず
                }
            }
        }
    }
}

// MultiByteToWideChar/WideCharToMultiByte と処理結果を比較（文字列）
TEST(EncSjisTest, CompareStringWithWindows)
{
    const int seed = 1234567;   // 初期シード
    std::mt19937 engine(seed);
    std::uniform_int_distribution<int> typeDistribution(0, 2);
    std::uniform_int_distribution<int> asciiDistribution(0x01, 0x7F);
    std::uniform_int_distribution<int> halfKanaDistribution(0xA1, 0xDF);
    std::uniform_int_distribution<int> kanjiDistribution(0x40, 0x7E);

    const size_t sjisArraySize = 1024;
    const size_t utf16ArraySize = sjisArraySize * 2;
    unsigned char pSrc[sjisArraySize];
    uint16_t pEncUtf16Dst[utf16ArraySize], pWinUtf16Dst[utf16ArraySize];
    unsigned char pEncSjisDst[sjisArraySize], pWinSjisDst[sjisArraySize];
    int stringLength = 0;   // コードポイント数

    // 変換元の生成
    for (int i = 0; i < sjisArraySize - 1; ++i)
    {
        int type = typeDistribution(engine);
        switch (type)
        {
        case 0:
            // ASCII と制御文字を入れる
            pSrc[i] = static_cast<unsigned char>(asciiDistribution(engine));
            break;
        case 1:
            // 半角カナを入れる
            pSrc[i] = static_cast<unsigned char>(halfKanaDistribution(engine));
            break;
        case 2:
            if (i >= sjisArraySize - 2)
            {
                // 配列に余裕がない場合は ASCII を入れる
                pSrc[i] = static_cast<unsigned char>(asciiDistribution(engine));
            }
            else
            {
                // JIS 漢字を入れる
                // 上位バイトによってどの下位バイトが使えるかまちまちなので上位バイトを決め打ち
                pSrc[i] = 0x89;
                i++;
                pSrc[i] = static_cast<unsigned char>(kanjiDistribution(engine));
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        stringLength++;
    }
    pSrc[sjisArraySize - 1] = '\0';

    // Shift_JIS -> UTF-16 変換
    auto encResult = ConvertStringSjisToUtf16(pEncUtf16Dst, utf16ArraySize, reinterpret_cast<char*>(pSrc));
    ASSERT_EQ(encResult, nn::util::CharacterEncodingResult_Success);
    int winResult = MultiByteToWideChar(932, 0, reinterpret_cast<char*>(pSrc), sjisArraySize, reinterpret_cast<wchar_t*>(pWinUtf16Dst), utf16ArraySize);
    ASSERT_NE(winResult, 0);

    for (int i = 0; i < stringLength; ++i)
    {
        ASSERT_EQ(pEncUtf16Dst[i], pWinUtf16Dst[i]);
    }
    pEncUtf16Dst[stringLength] = '\0';
    pWinUtf16Dst[stringLength] = '\0';

    // UTF-16 -> Shift_JIS 変換
    encResult = ConvertStringUtf16ToSjis(reinterpret_cast<char*>(pEncSjisDst), sjisArraySize, pEncUtf16Dst);
    ASSERT_EQ(encResult, nn::util::CharacterEncodingResult_Success);
    winResult = WideCharToMultiByte(932, 0, reinterpret_cast<wchar_t*>(pWinUtf16Dst), stringLength, reinterpret_cast<char*>(pWinSjisDst), sjisArraySize, NULL, NULL);
    ASSERT_NE(winResult, 0);

    for (int i = 0; i < sjisArraySize - 1; ++i)
    {
        ASSERT_EQ(pEncSjisDst[i], pWinSjisDst[i]);
    }

}
#endif

