﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// general, neon 共通のテストを書くヘッダ
// このヘッダをインクルードする .cpp 側にて、どちらの実装を使用するかを決定する。

#include <nnt/benchmark/benchmark.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_VectorApi.h>
#include <nn/util/util_QuaternionApi.h>
#include <nn/util/util_Constant.h>

#if NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_GENERAL

    #define BENCHMARK_VECTOR2_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::Vector2fType); \

    #define BENCHMARK_VECTOR3_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::Vector3fType); \

    #define BENCHMARK_VECTOR4_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::Vector4fType); \

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::general::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.general.h"

#elif NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_NEON

    #define BENCHMARK_VECTOR2_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::Vector2fType); \

    #define BENCHMARK_VECTOR3_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::Vector3fType); \

    #define BENCHMARK_VECTOR4_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::Vector4fType); \

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::neon::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.neon.h"

#endif

#define BENCHMARK_VECTOR2(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_VECTOR2_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

#define BENCHMARK_VECTOR3(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_VECTOR3_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

#define BENCHMARK_VECTOR4(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_VECTOR4_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

//
// 2 次元ベクトル
//
BENCHMARK_VECTOR2(Vector2Benchmark, Set)
{
    static nn::util::Vector2fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, SetX)
{
    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetX)(&vector, -3.f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, SetY)
{
    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetY)(&vector, 1.f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, GetX)
{
    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f);

    while (state.KeepRunning()) {
        static float result = BENCHMARK_WRAPPER_FUNC(VectorGetX)(vector);
        NN_UNUSED(result);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, GetY)
{
    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f);

    while (state.KeepRunning()) {
        static float result = BENCHMARK_WRAPPER_FUNC(VectorGetY)(vector);
        NN_UNUSED(result);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Load)
{
    static nn::util::Vector2fType vector;

    static float value[2] = { -3.f, 1.f };

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLoad)(&vector, value);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Store)
{
    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f);

    static float value[2];

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorStore)(value, vector);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Zero)
{
    static nn::util::Vector2fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorZero)(&vector);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, IsZero)
{
    static bool result;

    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 1.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZero)(vector);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, IsZeroWithError)
{
    float error = 0.0001f;

    static bool result;

    static nn::util::Vector2fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 2 * error);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZero)(vector, error);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Add)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorAdd)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Subtract)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSubtract)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, MultiplyScalar)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, -1.f / 2.f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, MultiplyVector)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, DivideScalar)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, -2.f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, DivideVector)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Dot)
{
    static float result;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorDot)(vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Cross)
{
    static float result;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorCross)(vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Length)
{
    static float result;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Distance)
{
    static float result;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Normalize)
{
    static nn::util::Vector2fType vector;
    static float result;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorNormalize)(&vector, vector1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Maximize)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMaximize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Minimize)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMinimize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, Lerp)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f);
    static nn::util::Vector2fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLerp)(&vector, vector1, vector2, 0.5f);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, TransformWithRowMajor)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f);

    static nn::util::MatrixRowMajor3x2fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, TransformWithColumnMajor)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f);

    static nn::util::MatrixColumnMajor3x2fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, TransformNormalWithRowMajor)
{
    static nn::util::Vector2fType vector;

    nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f);

    nn::util::MatrixRowMajor3x2fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformNormal)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR2(Vector2Benchmark, TransformNormalWithColumnMajor)
{
    static nn::util::Vector2fType vector;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f);

    static nn::util::MatrixColumnMajor3x2fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformNormal)(&vector, vector1, matrix1);
    }
}

//
// 3 次元ベクトル
//
BENCHMARK_VECTOR3(Vector3Benchmark, Set)
{
    static nn::util::Vector3fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, SetX)
{
    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetX)(&vector, -3.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, SetY)
{
    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetY)(&vector, 1.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, SetZ)
{
    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetZ)(&vector, 1.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, GetX)
{
    static float result;

    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetX)(vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, GetY)
{
    static float result;

    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetY)(vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, GetZ)
{
    static float result;

    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetZ)(vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Load)
{
    static nn::util::Vector3fType vector;

    static float value[3] = { -3.f, 1.f, -2.f };

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLoad)(&vector, value);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Store)
{
    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f);

    static float value[3];

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorStore)(value, vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Zero)
{
    static nn::util::Vector3fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorZero)(&vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, IsZero)
{
    static bool result;

    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 1.f, 2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZero)(vector);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, IsZeroWithError)
{
    float error = 0.0001f;

    static bool result;

    static nn::util::Vector3fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 0.f, 2 * error);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZero)(vector, error);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Add)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorAdd)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Subtract)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSubtract)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, MultiplyScalar)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, -1.f / 2.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, MultiplyVector)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, DivideScalar)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, -2.f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, DivideVector)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Dot)
{
    static float result;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorDot)(vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Cross)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorCross)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Length)
{
    static float result;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Distance)
{
    static float result;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Normalize)
{
    static nn::util::Vector3fType vector;
    static float result;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorNormalize)(&vector, vector1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Maximize)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMaximize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Minimize)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMinimize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Lerp)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f);
    static nn::util::Vector3fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLerp)(&vector, vector1, vector2, 0.5f);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, Rotate)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, 0.1336306184530258f, 0.2672612369060516f, -0.4008918404579163f, 0.8660253882408142f);

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorRotate)(&vector, vector1, quaternion1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformWithRowMajor4x3)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixRowMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformWithRowMajor4x4)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixRowMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformWithColumnMajor4x3)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixRowMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformWithColumnMajor4x4)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixColumnMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformNormalWithRowMajor)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixRowMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformNormal)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformNormalWithColumnMajor)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixColumnMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformNormal)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformCoordWithRowMajor)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixRowMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformCoord)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR3(Vector3Benchmark, TransformCoordWithColumnMajor)
{
    static nn::util::Vector3fType vector;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f);

    static nn::util::MatrixColumnMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransformCoord)(&vector, vector1, matrix1);
    }
}

//
// 4 次元ベクトル
//
BENCHMARK_VECTOR4(Vector4Benchmark, Set)
{
    static nn::util::Vector4fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, SetX)
{
    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetX)(&vector, -3.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, SetY)
{
    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetY)(&vector, 1.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, SetZ)
{
    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetZ)(&vector, -2.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, SetW)
{
    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -1.f, -1.f, -1.f, -1.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSetW)(&vector, 4.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, GetX)
{
    static float result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetX)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, GetY)
{
    static float result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetY)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, GetZ)
{
    static float result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetZ)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, GetW)
{
    static float result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorGetW)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Load)
{
    static nn::util::Vector4fType vector;

    static float value[4] = { -3.f, 1.f, -2.f, 4.f };

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLoad)(&vector, value);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Store)
{
    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, -3.f, 1.f, -2.f, 4.f);

    static float value[4];

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorStore)(value, vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Zero)
{
    static nn::util::Vector4fType vector;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorZero)(&vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, IsZero)
{
    static bool result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 1.f, 2.f, 3.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZero)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, IsZeroWOne)
{
    static bool result;

    static nn::util::Vector4fType vector;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector, 0.f, 1.f, 2.f, 3.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorIsZeroWOne)(vector);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Add)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorAdd)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Subtract)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorSubtract)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, MultiplyScalar)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, -1.f / 2.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, MultiplyVector)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMultiply)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, DivideScalar)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, -2.f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, DivideVector)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorDivide)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Dot)
{
    static float result;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorDot)(vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Length)
{
    static float result;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Distance)
{
    static float result;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorLength)(vector1);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Normalize)
{
    static nn::util::Vector4fType vector;
    static float result;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(VectorNormalize)(&vector, vector1);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Maximize)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMaximize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Minimize)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorMinimize)(&vector, vector1, vector2);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, Lerp)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType vector2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorLerp)(&vector, vector1, vector2, 0.5f);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, TransformWithRowMajor)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f, -8.f);

    static nn::util::MatrixRowMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}

BENCHMARK_VECTOR4(Vector4Benchmark, TransformWithColumnMajor)
{
    static nn::util::Vector4fType vector;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, 2.f, -4.f, 6.f, -8.f);

    static nn::util::MatrixColumnMajor4x4fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(VectorTransform)(&vector, vector1, matrix1);
    }
}
