﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/util/util_PlacementArray.h>

namespace {

class Example
{
public:
    Example()
        : m_Count( s_Count )
        , m_IsDestructed( false )
    {
        ++s_Count;
    }

    ~Example()
    {
        if ( m_IsDestructed )
        {
            s_IsSuccess = false;
        }
        else
        {
            m_IsDestructed = true;
        }
        --s_Count;
    }

    int GetCount() const
    {
        return m_Count;
    }

    static void Initialize()
    {
        s_IsSuccess = true;
        s_Count = 0;
    }

public:
    static bool s_IsSuccess;
    static int s_Count;

private:
    int m_Count;
    bool m_IsDestructed;
};

bool Example::s_IsSuccess = true;
int Example::s_Count = 0;

} // anonymous namespace

namespace nn { namespace util {

template class nn::util::PlacementArray< int >;
template class nn::util::PtrPlacementArray< int >;
template class nn::util::PlacementArray< Example >;
template class nn::util::PtrPlacementArray< Example >;

}} // nn::util

namespace {

TEST( PlacementArrayTest, Empty )
{
    nn::util::PtrPlacementArray< int > empty;

    EXPECT_TRUE( empty.empty() );
    EXPECT_EQ( 0, empty.size() );
    EXPECT_EQ( NULL, empty.data() );
}

TEST( PlacementArrayTest, Access )
{
    typedef nn::util::PtrPlacementArray< int > Container;
    const int count = 8;
    size_t bufferSize = Container::CalculateWorkMemorySize( count );
    void* buffer = malloc( bufferSize );
    Container arr( buffer, bufferSize, count );
    const Container& carr = arr;

    EXPECT_FALSE( arr.empty() );
    EXPECT_EQ( count, arr.size() );

    int sample[] = { 0, 1, 2, 3, 4, 5, 6, 7 };
    for ( int i = 0; i < count; ++i )
    {
        arr.SetPtr( i,  &sample[ i ] );
    }

    for ( int i = 0; i < count; ++i )
    {
        EXPECT_EQ( sample[ i ], arr[ i ] );
        EXPECT_EQ( sample[ i ], arr.at( i ) );
        EXPECT_EQ( sample[ i ], carr[ i ] );
        EXPECT_EQ( sample[ i ], carr.at( i ) );
    }

    EXPECT_EQ( sample[ 0 ], arr.front() );
    EXPECT_EQ( sample[ 0 ], carr.front() );
    EXPECT_EQ( sample[ count - 1 ], arr.back() );
    EXPECT_EQ( sample[ count - 1 ], carr.back() );

    for ( Container::const_iterator iter = arr.cbegin(), citer = carr.begin(), end = arr.cend();
        iter != end; ++iter, ++citer )
    {
        EXPECT_EQ( *iter, *citer );
    }

    EXPECT_EQ( count, carr.end() - carr.begin() );
    free( buffer );
}

TEST( PlacementArrayTest, Destructor )
{
    const int count = 7;
    size_t bufferSize = nn::util::PlacementArray< Example >::CalculateWorkMemorySize( count );
    void* buffer = malloc( bufferSize );
    Example::Initialize();
    {
        nn::util::PlacementArray< Example > arr( buffer, bufferSize, count );
        EXPECT_EQ( count, Example::s_Count );
    }
    EXPECT_EQ( 0, Example::s_Count );
    EXPECT_TRUE( Example::s_IsSuccess );
    free( buffer );
}

TEST( PlacementArrayTest, Sort )
{
    typedef nn::util::PtrPlacementArray< Example > ExamplePtrArray;
    const int count = 16;
    size_t bufferSize = ExamplePtrArray::CalculateWorkMemorySize( count );
    void* buffer = malloc( bufferSize );

    Example::Initialize();
    ExamplePtrArray arr( buffer, bufferSize, count );

    for ( int i = 0; i < count; ++i )
    {
        arr.SetPtr( i,  new Example() );
        EXPECT_EQ( i, arr[ i ].GetCount() );
    }
    arr.Sort( []( const Example& lhs, const Example& rhs ){ return lhs.GetCount() > rhs.GetCount(); } );

    for ( int i = 0; i < count; ++i )
    {
        EXPECT_EQ( count - i - 1, arr[ i ].GetCount() );
    }

    for ( ExamplePtrArray::iterator iter = arr.begin(), end = arr.end(); iter != end; ++iter )
    {
        delete( iter.ToPtr() );
    }

    free( buffer );
}

TEST( PlacementArrayDeathTest, Access )
{
    typedef nn::util::PtrPlacementArray< int > Container;
    Container arr;
    const Container& carr = arr;

    EXPECT_DEATH_IF_SUPPORTED( arr[ 0 ], "" );
    EXPECT_DEATH_IF_SUPPORTED( carr[ 0 ], "" );
    EXPECT_DEATH_IF_SUPPORTED( arr.at( 0 ), "" );
    EXPECT_DEATH_IF_SUPPORTED( carr.at( 0 ), "" );

    EXPECT_DEATH_IF_SUPPORTED( arr.front(), "" );
    EXPECT_DEATH_IF_SUPPORTED( carr.front(), "" );
    EXPECT_DEATH_IF_SUPPORTED( arr.back(), "" );
    EXPECT_DEATH_IF_SUPPORTED( carr.back(), "" );
}

} // anonymous namespace
