﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnc/nn_Common.h>
#include <nnc/nn_Abort.h>
#include <nnc/nn_Assert.h>
#include <nnc/nn_Log.h>
#include <string.h>

// NNC_ALIGNAS には整数リテラルしか書けないので、const 定数ではなくマクロを使用する。
#define NNC_DETAIL_TEST_UTIL_ALIGNMENT 16 // NOLINT(readability/define)


#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
// 明示的にパディングを挟むと NNC_ALIGNAS のテストにならないのでここでは警告を無効化します。
#pragma warning( push )
#pragma warning( disable:4324 )
#endif

NNC_NORETURN void MacroForCDeathTest_ExpectedAssertFailure( void )
{
    NNC_ASSERT(false, "Expected assert failure.");
    for (;;) {}
}

NNC_NORETURN void MacroForCDeathTest_ExpectedAbort( void )
{
    NNC_ABORT_UNLESS(false, "Expected abort.");
    for (;;) {}
}

NNC_NORETURN NNC_NOINLINE
void ThrowException( void )
{
    NNC_ASSERT(true, "assert.");
    NNC_ABORT_UNLESS(true, "abort.");
    for (;;) {}
}

NNC_FORCEINLINE bool ForceInlineFunc(void* ptr)
{
    return ptr != NULL;
}

static NNC_INLINE bool StaticInlineFunc(void* ptr)
{
    return ptr != NULL;
}

extern NNC_INLINE bool ExternInlineFunc(void* ptr)
{
    return ptr != NULL;
}

struct NNC_ALIGNAS( NNC_DETAIL_TEST_UTIL_ALIGNMENT ) MacroTest
{
    // C言語では未対応
    //static const int Alignment = NNC_DETAIL_TEST_UTIL_ALIGNMENT;

    char m_Padding;
};

struct AlignmentTest
{
    char padding;
    NNC_ALIGNAS( NNC_DETAIL_TEST_UTIL_ALIGNMENT ) int aligned;
};

#if NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC
#pragma warning( pop )
#endif

// C言語では未対応
//NNC_DEFINE_STATIC_CONSTANT( MacroTest::Alignment );

// この関数自身の名前を NNC_CURRENT_FUNCTION_NAME で取得します
const char* GetNameOfThisFunctionItself()
{
    return NNC_CURRENT_FUNCTION_NAME;
}

void InvokeUnexpectedDefaultUnlessZero(int n)
{
    switch (n)
    {
    case 0:
        break;
    default: NNC_UNEXPECTED_DEFAULT;
    }
}

//--------------------------------------------------------------------------------------------------

#define EXPECT_EQ( Val1, Val2 ) if (NNC_STATIC_CONDITION(Val1 != Val2)) return false;
#define EXPECT_NE( Val1, Val2 ) if (NNC_STATIC_CONDITION(Val1 == Val2)) return false;

bool MacroForC_LanguageMacro( void )
{
    NNC_ALIGNAS( NNC_DETAIL_TEST_UTIL_ALIGNMENT ) char aligned = 0;
    NNC_UNUSED( aligned ); // 変数としては使用しないので未使用扱いとします。

    EXPECT_EQ( NNC_DETAIL_TEST_UTIL_ALIGNMENT, NNC_ALIGNOF( aligned ) );
    EXPECT_EQ( NNC_DETAIL_TEST_UTIL_ALIGNMENT, NNC_ALIGNOF( struct AlignmentTest ) );
    EXPECT_EQ( NNC_DETAIL_TEST_UTIL_ALIGNMENT, NNC_ALIGNOF( struct MacroTest ) );

    EXPECT_EQ( 16, sizeof( struct MacroTest ) );
    EXPECT_EQ( 128, NNC_BITSIZEOF( struct MacroTest ) );

    // NNC_CURRENT_FUNCTION_NAME の値に関数名が含まれていることを期待します。
    // 処理系定義なので厳密には保証がありませんが、
    // NNC_CURRENT_FUNCTION_NAME の実装が誤っているにもかかわらずテストを通過することはありません。
    EXPECT_NE( NULL, strstr(GetNameOfThisFunctionItself(), "GetNameOfThisFunctionItself") );

    return true;
}

bool MacroForC_CompileAndRun( void )
{
    // コンパイル及び実行のテストです。
    // テストを完走することでしか結果を取得できません。

    // コピー禁止かつムーブ禁止のクラスです。
    struct MacroTest macroTest;
    bool boolean;

    static int NNC_MAKE_TEMPORARY_NAME( dummy ) = 0;
    static int NNC_MAKE_TEMPORARY_NAME( dummy ) = 1;

    // 修飾子付きの関数群のテストです。

    //ThrowException();

    // FORCEINLINE のテスト
    boolean = ForceInlineFunc( &macroTest );
    NNC_UNUSED( boolean );

    boolean = StaticInlineFunc( &macroTest );
    NNC_UNUSED( boolean );

    boolean = ExternInlineFunc( &macroTest );
    NNC_UNUSED( boolean );

    // 各種マクロのテストです。

    do
    {
    } while ( NNC_STATIC_CONDITION( 0 ) );

    // Windows.h で定義されている min マクロを回避します。
    //int min = min NNC_PREVENT_MACRO_FUNC ();
    //NNC_UNUSED( min );

    NNC_PRAGMA_PUSH_WARNINGS
    NNC_DISABLE_WARNING_UNUSED_VAR
    NNC_DISABLE_WARNING_LOCAL_STATIC

    NNC_PRAGMA_POP_WARNINGS

    switch (0)
    {
    case 0:
        // Clang でエラーとなるため、C言語版では未対応
        //NNC_FALL_THROUGH;
        break;
    default:
        break;
    }

    return true;
}

bool MacroForCDeathTest_UnexpectedDefault( void )
{
    // NNC_UNEXPECTED_DEFAULT で停止することのテストです。
    // 呼び出し関数側で EXPECT_DEATH してください。
    InvokeUnexpectedDefaultUnlessZero(-1);

    return false;
}

NNC_PRAGMA_PUSH_WARNINGS
NNC_ENABLE_WARNING_IMPLICIT_PADDING_WITHIN_CLASS

#if 0 // VC で Warning が出るためパディングは一旦サポートしない
bool MacroForC_PaddingMacro( void )
{
    // 各マクロのサイズとアライメントが正しいことの確認
    {
        struct PaddingSizeTest1
        {
            char a;
            NNC_PADDING1;
        };

        struct PaddingSizeTest2
        {
            char a;
            NNC_PADDING2;
        };

        struct PaddingSizeTest3
        {
            char a;
            NNC_PADDING3;
        };

        struct PaddingSizeTest4
        {
            char a;
            NNC_PADDING4;
        };

        struct PaddingSizeTest5
        {
            char a;
            NNC_PADDING5;
        };

        struct PaddingSizeTest6
        {
            char a;
            NNC_PADDING6;
        };

        struct PaddingSizeTest7
        {
            char a;
            NNC_PADDING7;
        };

        struct PaddingSizeTest8
        {
            char a;
            NNC_PADDING8;
        };

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest1), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest1), 2);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest2), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest2), 3);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest3), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest3), 4);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest4), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest4), 5);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest5), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest5), 6);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest6), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest6), 7);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest7), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest7), 8);

        EXPECT_EQ(NNC_ALIGNOF(struct PaddingSizeTest8), 1);
        EXPECT_EQ(sizeof(struct PaddingSizeTest8), 9);
    }
    return true;
}
#endif

NNC_PRAGMA_POP_WARNINGS
