﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/nntest.h>
#include <nn/nn_Common.h>

template <typename TList>
void TestListIterator()
{
    typedef typename TList::value_type ValueType;
    typedef typename TList::iterator IteratorType;
    typedef typename TList::reverse_iterator ReverseIteratorType;

    TList list;
    const int nodeNumber = 10;
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        list.push_back(items[i]);
    }

    // イテレータの操作
    {
        IteratorType iterator = list.begin();
        IteratorType endIterator = list.end();
        int n = 0;
        // インクリメントのチェック
        while (iterator != endIterator)
        {
            EXPECT_EQ(iterator->Get(), n);
            n++;
            iterator++;
        }
        EXPECT_EQ(n, nodeNumber);
        n--;
        iterator--;
        // デクリメントのチェック
        while (iterator != endIterator)
        {
            EXPECT_EQ(iterator->Get(), n);
            n--;
            iterator--;
        }
        EXPECT_EQ(n, -1);
    }

    // リバースイテレータの操作
    {
        ReverseIteratorType reverseIterator = list.rbegin();
        ReverseIteratorType endIterator = list.rend();
        int n = nodeNumber - 1;
        // インクリメントのチェック
        while (reverseIterator != endIterator)
        {
            EXPECT_EQ(reverseIterator->Get(), n);
            n--;
            reverseIterator++;
        }
        EXPECT_EQ(n, -1);
        n++;
        reverseIterator--;
        // デクリメントのチェック
        while (reverseIterator != endIterator)
        {
            EXPECT_EQ(reverseIterator->Get(), n);
            n++;
            reverseIterator--;
        }
        EXPECT_EQ(n, nodeNumber);
    }

    // clearおよびemptyのテスト
    EXPECT_EQ(list.size(), nodeNumber);
    EXPECT_EQ(list.empty(), false);
    list.clear();
    EXPECT_EQ(list.size(), 0);
    EXPECT_EQ(list.empty(), true);
}

template <typename TList>
void TestListEraseAndInsert()
{
    typedef typename TList::value_type ValueType;
    typedef typename TList::iterator IteratorType;

    TList list;
    const int nodeNumber = 10;
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        list.push_back(items[i]);
    }

    // 偶数の要素を削除
    {
        IteratorType iterator = list.begin();
        IteratorType endIterator = list.end();
        while (iterator != endIterator)
        {
            if (iterator->Get() % 2 == 0)
            {
                IteratorType eraseIterator = iterator;
                iterator++;
                list.erase(eraseIterator);
                continue;
            }
            iterator++;
        }
    }

    // 要素の挿入
    {
        IteratorType iterator = list.begin();
        IteratorType endIterator = list.end();
        while (iterator != endIterator)
        {
            list.insert(iterator, items[nodeNumber - 1 - iterator->Get()]);
            iterator++;
        }
    }

    // 検証
    EXPECT_EQ(list.size(), nodeNumber);
    {
        int numbers[nodeNumber] = { 8, 1, 6, 3, 4, 5, 2, 7, 0, 9 };
        for (int i = 0; i < nodeNumber; i++)
        {
            EXPECT_EQ(list.front().Get(), numbers[i]);
            list.pop_front();
        }
    }
}

template <typename TList>
void TestListSplice()
{
    typedef typename TList::value_type ValueType;
    typedef typename TList::iterator IteratorType;

    TList listA;
    TList listB;
    const int nodeNumber = 10;
    ValueType items[nodeNumber];

    // 要素を半分ずつ振り分けておく
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        if (i < nodeNumber / 2)
        {
            listA.push_back(items[i]);
        }
        else
        {
            listB.push_back(items[i]);
        }
    }

    // 1つ転送
    {
        IteratorType iteratorA = listA.begin();
        IteratorType iteratorB = listB.begin();
        iteratorA++;
        iteratorB++;
        listA.splice(iteratorA, listB, iteratorB);
    }

    // 領域を転送
    {
        IteratorType iteratorA = listA.begin();
        IteratorType iteratorB = listB.begin();
        IteratorType iteratorC = listB.begin();
        iteratorA++;
        iteratorC++;
        iteratorC++;
        listA.splice(iteratorA, listB, iteratorB, iteratorC);
    }

    // 全て転送
    {
        IteratorType iterator = listA.begin();
        iterator++;
        listA.splice(iterator, listB);
    }

    // 検証
    EXPECT_EQ(listA.size(), nodeNumber);
    EXPECT_EQ(listB.size(), 0);
    {
        int numbers[nodeNumber] = { 0, 8, 9, 5, 7, 6, 1, 2, 3, 4 };
        for (int i = 0; i < nodeNumber; i++)
        {
            EXPECT_EQ(listA.front().Get(), numbers[i]);
            listA.pop_front();
        }
    }
}

template <typename TList>
void TestListDuplicatedInsert()
{
    typedef typename TList::value_type ValueType;

    TList list;
    const int nodeNumber = 10;
    ValueType items[nodeNumber];

    for (int i = 0; i < nodeNumber; i++)
    {
        list.push_back(items[i]);
    }

    for (int i = 0; i < nodeNumber; i++)
    {
        EXPECT_DEATH_IF_SUPPORTED(list.push_back(items[i]), "");
        EXPECT_DEATH_IF_SUPPORTED(list.push_front(items[i]), "");
        EXPECT_DEATH_IF_SUPPORTED(list.insert(list.begin(), items[i]), "");
    }
}
