﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/util/util_FloatFormat.h>
#include <limits>
#include <cmath>

template class nn::util::FloatFormat< 1, 8, 23 >;
template class nn::util::FloatFormat< 1, 5, 10 >;
template class nn::util::FloatFormat< 0, 5, 6 >;
template class nn::util::FloatFormat< 0, 5, 5 >;

namespace {

const float FloatValueZero = 0.0f;
const float FloatValueMin = std::numeric_limits< float >::min();
const float FloatValueMax = std::numeric_limits< float >::max();
const float FloatValueDenormalMin = std::numeric_limits< float >::denorm_min();
const float FloatValueDenormalMax = FloatValueMin - FloatValueDenormalMin;
const float FloatValueInfinity = std::numeric_limits< float >::infinity();
const float FloatValueSignalingNan = std::numeric_limits< float >::signaling_NaN();
const float FloatValueQuietNan = std::numeric_limits< float >::quiet_NaN();

}

TEST( FloatFormatTest, Float32 )
{
    typedef nn::util::FloatFormat< 1, 8, 23 > Float32;

    EXPECT_EQ( 0x00000000u, Float32::Encode( FloatValueZero ) );
    EXPECT_EQ( 0x00800000u, Float32::Encode( FloatValueMin ) );
    EXPECT_EQ( 0x7F7FFFFFu, Float32::Encode( FloatValueMax ) );
    EXPECT_EQ( 0x00000000u, Float32::Encode( FloatValueDenormalMin ) );
    EXPECT_EQ( 0x00000000u, Float32::Encode( FloatValueDenormalMax ) );
    EXPECT_EQ( 0x7F800000u, Float32::Encode( FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x7F800001u ), Float32::Encode( FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x7FFFFFFFu ), Float32::Encode( FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x7F800001u ), Float32::Encode( FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x7FFFFFFFu ), Float32::Encode( FloatValueQuietNan ) );

    EXPECT_EQ( 0x80000000u, Float32::Encode( - FloatValueZero ) | 0x80000000u );
    EXPECT_EQ( 0x80800000u, Float32::Encode( - FloatValueMin ) );
    EXPECT_EQ( 0xFF7FFFFFu, Float32::Encode( - FloatValueMax ) );
    EXPECT_EQ( 0x80000000u, Float32::Encode( - FloatValueDenormalMin ) );
    EXPECT_EQ( 0x80000000u, Float32::Encode( - FloatValueDenormalMax ) );
    EXPECT_EQ( 0xFF800000u, Float32::Encode( - FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0xFF800001u ), Float32::Encode( - FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0xFFFFFFFFu ), Float32::Encode( - FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0xFF800001u ), Float32::Encode( - FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0xFFFFFFFFu ), Float32::Encode( - FloatValueQuietNan ) );

    EXPECT_EQ( FloatValueZero, Float32::Decode( 0x00000000u ) );
    EXPECT_EQ( FloatValueMin, Float32::Decode( 0x00800000u ) );
    EXPECT_EQ( FloatValueMax, Float32::Decode( 0x7F7FFFFFu ) );
    EXPECT_EQ( FloatValueZero, Float32::Decode( 0x00000001u ) );
    EXPECT_EQ( FloatValueZero, Float32::Decode( 0x007FFFFFu ) );
    EXPECT_EQ( FloatValueInfinity, Float32::Decode( 0x7F800000u ) );

    EXPECT_EQ( - FloatValueZero, Float32::Decode( 0x80000000u ) );
    EXPECT_EQ( - FloatValueMin, Float32::Decode( 0x80800000u ) );
    EXPECT_EQ( - FloatValueMax, Float32::Decode( 0xFF7FFFFFu ) );
    EXPECT_EQ( - FloatValueZero, Float32::Decode( 0x80000001u ) );
    EXPECT_EQ( - FloatValueZero, Float32::Decode( 0x807FFFFFu ) );
    EXPECT_EQ( - FloatValueInfinity, Float32::Decode( 0xFF800000u ) );
}

TEST( FloatFormatTest, Float16 )
{
    typedef nn::util::FloatFormat< 1, 5, 10 > Float16;

    float min = std::pow( 2.0f, 1 - Float16::ExponentialBias );
    float maxExponential = static_cast< float >( 1 << ( ( 1 << Float16::ExponentialBit ) - Float16::ExponentialBias - 2 - Float16::FractionalBit ) );
    float max = ( ( 1 << ( 1 + Float16::FractionalBit ) ) - 1 ) * maxExponential;

    EXPECT_EQ( 0x0000u, Float16::Encode( FloatValueZero ) );
    EXPECT_EQ( 0x0000u, Float16::Encode( FloatValueMin ) );
    EXPECT_EQ( 0x7C00u, Float16::Encode( FloatValueMax ) );
    EXPECT_EQ( 0x0000u, Float16::Encode( FloatValueDenormalMin ) );
    EXPECT_EQ( 0x0000u, Float16::Encode( FloatValueDenormalMax ) );
    EXPECT_EQ( 0x7C00u, Float16::Encode( FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x7C01u ), Float16::Encode( FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x7FFFu ), Float16::Encode( FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x7C01u ), Float16::Encode( FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x7FFFu ), Float16::Encode( FloatValueQuietNan ) );
    EXPECT_EQ( 0x0400u, Float16::Encode( min ) );
    EXPECT_EQ( 0x7BFFu, Float16::Encode( max ) );

    EXPECT_EQ( 0x8000u, Float16::Encode( - FloatValueZero ) | 0x8000u );
    EXPECT_EQ( 0x8000u, Float16::Encode( - FloatValueMin ) );
    EXPECT_EQ( 0xFC00u, Float16::Encode( - FloatValueMax ) );
    EXPECT_EQ( 0x8000u, Float16::Encode( - FloatValueDenormalMin ) );
    EXPECT_EQ( 0x8000u, Float16::Encode( - FloatValueDenormalMax ) );
    EXPECT_EQ( 0xFC00u, Float16::Encode( - FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0xFC01u ), Float16::Encode( - FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0xFFFFu ), Float16::Encode( - FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0xFC01u ), Float16::Encode( - FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0xFFFFu ), Float16::Encode( - FloatValueQuietNan ) );
    EXPECT_EQ( 0x8400u, Float16::Encode( - min ) );
    EXPECT_EQ( 0xFBFFu, Float16::Encode( - max ) );

    EXPECT_EQ( FloatValueZero,       Float16::Decode( 0x0000u ) );
    EXPECT_EQ( min,         Float16::Decode( 0x0400u ) );
    EXPECT_EQ( max,         Float16::Decode( 0x7BFFu ) );
    EXPECT_EQ( FloatValueZero,       Float16::Decode( 0x0001u ) );
    EXPECT_EQ( FloatValueZero,       Float16::Decode( 0x03FFu ) );
    EXPECT_EQ( FloatValueInfinity,   Float16::Decode( 0x7C00u ) );

    EXPECT_EQ( - FloatValueZero,      Float16::Decode( 0x8000u ) );
    EXPECT_EQ( - min,        Float16::Decode( 0x8400u ) );
    EXPECT_EQ( - max,        Float16::Decode( 0xFBFFu ) );
    EXPECT_EQ( - FloatValueZero,      Float16::Decode( 0x8001u ) );
    EXPECT_EQ( - FloatValueZero,      Float16::Decode( 0x83FFu ) );
    EXPECT_EQ( - FloatValueInfinity, Float16::Decode( 0xFC00u ) );

}

TEST( FloatFormatTest, Float11 )
{
    typedef nn::util::FloatFormat< 0, 5, 6 > Float11;

    float min = std::pow( 2.0f, 1 - Float11::ExponentialBias );
    float maxExponential = static_cast< float >( 1 << ( ( 1 << Float11::ExponentialBit ) - Float11::ExponentialBias - 2 - Float11::FractionalBit ) );
    float max = ( ( 1 << ( 1 + Float11::FractionalBit ) ) - 1 ) * maxExponential;

    EXPECT_EQ( 0x0000u, Float11::Encode( FloatValueZero ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( FloatValueMin ) );
    EXPECT_EQ( 0x07BFu, Float11::Encode( FloatValueMax ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( FloatValueDenormalMin ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( FloatValueDenormalMax ) );
    EXPECT_EQ( 0x07C0u, Float11::Encode( FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x07C1u ), Float11::Encode( FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x07FFu ), Float11::Encode( FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x07C1u ), Float11::Encode( FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x07FFu ), Float11::Encode( FloatValueQuietNan ) );
    EXPECT_EQ( 0x0040u, Float11::Encode( min ) );
    EXPECT_EQ( 0x07BFu, Float11::Encode( max ) );

    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueZero ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueMin ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueMax ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueDenormalMin ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueDenormalMax ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x07C1u ), Float11::Encode( - FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x07FFu ), Float11::Encode( - FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x07C1u ), Float11::Encode( - FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x07FFu ), Float11::Encode( - FloatValueQuietNan ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - min ) );
    EXPECT_EQ( 0x0000u, Float11::Encode( - max ) );

    EXPECT_EQ( FloatValueZero,       Float11::Decode( 0x0000u ) );
    EXPECT_EQ( min,         Float11::Decode( 0x0040u ) );
    EXPECT_EQ( max,         Float11::Decode( 0x07BFu ) );
    EXPECT_EQ( FloatValueZero,       Float11::Decode( 0x0001u ) );
    EXPECT_EQ( FloatValueZero,       Float11::Decode( 0x003Fu ) );
    EXPECT_EQ( FloatValueInfinity,   Float11::Decode( 0x07C0u ) );
}

TEST( FloatFormatTest, Float10 )
{
    typedef nn::util::FloatFormat< 0, 5, 5 > Float10;

    float min = std::pow( 2.0f, 1 - Float10::ExponentialBias );
    float maxExponential = static_cast< float >( 1 << ( ( 1 << Float10::ExponentialBit ) - Float10::ExponentialBias - 2 - Float10::FractionalBit ) );
    float max = ( ( 1 << ( 1 + Float10::FractionalBit ) ) - 1 ) * maxExponential;

    EXPECT_EQ( 0x0000u, Float10::Encode( FloatValueZero ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( FloatValueMin ) );
    EXPECT_EQ( 0x03DFu, Float10::Encode( FloatValueMax ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( FloatValueDenormalMin ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( FloatValueDenormalMax ) );
    EXPECT_EQ( 0x03E0u, Float10::Encode( FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x03E1u ), Float10::Encode( FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x03FFu ), Float10::Encode( FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x03E1u ), Float10::Encode( FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x03FFu ), Float10::Encode( FloatValueQuietNan ) );
    EXPECT_EQ( 0x0020u, Float10::Encode( min ) );
    EXPECT_EQ( 0x03DFu, Float10::Encode( max ) );

    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueZero ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueMin ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueMax ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueDenormalMin ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueDenormalMax ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - FloatValueInfinity ) );
    EXPECT_LE(  static_cast< int >( 0x03E1u ), Float10::Encode( - FloatValueSignalingNan ) );
    EXPECT_GE(  static_cast< int >( 0x03FFu ), Float10::Encode( - FloatValueSignalingNan ) );
    EXPECT_LE(  static_cast< int >( 0x03E1u ), Float10::Encode( - FloatValueQuietNan ) );
    EXPECT_GE(  static_cast< int >( 0x03FFu ), Float10::Encode( - FloatValueQuietNan ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - min ) );
    EXPECT_EQ( 0x0000u, Float10::Encode( - max ) );

    EXPECT_EQ( FloatValueZero,       Float10::Decode( 0x0000u ) );
    EXPECT_EQ( min,         Float10::Decode( 0x0020u ) );
    EXPECT_EQ( max,         Float10::Decode( 0x03DFu ) );
    EXPECT_EQ( FloatValueZero,       Float10::Decode( 0x0001u ) );
    EXPECT_EQ( FloatValueZero,       Float10::Decode( 0x001Fu ) );
    EXPECT_EQ( FloatValueInfinity,   Float10::Decode( 0x03E0u ) );
}
