﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RadioGroup.h"
#include "../UsbDiag_Theme.h"

namespace nnt {
namespace usb {

RadioGroup::RadioGroup(const glv::Rect& rect, size_t nx)
    : glv::Table(std::string(nx, 'x').c_str(), 0, 0)
    , m_Count(0)
    , m_Rect(rect)
    , m_pActiveButton(nullptr)
{
    disable(glv::Property::FocusHighlight |
            glv::Property::HitTest        |
            glv::Property::DrawGrid       |
            glv::Property::DrawBorder     |
            glv::Property::DrawBack);

    cloneStyle().colors().border = glv::Color(1, 0, 0);
}

RadioGroup::RadioGroup(std::initializer_list<std::string> captions, const glv::Rect& rect, size_t nx)
    : RadioGroup(rect, nx)
{
    onStyle  = {
        25, glv::Color(0), glv::Color(0.6), glv::Color(0.1), glv::Place::CC
    };
    offStyle = {
        25, glv::Color(1), glv::Color(0.0), glv::Color(0.5), glv::Place::CC
    };

    for (const auto& caption : captions)
    {
        Add(caption);
    }
}

RadioGroup::~RadioGroup()
{
}

RadioGroup&
RadioGroup::Add(const std::string& caption)
{
    add(new RadioButton(m_Rect, caption, *this, m_Count++));
    arrange();

    return *this;
}

glv::View*
RadioGroup::Select(int index)
{
    auto *view = child;

    NN_ASSERT(view != nullptr);

    for (int i = 0; i < index; i++)
    {
        view = view->sibling;
        NN_ASSERT(view != nullptr);
    }

    auto& button = *static_cast<RadioButton*>(view);

    Select(button);
    button.SetState(Button::State::On);

    return view;
}

void
RadioGroup::Select(RadioButton& button)
{
    if (&button == m_pActiveButton)
    {
        return;
    }

    if (m_pActiveButton)
    {
        m_pActiveButton->ToggleState();
        m_pActiveButton->enable(glv::Property::Controllable);

    }

    m_pActiveButton = &button;
    m_pActiveButton->disable(glv::Property::Controllable);

    OnChange(m_pActiveButton->GetValue());
}

void
RadioGroup::UpdateFocus()
{
    auto& g = static_cast< glv::GLV& >( this->root() );
    if (m_pActiveButton)
    {
        g.setFocus(m_pActiveButton);
    }
}

void
RadioGroup::OnChange(int value)
{
    notify(this, glv::Update::Value, &value);
}

RadioButton::RadioButton(const glv::Rect& rect, const std::string& caption,
                         RadioGroup& group, int value)
    : Button(rect, caption, false, group.onStyle, group.offStyle)
    , m_Group(group)
    , m_Value(value)
{
    disable(glv::Property::FocusToTop);
}

void
RadioButton::OnClicked()
{
    m_Group.Select(*this);
}

int
RadioButton::GetValue() const
{
    return m_Value;
}

} // ~usb
} // ~nnt
