﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../UsbDiag_Theme.h"
#include "Button.h"

namespace nnt {
namespace usb {

Button::Button(const glv::Rect& rect,
               const std::string& caption,
               bool  momentary,
               const Style& styleOn,
               const Style& styleOff)
    : glv::Button(rect, momentary, 0, 0)
    , m_Label(caption)
    , m_State(State::Off)
    , m_StyleOn(styleOn)
    , m_StyleOff(styleOff)
{
    cloneStyle();
    m_Label.cloneStyle();

    // The lable is put into a frame, so it's easier to handle the padding
    m_Frame.extent(
        rect.width()  - ButtonPaddingX * 2,
        rect.height() - ButtonPaddingY * 2
    );
    m_Frame.enable(glv::Property::CropChildren);
    m_Frame.disable(glv::Property::HitTest);

    *this << (m_Frame << m_Label);

    m_Frame.anchor(glv::Place::CC);
    m_Frame.pos(glv::Place::CC, width() * 0.5f, height() * 0.5f);

    UpdateStyle();
}

bool
Button::onEvent(glv::Event::t e, glv::GLV& g)
{
    bool isPress   = false;
    bool isRelease = false;

    switch (e)
    {
    case glv::Event::DebugPadDown:
        if (g.getDebugPadEvent().IsButtonDown<glv::DebugPadEventType::Button::Ok>())
        {
            isPress = true;
        }
        break;

    case glv::Event::BasicPadDown:
        if (g.getBasicPadEvent().IsButtonDown<glv::BasicPadEventType::Button::Ok>())
        {
            isPress = true;
        }
        break;

    case glv::Event::MouseDown:
        isPress = true;
        break;

    case glv::Event::DebugPadUp:
        if (g.getDebugPadEvent().IsButtonUp<glv::DebugPadEventType::Button::Ok>())
        {
            isRelease = true;
        }
        break;

    case glv::Event::BasicPadUp:
        if (g.getBasicPadEvent().IsButtonUp<glv::BasicPadEventType::Button::Ok>())
        {
            isRelease = true;
        }
        break;

    case glv::Event::MouseUp:
        isRelease = true;

    default:
        break;
    }

    if (isPress)
    {
        if (enabled(glv::Property::Momentary))
        {
            ToggleState();
        }
        return false;
    }

    if (isRelease)
    {
        ToggleState();
        OnClicked();
        return false;
    }

    return true;
}

Button::State
Button::GetState() const
{
    return m_State;
}

void
Button::SetState(State state)
{
    if (state != m_State)
    {
        m_State = state;
        UpdateStyle();
    }
}

void
Button::ToggleState()
{
    UpdateStyle();
    SetState(m_State == State::On ? State::Off : State::On);
}

void
Button::OnClicked()
{
    // do nothing
}

void
Button::StyleOn(const Style& style)
{
    m_StyleOn = style;
}

void
Button::StyleOff(const Style& style)
{
    m_StyleOff = style;
}

void
Button::ApplyStyle(const Style& style)
{
    float sx, sy;

    colors().back   = style.backColor;
    colors().border = style.borderColor;

    m_Label.size(style.textSize);
    m_Label.colors().text = style.textColor;

    switch (style.captionAlignment)
    {
#define CS(p,x,y) case glv::Place::p: sx=x; sy=y; break;
        CS(TL, 0, 0  ) CS(TC, 0.5, 0  ) CS(TR, 1, 0)
        CS(CL, 0, 0.5) CS(CC, 0.5, 0.5) CS(CR, 1, 0.5)
        CS(BL, 0, 1  ) CS(BC, 0.5, 1  ) CS(BR, 1, 1)
#undef CS
    default:
        break;
    }

    m_Label.anchor(style.captionAlignment);
    m_Label.pos(style.captionAlignment, m_Frame.width() * sx, m_Frame.height() * sy);
}

void
Button::UpdateStyle()
{
    ApplyStyle(m_State == State::On ? m_StyleOn : m_StyleOff);
}

} // ~usb
} // ~nnt
