﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include "UsbDiag_UsbModel.h"

using namespace nn::usb;

namespace nnt {
namespace usb {

UsbModel g_Model;

UsbModel::UsbModel() NN_NOEXCEPT
    : m_PollCradle(false)
    , m_IsMenuOn(false)
    , m_pDetailView(nullptr)
    , m_ShowDetail(false)
    , m_TestMode(nn::usb::TestMode_Invalid)
    , m_DriveStrengthOffset(0)
    , m_IsOffsetSet(false)
{
    // N.B. Notifier class won't allocate enough slots for mHandlers, so we
    // must do it here. (Event_NumOfEvents > glv::Update::NumTypes)
    mHandlers = new std::vector<Handler>[Event_NumOfEvents];

    m_Pm.Initialize();
    m_Host.Initialize();
}

UsbModel::~UsbModel() NN_NOEXCEPT
{
    // mHandlers will be deleted by Notifier::~Notifier()

    m_Host.Finalize();
    m_Pm.Finalize();
}

void UsbModel::MonitorStart() NN_NOEXCEPT
{
    pd::InitializeCradle();
    pd::OpenCradleSession(&m_Cradle);
    m_CradlePdcHFwVersion = 0;

    m_Done = false;
    m_DataRole = UsbDataRole_Unknown;

    nn::os::CreateThread(&m_Thread,
                         ThreadEntry,
                         this,
                         m_ThreadStack,
                         sizeof(m_ThreadStack),
                         nn::os::DefaultThreadPriority);
    nn::os::StartThread(&m_Thread);
}

void UsbModel::MonitorStop() NN_NOEXCEPT
{
    m_Done = true;

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    pd::CloseCradleSession(&m_Cradle);
    pd::FinalizeCradle();
}

void UsbModel::Monitor() NN_NOEXCEPT
{
    nn::os::SystemEventType  interfaceEvent;
    nn::os::SystemEventType *pPowerEvent = m_Pm.GetPowerEvent();
    nn::os::SystemEventType *pDataEvent  = m_Pm.GetDataEvent();

    DeviceFilter filter = InvalidDeviceFilter;
    m_Host.CreateInterfaceAvailableEvent(&interfaceEvent,
                                         nn::os::EventClearMode_ManualClear,
                                         0,
                                         &filter);
    // initial poll
    UpdatePowerState();
    UpdateDataRole();

    while (!m_Done)
    {
        int signaled = nn::os::TimedWaitAny(nn::TimeSpan::FromMilliSeconds(1000),
                                            pPowerEvent,
                                            pDataEvent,
                                            &interfaceEvent);

        switch (signaled)
        {
        case 0:  // power event
            nn::os::ClearSystemEvent(pPowerEvent);
            UpdatePowerState();
            break;

        case 1:  // data event
            nn::os::ClearSystemEvent(pDataEvent);
            UpdateDataRole();
            break;

        case 2:  // interface event
            nn::os::ClearSystemEvent(&interfaceEvent);
            UpdateInterfaceList();
            break;

        case -1: // timeout
            // do nothing
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        UpdateInterfaceList();
        UpdateCradleState();
    }

    m_Host.DestroyInterfaceAvailableEvent(&interfaceEvent, 0);
}

bool UsbModel::IsLogEnabled(LogModule module)
{
    return m_Pm.GetDiagData(DiagData_LogModule) & (1 << module);
}

bool UsbModel::IsPollingCradle()
{
    return m_PollCradle;
}

void UsbModel::ToggleMenu()
{
    ToggleMenu(!m_IsMenuOn);
}

void UsbModel::ToggleMenu(bool isOn)
{
    if (m_IsMenuOn == isOn)
    {
        return;
    }

    if (isOn)
    {
        notify(Event_ShowMenu, nullptr);
    }
    else
    {
        notify(Event_HideMenu, nullptr);
    }

    m_IsMenuOn = isOn;
}

void UsbModel::ToggleLogEnabled(LogModule module, bool enabled)
{
    uint32_t value = m_Pm.GetDiagData(DiagData_LogModule);

    if (enabled)
    {
        value |= 1 << module;
    }
    else
    {
        value &= ~(1 << module);
    }

    m_Pm.SetDiagData(DiagData_LogModule, value);
}

void UsbModel::ToggleCradlePolling(bool enabled)
{
    m_PollCradle = enabled;
}

    UsbStringDescriptor languageStringDescriptor            = {4,   UsbDescriptorType_String, {0x0409}};
    UsbStringDescriptor manufacturerStringDescriptor        = {18,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
    UsbStringDescriptor productStringDescriptor             = {32,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'T', 'e', 's', 't'}};
    UsbStringDescriptor serialNumberStringDescriptor        = {26,  UsbDescriptorType_String, {'S', 'e', 'r', 'i', 'a', 'l', 'N', 'u', 'm', 'b', 'e', 'r'}};
    UsbStringDescriptor interfaceStringDescriptor           = {16,  UsbDescriptorType_String, {'U', 's', 'b', 'D', 'i', 'a', 'g'}};

    UsbDeviceDescriptor fullSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0110,                                                  // bcdUSB 1.1
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x40,                                                    // bMaxPacketSize0
        0x057e,                                                  // idVendor
        0x3001,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    UsbDeviceDescriptor highSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0200,                                                  // bcdUSB 2.0
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x40,                                                    // bMaxPacketSize0
        0x057e,                                                  // idVendor
        0x3001,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    UsbDeviceDescriptor superSpeedDeviceDescriptor =
    {
        ::nn::usb::UsbDescriptorSize_Device,                     // bLength
        ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
        0x0300,                                                  // bcdUSB 3.0
        0x00,                                                    // bDeviceClass
        0x00,                                                    // bDeviceSubClass
        0x00,                                                    // bDeviceProtocol
        0x09,                                                    // bMaxPacketSize0, SS 512 or 2^9
        0x057e,                                                  // idVendor
        0x3001,                                                  // idProduct
        0x0100,                                                  // bcdDevice
        0x01,                                                    // iManufacturer
        0x02,                                                    // iProduct
        0x03,                                                    // iSerialNumber
        0x01                                                     // bNumConfigurations
    };


    uint8_t binaryObjectStore[] =
    {
        0x05,                                       // bLength
        UsbDescriptorType_Bos,                      // bDescriptorType
        0x16,0x00,                                  // Length of this descriptor and all sub descriptors
        0x02,                                       // Number of device capability descriptors

        // USB 2.0 extension
        0x07,                                       // bLength
        UsbDescriptorType_DeviceCapability,         // bDescriptorType
        0x02,                                       // USB 2.0 extension capability type
        0x02,0x00,0x00,0x00,                        // Supported device level features: LPM support

        // SuperSpeed device capability
        0x0A,                                       // bLength
        UsbDescriptorType_DeviceCapability,         // bDescriptorType
        0x03,                                       // SuperSpeed device capability type
        0x00,                                       // Supported device level features
        0x0e,0x00,                                  // Speeds supported by the device : SS, HS
        0x03,                                       // Functionality support
        0x00,                                       // U1 Device Exit latency
        0x00,0x00                                   // U2 Device Exit latency
    };


    UsbInterfaceDescriptor ifDescriptor =
    {
        UsbDescriptorSize_Interface,    // bLength
        UsbDescriptorType_Interface,    // bDescriptorType
        0,                              // bInterfaceNumber
        0,                              // bAlternateSetting
        0,                              // bNumEndpoints
        0xff,                           // bInterfaceClass
        0xff,                           // bInterfaceSubClass
        0xff,                           // bInterfaceProtocol
        4,                              // iInterface
    };


void UsbModel::ToggleDummyDevice(bool enabled)
{
    if (enabled)
    {
        uint8_t stringIndex;

        m_Device.Initialize(ComplexId_Tegra21x);
        m_Device.ClearDeviceData();

        // string descriptors
        m_Device.AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor);
        m_Device.AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor);
        m_Device.AddUsbStringDescriptor(&stringIndex, &productStringDescriptor);
        m_Device.AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor);
        m_Device.AddUsbStringDescriptor(&stringIndex, &interfaceStringDescriptor);

        // device descriptors
        m_Device.SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   UsbDeviceSpeed_Full);
        m_Device.SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   UsbDeviceSpeed_High);
        m_Device.SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  UsbDeviceSpeed_Super);
        m_Device.SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore));

        m_Interface.Initialize(&m_Device, ifDescriptor.bInterfaceNumber);

        // For each speed, append descriptor in the order to appear in the
        // configuration descriptor stream, this allows any descriptors to appear
        // in the stream as needed :)
        m_Interface.AppendConfigurationData(UsbDeviceSpeed_Full, &ifDescriptor, sizeof(UsbInterfaceDescriptor));
        m_Interface.AppendConfigurationData(UsbDeviceSpeed_High, &ifDescriptor, sizeof(UsbInterfaceDescriptor));
        m_Interface.AppendConfigurationData(UsbDeviceSpeed_Super, &ifDescriptor, sizeof(UsbInterfaceDescriptor));

        m_Interface.Enable();
    }
    else
    {
        m_Interface.Disable();

        m_Interface.Finalize();
        m_Device.Finalize();
    }
}

void UsbModel::UpdatePowerState() NN_NOEXCEPT
{
    UsbPowerState state;

    m_Pm.GetPowerState(&state);

    if (memcmp(&state, &m_PowerState, sizeof(UsbPowerState)) != 0)
    {
        notify(Event_UpdatePowerState, &state);
        m_PowerState = state;
    }
}

void UsbModel::UpdateDataRole() NN_NOEXCEPT
{
    UsbDataRole role;

    m_Pm.GetDataRole(&role);

    if (role != m_DataRole)
    {
        notify(Event_UpdateDataRole, &role);
        m_DataRole = role;
    }
}

void UsbModel::UpdateCradleState() NN_NOEXCEPT
{
    nn::Result   result;
    pd::VdmPdcHFwVersion version;

    if (!m_PollCradle)
    {
        return;
    }

    result = pd::GetCradleVdo(reinterpret_cast<pd::Vdo*>(&version),
                              &m_Cradle,
                              pd::CradleVdmCommand_PdcHFwVersion);
    if (result.IsFailure())
    {
        version = 0;
    }

    if (version != m_CradlePdcHFwVersion)
    {
        notify(Event_UpdateCradleState, &version);
        m_CradlePdcHFwVersion = version;
    }
}

void UsbModel::UpdateInterfaceList() NN_NOEXCEPT
{
    int32_t count;
    DeviceFilter filter = InvalidDeviceFilter;

    memset(m_Buffer, 0, sizeof(m_Buffer));
    m_Host.QueryAllInterfaces(&count, m_Buffer, sizeof(m_Buffer), &filter);

    if (count == m_IfCount &&
        memcmp(m_QueryOutputs, m_Buffer, sizeof(m_Buffer[0]) * count) == 0)
    {
        // nothing changed
        return;
    }

    m_IfCount = count;
    memset(m_QueryOutputs, 0, sizeof(m_QueryOutputs));
    memcpy(m_QueryOutputs, m_Buffer, sizeof(m_Buffer[0]) * count);

    // parse the query output
    m_DeviceMap.clear();
    for (int i = 0; i < count; i++)
    {
        InterfaceProfile& ifProfile = m_Buffer[i].ifProfile;
        DeviceProfile& deviceProfile = m_Buffer[i].deviceProfile;

        DeviceUid uid = deviceProfile.deviceUid;

        if (m_DeviceMap.find(uid) == m_DeviceMap.end())
        {
            memcpy(m_DeviceMap[uid].name, deviceProfile.deviceDebugName, HsLimitMaxDebugNameSize);
            m_DeviceMap[uid].deviceDesc = deviceProfile.deviceDesc;
            m_DeviceMap[uid].configDesc = deviceProfile.cfgDesc;
        }

        Interface interface;
        interface.ifDesc = ifProfile.ifDesc;

        for (int i = 1; i < UsbLimitMaxEndpointPairCount; i++)
        {
            Endpoint ep;
            UsbEndpointDescriptor& epInDesc  = ifProfile.epInDesc[i];
            UsbEndpointDescriptor& epOutDesc = ifProfile.epOutDesc[i];

            if (epInDesc.bLength > 0)
            {
                ep.epDesc = epInDesc;
                interface.endpoints.push_back(ep);
            }

            if (epOutDesc.bLength > 0)
            {
                ep.epDesc = epOutDesc;
                interface.endpoints.push_back(ep);
            }
        }

        m_DeviceMap[uid].interfaces.push_back(interface);
    }

    notify(Event_UpdateDevice, &m_DeviceMap);
}

void UsbModel::UpdateDetail(glv::View& detailView) NN_NOEXCEPT
{
    notify(Event_UpdateDetail, &detailView);

    if (&detailView == m_pDetailView)
    {
        m_ShowDetail = !m_ShowDetail;
    }
    else
    {
        m_ShowDetail = true;
        m_pDetailView = &detailView;
    }

    if (m_ShowDetail)
    {
        notify(Event_ShowDetail, nullptr);
    }
    else
    {
        notify(Event_HideDetail, nullptr);
    }
}

void UsbModel::SetTestMode(nn::usb::TestMode mode)
{
    m_TestMode = mode;

    if (!m_IsOffsetSet || m_TestMode == nn::usb::TestMode_Invalid)
    {
        return;
    }

    uint32_t driveStrength;
    if (m_Host.SetTestMode(0, m_TestMode, m_DriveStrengthOffset, &driveStrength).IsSuccess())
    {
        notify(Event_UpdateDriveStrength, &driveStrength);
    }
}

void UsbModel::SetDriveStrengthOffset(int offset)
{
    m_DriveStrengthOffset = offset;
    m_IsOffsetSet         = true;
    SetTestMode(m_TestMode);
}

} // ~usb
} // ~nnt
