﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/util/util_FormatString.h>
#include <nn/usb/usb_Types.h>

#include "UsbDiag_Theme.h"
#include "UsbDiag_IfView.h"
#include "UsbDiag_EpView.h"
#include "UsbDiag_LabelView.h"

using namespace nn::usb;

namespace nnt {
namespace usb {

IfView::IfView(const Interface& interface) NN_NOEXCEPT
    : m_Interface(interface)
    , m_Detail("x-,<-,<<")
{
    disable(glv::Property::FocusToTop);

    CreateDetailView();

    nn::util::SNPrintf(
        m_String, sizeof(m_String),
        "Interface %d / %s",
        m_Interface.ifDesc.bInterfaceNumber,
        GetClassString(m_Interface.ifDesc.bInterfaceClass)
    );

    LabelView        *ifLabel = new LabelView(m_String, 10, 10, m_Detail);
    AdaptiveListView *epList  = new AdaptiveListView();

    for (auto& ep : m_Interface.endpoints)
    {
        *epList << new EpView(ep);
    }

    epList->Adapt(true);
    ifLabel->width(IfWidth - epList->width());

    *this << ifLabel << epList;

    Adapt(false);
}

IfView::~IfView() NN_NOEXCEPT
{
    // nothing special
}

void IfView::CreateDetailView() NN_NOEXCEPT
{
    const UsbInterfaceDescriptor& ifDesc = m_Interface.ifDesc;

    const char *title = "Interface Descriptor";
    const char *name =
        "bLength\n"
        "bDescriptorType\n"
        "bInterfaceNumber\n"
        "bAlternateSetting\n"
        "bNumEndpoints\n"
        "bInterfaceClass\n"
        "bInterfaceSubClass\n"
        "bInterfaceProtocol\n"
        "iInterface";

    nn::util::SNPrintf(
        m_String, sizeof(m_String),
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x",
        ifDesc.bLength,
        ifDesc.bDescriptorType,
        ifDesc.bInterfaceNumber,
        ifDesc.bAlternateSetting,
        ifDesc.bNumEndpoints,
        ifDesc.bInterfaceClass,
        ifDesc.bInterfaceSubClass,
        ifDesc.bInterfaceProtocol,
        ifDesc.iInterface
    );

    m_Detail << new glv::Label(title) << new glv::Divider()
             << new glv::Label(name) << new glv::Label(m_String);

    m_Detail.arrange();

    m_Detail.disable(glv::Property::HitTest);
}

const char* IfView::GetClassString(uint8_t code) NN_NOEXCEPT
{
    struct ClassStringMap {
        uint8_t code;
        const char *string;
    } theMaps[] = {
#define MAP(type) {nn::usb::UsbClass_##type, #type}
        MAP(PerInterface),
        MAP(Audio),
        MAP(Comm),
        MAP(Hid),
        MAP(Physical),
        MAP(StillImage),
        MAP(Printer),
        MAP(MassStorage),
        MAP(Hub),
        MAP(CdcData),
        MAP(SmartCard),
        MAP(ContentSecurity),
        MAP(Video),
        MAP(Healthcare),
        MAP(AudioVideo),
        MAP(Billboard),
        MAP(Diagnostic),
        MAP(WirelessController),
        MAP(Misc),
        MAP(AppSpecific),
        MAP(VendorSpecific),
#undef MAP
    };

    for (auto& aMap : theMaps)
    {
        if (aMap.code == code)
        {
            return aMap.string;
        }
    }

    return "UnknownClass";
}


} // ~usb
} // ~nnt
