﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/usb/usb_Pm.h>

#include "UsbDiag_Theme.h"
#include "UsbDiag_HomeView.h"
#include "UsbDiag_UsbModel.h"

namespace nnt {
namespace usb {

HomeView::HomeView(Stage& stage) NN_NOEXCEPT
    : Scene(stage, "Home")
    , m_DeviceList(DeviceListWidth, height() - 2)
    , m_Detail(width() - ContentGap - DeviceListWidth - 2, height() - 2)
    , m_Message("Cable Removed")
    , m_ShowDetail(false)
{
    disable(glv::Property::DrawBack   |
            glv::Property::Animate);

    *this << m_DeviceList << m_Detail << m_Message;

    // disable device list by default
    m_DeviceList.disable(glv::Property::Visible);

    // disable detail view by default
    m_Detail.enable(glv::Property::CropSelf);
    m_Detail.disable(glv::Property::Visible);

    // enable state message by default
    m_Message.enable(glv::Property::Visible);
    m_Message.size(NoticeFontSize);
    m_Message.pos(glv::Place::CC, width() / 2, height() / 2);

    enable(glv::Property::DrawBorder);

    HideDetail();

    g_Model.attach(OnDataRoleChange, Event_UpdateDataRole, this);
    g_Model.attach(OnDeviceUpdate,   Event_UpdateDevice,   this);
    g_Model.attach(OnShowDetail,     Event_ShowDetail,     this);
    g_Model.attach(OnHideDetail,     Event_HideDetail,     this);
}

HomeView::~HomeView() NN_NOEXCEPT
{
    g_Model.detach(OnHideDetail,     Event_HideDetail,     this);
    g_Model.detach(OnShowDetail,     Event_ShowDetail,     this);
    g_Model.detach(OnDeviceUpdate,   Event_UpdateDevice,   this);
    g_Model.detach(OnDataRoleChange, Event_UpdateDataRole, this);
}

void HomeView::onAnimate(double dsec)
{
    if (m_ShowDetail) // show detail
    {
        if (m_DeviceList.width() <= DeviceListWidth)
        {
            disable(glv::Property::Animate);
            ShowDetail();
        }
        else
        {
            // stretch DeviceListView
            m_DeviceList.width(
                std::max(
                    double(DeviceListWidth),
                    m_DeviceList.width() - AnimationSpeed * dsec
                )
            );

            m_Detail.posRightOf(m_DeviceList, ContentGap);
        }
    }
    else // hide detail
    {
        if (m_DeviceList.width() >= width())
        {
            disable(glv::Property::Animate);
            HideDetail();
        }
        else
        {
            // stretch DeviceListView
            m_DeviceList.width(
                std::min(
                    double(width()),
                    m_DeviceList.width() + AnimationSpeed * dsec
                )
            );

            m_Detail.posRightOf(m_DeviceList, ContentGap);
        }
    }
}

void HomeView::ShowDetail() NN_NOEXCEPT
{
    m_DeviceList.pos(glv::Place::TL, 1, 1);
    m_DeviceList.width(DeviceListWidth);
    m_Detail.posRightOf(m_DeviceList, ContentGap);
}

void HomeView::HideDetail() NN_NOEXCEPT
{
    m_DeviceList.pos(glv::Place::TL, 1, 1);
    m_DeviceList.width(width() - 2);
    m_Detail.posRightOf(m_DeviceList, ContentGap);
}

void HomeView::OnDataRoleChange(const glv::Notification& n) NN_NOEXCEPT
{
    const char *str;
    HomeView *self = n.receiver<HomeView>();
    const nn::usb::UsbDataRole& role = *n.data<nn::usb::UsbDataRole>();

    switch (role)
    {
    case nn::usb::UsbDataRole_Unknown:
        str = "Cable Removed";
        break;

    case nn::usb::UsbDataRole_Host:
        str = "No Device Connected";
        break;

    case nn::usb::UsbDataRole_Device:
        str = "Device Mode";
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    // make sure detail view is hidden
    self->HideDetail();

    // update state message
    self->m_Message.setValue(str);
    self->m_Message.pos(glv::Place::CC, self->width() / 2, self->height() / 2);
}

void HomeView::OnDeviceUpdate(const glv::Notification& n) NN_NOEXCEPT
{
    HomeView *self = n.receiver<HomeView>();
    const DeviceMap& deviceMap = *n.data<DeviceMap>();

    if (deviceMap.size() == 0)
    {
        // disable detail and device list view
        self->m_Detail.disable(glv::Property::Visible);
        self->m_DeviceList.disable(glv::Property::Visible);

        // enable status message
        self->m_Message.enable(glv::Property::Visible);
        self->enable(glv::Property::DrawBorder);
    }
    else
    {
        // enable detail and device list view
        self->m_Detail.enable(glv::Property::Visible);
        self->m_DeviceList.enable(glv::Property::Visible);

        // disable status message
        self->m_Message.disable(glv::Property::Visible);
        self->disable(glv::Property::DrawBorder);

        // hide the detail view by default
        self->HideDetail();
    }
}

void HomeView::OnShowDetail(const glv::Notification& n) NN_NOEXCEPT
{
    HomeView *self = n.receiver<HomeView>();

    self->enable(glv::Property::Animate);

    self->m_ShowDetail = true;
}

void HomeView::OnHideDetail(const glv::Notification& n) NN_NOEXCEPT
{
    HomeView *self = n.receiver<HomeView>();

    self->enable(glv::Property::Animate);

    self->m_ShowDetail = false;
}

} // ~usb
} // ~nnt
