﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "UsbDiag_Theme.h"
#include "UsbDiag_UsbModel.h"
#include "UsbDiag_FooterView.h"

namespace nnt {
namespace usb {

FooterView::FooterView(const unsigned width, const unsigned height) NN_NOEXCEPT
    : glv::View(glv::Rect(width, height))
    , m_UsbModeLabel("Data: Detached | Powr: Detached")
    , m_HeartBeat(height - FooterPaddingY * 2)
    , m_PowerState{nn::usb::UsbPowerRole_Unknown}
    , m_DataRole(nn::usb::UsbDataRole_Unknown)
    , m_CradleVersion(0)
{
    enable(glv::Property::HitTest);

    disable(glv::Property::FocusHighlight |
            glv::Property::FocusToTop);

    m_HeartBeat.pos(glv::Place::TR,
                    glv::Rect::width() - FooterPaddingX,
                    FooterPaddingY);

    UpdateLabel();

    *this << m_HeartBeat << m_UsbModeLabel;

    g_Model.attach(OnPowerStateChange,  Event_UpdatePowerState,  this);
    g_Model.attach(OnDataRoleChange,    Event_UpdateDataRole,    this);
    g_Model.attach(OnCradleStateChange, Event_UpdateCradleState, this);
}

FooterView::~FooterView() NN_NOEXCEPT
{
    g_Model.detach(OnCradleStateChange, Event_UpdateCradleState, this);
    g_Model.detach(OnDataRoleChange,    Event_UpdateDataRole,    this);
    g_Model.detach(OnPowerStateChange,  Event_UpdatePowerState,  this);
}


bool FooterView::onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
{
    if (e == glv::Event::MouseUp)
    {
        g_Model.ToggleMenu();
        return false;
    }
    else
    {
        return true;
    }
}

void FooterView::UpdateLabel() NN_NOEXCEPT
{
    const char *str = const_cast<const char*>(m_Buffer);
    struct {
        uint16_t version;
        const char *name;
    } cradleVersionMap[] = {
        { nn::usb::pd::CradlePdcHFwRevisionDp1,   "DP-1"   },
        { nn::usb::pd::CradlePdcHFwRevisionEp2_2, "EP-2-2" },
        { nn::usb::pd::CradlePdcHFwRevisionEp2,   "EP-2"   },
        { nn::usb::pd::CradlePdcHFwRevisionEp1,   "EP-1"   },
        { nn::usb::pd::CradlePdcHFwRevisionFp2,   "FP-2"   },
    };

    // Cradle information
    const char *cradleName = nullptr;
    for (int i = 0; i < sizeof(cradleVersionMap) / sizeof(cradleVersionMap[0]); i++)
    {
        if (cradleVersionMap[i].version == m_CradleVersion)
        {
            cradleName = cradleVersionMap[i].name;
            break;
        }
    }

    // Data role information
    const char *dataRole = nullptr;
    switch (m_DataRole)
    {
    case nn::usb::UsbDataRole_Unknown:
        dataRole = "Detached";
        break;

    case nn::usb::UsbDataRole_Host:
        dataRole = "Host";
        break;

    case nn::usb::UsbDataRole_Device:
        dataRole = "Device";
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    // Power state information
    char powerRole[64];
    switch (m_PowerState.role)
    {
    case nn::usb::UsbPowerRole_Unknown:
        nn::util::SNPrintf(powerRole, sizeof(powerRole),
                           "Detached");
        break;

    case nn::usb::UsbPowerRole_Sink:
        if (m_PowerState.charger == nn::usb::UsbChargerType_Pd)
        {
            nn::util::SNPrintf(powerRole, sizeof(powerRole),
                               "Sink, PD");
        }
        else
        {
            const char *chargerType = nullptr;
            switch (m_PowerState.charger)
            {
            case nn::usb::UsbChargerType_TypeC15:
            case nn::usb::UsbChargerType_TypeC30:
                chargerType = "TypeC";
                break;

            case nn::usb::UsbChargerType_Dcp:
                chargerType = "DCP";
                break;

            case nn::usb::UsbChargerType_Cdp:
                chargerType = "CDP";
                break;

            case nn::usb::UsbChargerType_Sdp:
                chargerType = "SDP";
                break;

            case nn::usb::UsbChargerType_Apple500ma:
                chargerType = "A500";
                break;

            case nn::usb::UsbChargerType_Apple1000ma:
                chargerType = "A1000";
                break;

            case nn::usb::UsbChargerType_Apple2000ma:
                chargerType = "A2000";
                break;

            default:
                chargerType = "Unknown";
                break;
            }
            nn::util::SNPrintf(powerRole, sizeof(powerRole),
                               "Sink, %s, %dmV, %dmA",
                               chargerType,
                               m_PowerState.voltage,
                               m_PowerState.current);
        }
        break;

    case nn::usb::UsbPowerRole_Source:
        nn::util::SNPrintf(powerRole, sizeof(powerRole),
                           "Source");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    if (m_CradleVersion != 0)
    {
        nn::util::SNPrintf(m_Buffer, sizeof(m_Buffer),
                           "Data: %s | Power: %s | Cradle Fw: %X (%s)",
                           dataRole,
                           powerRole,
                           m_CradleVersion,
                           cradleName ? cradleName : "???");
    }
    else
    {
        nn::util::SNPrintf(m_Buffer, sizeof(m_Buffer),
                           "Data: %s | Power: %s",
                           dataRole,
                           powerRole);
    }

    m_UsbModeLabel.size(NormalFontSize);
    m_UsbModeLabel.setValue(str);
    m_UsbModeLabel.pos(glv::Place::TL, FooterPaddingX, FooterPaddingY);
} // NOLINT(impl/function_size)

void FooterView::OnPowerStateChange(const glv::Notification& n) NN_NOEXCEPT
{
    FooterView *self = n.receiver<FooterView>();

    self->m_PowerState = *n.data<nn::usb::UsbPowerState>();
    self->UpdateLabel();
}

void FooterView::OnDataRoleChange(const glv::Notification& n) NN_NOEXCEPT
{
    FooterView *self = n.receiver<FooterView>();

    self->m_DataRole = *n.data<nn::usb::UsbDataRole>();
    self->UpdateLabel();
}

void FooterView::OnCradleStateChange(const glv::Notification& n) NN_NOEXCEPT
{
    FooterView *self = n.receiver<FooterView>();

    self->m_CradleVersion = *n.data<nn::usb::pd::VdmPdcHFwVersion>();
    self->UpdateLabel();
}

} // ~usb
} // ~nnt
