﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nns {
namespace iodevices {

struct MicData
{
    MicData()
        : appendEvent(nn::os::EventClearMode_AutoClear)
        , mutex(false)
        , frameSize(0)
        , readIndex(0)
        , writeIndex(0)
        , readyFrames(0)
        , numFrames(0)
        , rmsIndex(0)
    {
        memset(rms, 0, sizeof(rms) / sizeof(rms[0]));
    }

    void Reset()
    {
        std::lock_guard<nn::os::Mutex> lock(mutex);
        readyFrames = readIndex = writeIndex = 0;
        count = 0;
    }


    void SetFrameSize(size_t _frameSize)
    {
        frameSize = _frameSize;
        numFrames = static_cast<int>(sizeof(buffer) / frameSize);
    }

    void PutFrame(void* data, size_t size)
    {
        NN_UNUSED(size);
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            memcpy(&buffer[writeIndex * frameSize], data, frameSize);
            writeIndex = ++writeIndex % numFrames;
            ++readyFrames;


            auto p = static_cast<int16_t*>(data);
            auto sampleCount = static_cast<int>(frameSize / sizeof(int16_t));
            float sum = 0;
            for (auto i = 0; i < sampleCount; ++i)
            {
                auto sample = p[i] / 32768.0f;
                sum += sample * sample;
            }
            sum /= sampleCount;
            rms[rmsIndex] = std::sqrtf(sum);
            if (++rmsIndex >= RmsCountMax)
            {
                rmsIndex = 0;
            }
        }
        appendEvent.Signal();
    }

    void GetFrame(void* data, size_t size)
    {
        NN_UNUSED(size);
        if (readyFrames == 0)
        {
            appendEvent.Wait();
        }
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            memcpy(data, &buffer[readIndex * frameSize], frameSize);
            memset(&buffer[readIndex * frameSize], 0, frameSize);
            readIndex = ++readIndex % numFrames;
            --readyFrames;
        }
    }

    void GetFrameAdd(void* data, size_t size)
    {
        while (inUse && readyFrames <= 0)
        {
            appendEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(5));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            int16_t* in = reinterpret_cast<int16_t*>(&buffer[readIndex * frameSize]);
            int16_t* out = reinterpret_cast<int16_t*>(data);
            for (size_t i = 0; i < size / 2; ++i)
            {
                out[i] += in[i];
            }
            readIndex = ++readIndex % numFrames;
            --readyFrames;
        }
    }

    nn::os::Event appendEvent;
    nn::os::Mutex mutex;
    size_t frameSize;
    int readIndex;
    int writeIndex;
    int readyFrames;
    int numFrames;
    bool inUse;
    int count;
    char buffer[4096 * 8];
    static const int RmsCountMax = 16;
    float rms[RmsCountMax];
    int rmsIndex;
};

} // namespace iodevices
} // namespace nns
