﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Methods/Fx3.h"

namespace nnt {
namespace usb {
namespace hs {


const Fx3DeviceMode deviceModes[] =
{
    {FX3_DEVICE_FULL_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_2},
    {FX3_DEVICE_HIGH_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_2},
    {FX3_DEVICE_SUPER_SPEED, FX3_CTRL_MPS_512, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_2},
};

// const int bulkTestValues[] =
// {
//     1,
//     2,
//     3,
//     511,
//     512,
//     513,
//     4095,
//     4096,
//     4097,
//     20 * 1024 - 1,
//     20 * 1024,
//     20 * 1024 + 1,
//     32 * 1024 - 1,
//     32 * 1024,
//     32 * 1024 + 1,
//     64 * 1024 - 1,
//     64 * 1024
// };

const int bulkTestValues[] =
{
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
    2 * 1024 * 1024,
};

enum
{
    TRANSFER_BUFFER_SIZE        = 2 * 1024 * 1024,
    BULK_TEST_VALUES_COUNT      = 10,
    THREAD_STACK_SIZE           = 4096,
};

static NN_ALIGNAS(4096) uint8_t g_TxBuffer[TRANSFER_BUFFER_SIZE];
static NN_ALIGNAS(4096) uint8_t g_RxBuffer[TRANSFER_BUFFER_SIZE];
static NN_ALIGNAS(4096) uint8_t g_TxThreadStack[THREAD_STACK_SIZE];
static NN_ALIGNAS(4096) uint8_t g_RxThreadStack[THREAD_STACK_SIZE];

class HsFx3BulkThroughtputTest : public ::testing::TestWithParam<Fx3DeviceMode>{};

INSTANTIATE_TEST_CASE_P(DeviceModes, HsFx3BulkThroughtputTest, ::testing::ValuesIn(deviceModes));

///////////////////////////////////////////////////////////////////////////////
TEST_P(HsFx3BulkThroughtputTest, BulkLoopback)
{
    ASSERT_TRUE(IsFx3Initialized());

    Fx3DeviceMode fx3DeviceMode;

    fx3DeviceMode.connectSpeed              = GetParam().connectSpeed;
    fx3DeviceMode.maxPacketSize             = GetParam().maxPacketSize;
    fx3DeviceMode.interfaceBitmap           = GetParam().interfaceBitmap;

    Fx3SetDeviceMode(&fx3DeviceMode, WAIT_SECONDS_FOR_ATTACH);

    uint8_t numAltSetting = 0;
    switch (GetParam().connectSpeed)
    {
        case FX3_DEVICE_FULL_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_FS;   break;
        case FX3_DEVICE_HIGH_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_HS;   break;
        case FX3_DEVICE_SUPER_SPEED:    numAltSetting = FX3_BULK_ALT_SETTINGS_SS;   break;
        default:                                                                    break;
    }

    for (uint8_t setting = 0; setting < numAltSetting; setting++)
    {
        Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_2, setting + 1);

        Fx3LoopbackParameters fx3LoopbackParameters;

        std::memset(&fx3LoopbackParameters, 0, sizeof(Fx3LoopbackParameters));

        fx3LoopbackParameters.endpointAddressTx                 = 0x02;
        fx3LoopbackParameters.pTxData                           = g_TxBuffer;
        fx3LoopbackParameters.pTxThreadStack                    = g_TxThreadStack;
        fx3LoopbackParameters.txThreadStackSize                 = THREAD_STACK_SIZE;

        fx3LoopbackParameters.endpointAddressRx                 = 0x82;
        fx3LoopbackParameters.pRxData                           = g_RxBuffer;
        fx3LoopbackParameters.pRxThreadStack                    = g_RxThreadStack;
        fx3LoopbackParameters.rxThreadStackSize                 = THREAD_STACK_SIZE;

        fx3LoopbackParameters.fx3TestDataParams.endpointAddress = FX3_ENDPOINT_BULK_2_OUT;
        fx3LoopbackParameters.fx3TestDataParams.entries         = BULK_TEST_VALUES_COUNT;
        fx3LoopbackParameters.fx3TestDataParams.dataSeed        = 0xffffeeee;

        // For CI graph labels
        fx3LoopbackParameters.speed             = GetParam().connectSpeed;
        fx3LoopbackParameters.altSetting        = setting;

        for (int i = 0; i < BULK_TEST_VALUES_COUNT; i++)
        {
            fx3LoopbackParameters.fx3TestDataParams.params[i].requestBytes   = bulkTestValues[i];
            fx3LoopbackParameters.fx3TestDataParams.params[i].transferBytes  = bulkTestValues[i];
        }

        Fx3LoopbackData(&fx3LoopbackParameters);
    }

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_2, 0);
}

} // hs
} // usb
} // nnt

