﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Methods/Fx3.h"

namespace nnt {
namespace usb {
namespace hs {

const Fx3DeviceMode deviceModes[] =
{
    {FX3_DEVICE_FULL_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
    {FX3_DEVICE_HIGH_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
    {FX3_DEVICE_SUPER_SPEED, FX3_CTRL_MPS_512, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
};

enum
{
    TRANSFER_BUFFER_SIZE        = 512,
};

static NN_ALIGNAS(4096) uint8_t g_Buffer[TRANSFER_BUFFER_SIZE];

static void PerformBulkStallTest(uint8_t altSetting)
{
    Fx3TestDataRarameters   fx3TestDataParameters;

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_1, altSetting);

    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress           = FX3_ENDPOINT_BULK_1_IN;
    fx3TestDataParameters.entries                   = 1;
    fx3TestDataParameters.dataSeed                  = 0xffffeeee;
    fx3TestDataParameters.params[0].requestBytes    = TRANSFER_BUFFER_SIZE;
    fx3TestDataParameters.params[0].transferBytes   = 0;    // this causes FX3 FW to stall

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    Fx3ClrStall(FX3_ENDPOINT_BULK_1_IN);    // after clearing stall, transfers should work again

    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress           = FX3_ENDPOINT_BULK_1_IN;
    fx3TestDataParameters.entries                   = 1;
    fx3TestDataParameters.dataSeed                  = 0xffffeeee;
    fx3TestDataParameters.params[0].requestBytes    = TRANSFER_BUFFER_SIZE;
    fx3TestDataParameters.params[0].transferBytes   = TRANSFER_BUFFER_SIZE;

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress           = FX3_ENDPOINT_BULK_1_OUT;
    fx3TestDataParameters.entries                   = 1;
    fx3TestDataParameters.dataSeed                  = 0xffffeeee;
    fx3TestDataParameters.params[0].requestBytes    = TRANSFER_BUFFER_SIZE;
    fx3TestDataParameters.params[0].transferBytes   = 0;    // this causes FX3 FW to stall

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    Fx3ClrStall(FX3_ENDPOINT_BULK_1_OUT);   // after clearing stall, transfers should work again

    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress           = FX3_ENDPOINT_BULK_1_OUT;
    fx3TestDataParameters.entries                   = 1;
    fx3TestDataParameters.dataSeed                  = 0xffffeeee;
    fx3TestDataParameters.params[0].requestBytes    = TRANSFER_BUFFER_SIZE;
    fx3TestDataParameters.params[0].transferBytes   = TRANSFER_BUFFER_SIZE;

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);
}

class HsFx3BulkStallTest : public ::testing::TestWithParam<Fx3DeviceMode>{};

INSTANTIATE_TEST_CASE_P(DeviceModes, HsFx3BulkStallTest, ::testing::ValuesIn(deviceModes));

///////////////////////////////////////////////////////////////////////////////
TEST_P(HsFx3BulkStallTest, AllAltSettings)
{
    ASSERT_TRUE(IsFx3Initialized());

    Fx3DeviceMode           fx3DeviceMode;

    fx3DeviceMode.connectSpeed              = GetParam().connectSpeed;
    fx3DeviceMode.maxPacketSize             = GetParam().maxPacketSize;
    fx3DeviceMode.interfaceBitmap           = GetParam().interfaceBitmap;

    Fx3SetDeviceMode(&fx3DeviceMode, WAIT_SECONDS_FOR_ATTACH);

    uint8_t numAltSetting = 0;
    switch (GetParam().connectSpeed)
    {
        case FX3_DEVICE_FULL_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_FS;   break;
        case FX3_DEVICE_HIGH_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_HS;   break;
        case FX3_DEVICE_SUPER_SPEED:    numAltSetting = FX3_BULK_ALT_SETTINGS_SS;   break;
        default:                                                                    break;
    }

    for (uint8_t setting = 1; setting <= numAltSetting; setting++)
    {
        PerformBulkStallTest(setting);
    }

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_1, 0);
}

} // hs
} // usb
} // nnt

