﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <cstdio>
#include <thread>
#include <future>

#include "DebugMonitor.h"

namespace nnt { namespace usb {


namespace {

const int ReadTimeoutInMs = 250;

HANDLE s_ComPortHandle = NULL;


HANDLE OpenComPort(LPCTSTR name)
{
    COMMTIMEOUTS timeouts;
    DCB dcb;
    HANDLE h = INVALID_HANDLE_VALUE;

    h = CreateFile(name, 0x80000000L,  0, 0, OPEN_EXISTING, FILE_FLAG_NO_BUFFERING | FILE_FLAG_WRITE_THROUGH, 0);
    if (h == INVALID_HANDLE_VALUE)
    {
        printf("FX3 DebugMonitor: Create File Failed\n");
        goto error;
    }

    if (!ClearCommError(h, NULL, NULL))
    {
        printf("FX3 DebugMonitor: Clear Comm Failed\n");
        goto error;
    }

    if (!SetupComm(h, 512, 512))
    {
        printf("FX3 DebugMonitor: Setup Comm Failed\n");
        goto error;
    }

    if (!PurgeComm(h, PURGE_TXABORT | PURGE_RXABORT |
                      PURGE_TXCLEAR | PURGE_RXCLEAR))
    {
        printf("FX3 DebugMonitor: Purge Comm Failed\n");
        goto error;
    }

    memset(&timeouts, 0, sizeof(timeouts));
    timeouts.ReadTotalTimeoutConstant = ReadTimeoutInMs;
    if (!SetCommTimeouts(h, &timeouts))
    {
        printf("FX3 DebugMonitor: Set Comm Timeout Failed\n");
        goto error;
    }

    memset(&dcb, 0, sizeof(dcb));
    dcb.DCBlength = sizeof(dcb);
    dcb.BaudRate = 115200;
    dcb.fBinary = TRUE;
    dcb.Parity = NOPARITY;
    dcb.ByteSize = 8;
    dcb.StopBits = ONESTOPBIT;

    if (!SetCommState(h, &dcb))
    {
        goto error;
    }

    if (!SetCommMask(h, 0))
    {
        goto error;
    }

    return h;
error:
    printf("FX3 DebugMonitor: Failed to open COM port %ls\n", name);
    if( h != INVALID_HANDLE_VALUE)
    {
        CloseHandle(h);
    }
    return NULL;
}

int DoReadLoop()
{
    HANDLE handle;
    int status = 0;

    while ((handle = s_ComPortHandle) !=NULL)
    {
        char buffer[81];
        DWORD readSize = 0;
        memset(buffer, 0, sizeof(buffer));
        if (!ReadFile(handle, buffer, sizeof(buffer) - 1, &readSize, NULL))
        {
            DWORD error = GetLastError();
            status = -static_cast<int>(error);
            switch (error)
            {
            case ERROR_FILE_NOT_FOUND:
                printf("FX3 DebugMonitor: ERROR_FILE_NOT_FOUND.\n");
                break;
            case ERROR_ACCESS_DENIED:
                printf("FX3 DebugMonitor: ERROR_ACCESS_DENIED.\n");
                break;
            default:
                printf("FX3 DebugMonitor: Error code: %lu.\n", error);
                break;
            }
            break;
        }

        if(readSize != 0)
        {
            printf("%s", buffer);
            fflush(stdout);
        }
    }

    return status;
}


} // namespace {

int InitializeDebugMonitor()
{
    int status = 0;

    printf("FX3 DebugMonitor: Initializing...\n");

    if((s_ComPortHandle = OpenComPort(TEXT("\\\\.\\COM256"))) == NULL)
    {
        status = -1;
    }

    return status;
}

int DoDebugMonitor()
{
    int status = 0;

    printf("FX3 DebugMonitor: Starting read loop.\n");

    if(s_ComPortHandle != NULL)
    {
        status = DoReadLoop();
    }

    printf("FX3 DebugMonitor: Read loop exited with, status %d.\n", status);

    return status;
}

int FinalizeDebugMonitor()
{
    if(s_ComPortHandle != NULL)
    {
        CloseHandle(s_ComPortHandle);
        s_ComPortHandle = NULL;
    }

    return 0;
}

}} // namespace nnt { namespace usb {
