﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init/init_Malloc.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/nn_Assert.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include "../Fx3Methods/Fx3.h"
#include "Fx3MessageLogger.h"

namespace nnt {
namespace usb {
namespace hs {

const size_t g_HeapSize = 16 * 1024 * 1024;

class UsbHsFx3TestEnvironment : public ::testing::Environment
{
public:

    virtual void SetUp() override
    {
        Fx3Initialize(WAIT_SECONDS_FOR_ATTACH);
        FX3_LOG("Initialized UsbHsFx3TestEnvironment.\n");
        if (IsFx3Initialized())
        {
            Fx3CheckFirmwareVersion();
        }
    }

    virtual void TearDown() override
    {
        Fx3DeviceMode fx3DeviceMode;

        fx3DeviceMode.connectSpeed      = FX3_DEVICE_FULL_SPEED;
        fx3DeviceMode.maxPacketSize     = FX3_CTRL_MPS_64;
        fx3DeviceMode.interfaceBitmap   = FX3_INTERFACE_CTRL;

        if (IsFx3Initialized())
        {
            Fx3Finalize();
            FX3_LOG("Finalized UsbHsFx3TestEnvironment.\n");
        }
        else
        {
            FX3_LOG("Fx3 Failed to Finalize. Never Initialized.\n");
        }
    }
};

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(g_HeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", g_HeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, g_HeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new UsbHsFx3TestEnvironment);
}

extern "C" void nnMain()
{
    int    argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new Fx3MessageLogger());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    const int exitCode = RUN_ALL_TESTS();

    ::nnt::Exit(exitCode);
}

} // hs
} // usb
} // nnt
