﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_UserExceptionHandler.h>

#include <nnt.h>

#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_Device.h>

using namespace nn::usb;

namespace nnt {
namespace usb {
namespace ds {

DsClient    dsClient;
DsInterface dsInterface;
DsEndpoint  dsEndpoint[2];
UsbState    state;

UsbStringDescriptor languageStringDescriptor            = {4,   UsbDescriptorType_String, {0x0409}};
UsbStringDescriptor manufacturerStringDescriptor        = {18,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
UsbStringDescriptor productStringDescriptor             = {32,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'T', 'e', 's', 't'}};
UsbStringDescriptor serialNumberStringDescriptor        = {26,  UsbDescriptorType_String, {'S', 'e', 'r', 'i', 'a', 'l', 'N', 'u', 'm', 'b', 'e', 'r'}};
UsbStringDescriptor interfaceStringDescriptor           = {16,  UsbDescriptorType_String, {'U', 's', 'b', 'D', 'i', 'a', 'g'}};

UsbDeviceDescriptor fullSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0110,                                                  // bcdUSB 1.1
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


UsbDeviceDescriptor highSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0200,                                                  // bcdUSB 2.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


UsbDeviceDescriptor superSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0300,                                                  // bcdUSB 3.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x09,                                                    // bMaxPacketSize0, SS 512 or 2^9
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


uint8_t binaryObjectStore[] =
{
    0x05,                                       // bLength
    UsbDescriptorType_Bos,                      // bDescriptorType
    0x16,0x00,                                  // Length of this descriptor and all sub descriptors
    0x02,                                       // Number of device capability descriptors

    // USB 2.0 extension
    0x07,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x02,                                       // USB 2.0 extension capability type
    0x02,0x00,0x00,0x00,                        // Supported device level features: LPM support

    // SuperSpeed device capability
    0x0A,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x03,                                       // SuperSpeed device capability type
    0x00,                                       // Supported device level features
    0x0e,0x00,                                  // Speeds supported by the device : SS, HS
    0x03,                                       // Functionality support
    0x00,                                       // U1 Device Exit latency
    0x00,0x00                                   // U2 Device Exit latency
};


UsbInterfaceDescriptor    usbInterfaceDescriptor = {
    UsbDescriptorSize_Interface,                // bLength
    UsbDescriptorType_Interface,                // bDescriptorType
    0,                                          // bInterfaceNumber
    0,                                          // bAlternateSetting
    2,                                          // bNumEndpoints
    0xff,                                       // bInterfaceClass
    0xff,                                       // bInterfaceSubClass
    0xff,                                       // bInterfaceProtocol
    4,                                          // iInterface
};

UsbEndpointDescriptor     fullSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
};


UsbEndpointDescriptor     highSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        512,                                    // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        512,                                    // wMaxPacketSize
        0                                       // bInterval
    },
};


UsbEndpointDescriptor     superSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
};


// Use this endpoint companion descriptor for both endpoints :)
UsbEndpointCompanionDescriptor superSpeedUsbEndpointCompanionDescriptor =
{
    0x06,                                       // bLength
    UsbDescriptorType_EndpointCompanion,        // bDescriptorType
    15,                                         // bMaxBurst
    0,                                          // bmAttributes
    0                                           // wBytesPerInterval
};


NN_USB_DMA_ALIGN uint8_t readBuffer[4 * 1024 * 1024];

const int TRB_DATA_SIZE     = 64 * 1024;
const int BULK_TRBS_IN_RING = 64;

nn::Result SetupDevice()
{
    NNT_EXPECT_RESULT_SUCCESS(dsClient.Initialize(ComplexId_Tegra21x));

    uint8_t stringIndex;
    NNT_EXPECT_RESULT_SUCCESS(dsClient.AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.AddUsbStringDescriptor(&stringIndex, &productStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.AddUsbStringDescriptor(&stringIndex, &interfaceStringDescriptor));

    NNT_EXPECT_RESULT_SUCCESS(dsClient.SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   UsbDeviceSpeed_Full));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   UsbDeviceSpeed_High));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  UsbDeviceSpeed_Super));
    NNT_EXPECT_RESULT_SUCCESS(dsClient.SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore)));

    NNT_EXPECT_RESULT_SUCCESS(dsInterface.Initialize(&dsClient, usbInterfaceDescriptor.bInterfaceNumber));

    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Full, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Full, &fullSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Full, &fullSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_High, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_High, &highSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_High, &highSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Super, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointCompanionDescriptor, sizeof(UsbEndpointCompanionDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointCompanionDescriptor, sizeof(UsbEndpointCompanionDescriptor)));

    NNT_EXPECT_RESULT_SUCCESS(dsEndpoint[0].Initialize(&dsInterface, 0x81));
    NNT_EXPECT_RESULT_SUCCESS(dsEndpoint[1].Initialize(&dsInterface, 0x01));

    NNT_EXPECT_RESULT_SUCCESS(dsInterface.Enable());
    NNT_EXPECT_RESULT_SUCCESS(dsClient.EnableDevice());

    // Wait for attach
    nn::os::SystemEventType *pStateChangeEvent = dsClient.GetStateChangeEvent();

    NN_LOG("Device enabled, waiting for attach\n");

    while (true)
    {
        nn::os::WaitSystemEvent(pStateChangeEvent);
        nn::os::ClearSystemEvent(pStateChangeEvent);

        UsbState state;

        NNT_EXPECT_RESULT_SUCCESS(dsClient.GetState(&state));

        if (state == UsbState_Configured)
        {
            break;
        }
    }

    NN_LOG("Device attached and configuered by host\n");

    return nn::ResultSuccess();
}


nn::Result TeardownDevice()
{
    NNT_EXPECT_RESULT_SUCCESS(dsClient.DisableDevice());
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.Disable());

    NNT_EXPECT_RESULT_SUCCESS(dsEndpoint[1].Finalize());
    NNT_EXPECT_RESULT_SUCCESS(dsEndpoint[0].Finalize());
    NNT_EXPECT_RESULT_SUCCESS(dsInterface.Finalize());
    NNT_EXPECT_RESULT_SUCCESS(dsClient.Finalize());

    return nn::ResultSuccess();
}


nn::Result PostReadExpectShort(uint32_t postSize, uint32_t expectSize)
{
    uint32_t bytesTransferred = 0;

    NN_LOG("postSize %d\n", postSize);

    NNT_EXPECT_RESULT_SUCCESS(dsEndpoint[1].PostBuffer(&bytesTransferred, readBuffer, postSize));

    NN_LOG("expectSize %d bytesTransferred %d\n", expectSize, bytesTransferred);

    if (bytesTransferred != expectSize)
    {
        NN_ABORT();
    }

    return nn::ResultSuccess();
}


TEST(DsShortTransfer, 1TRB)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE, 64));                              // Short on 1st TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, 2TRB)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 2, 64));                          // Short on 1st TRB
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 2, TRB_DATA_SIZE + 64));          // Short on 2nd TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, 3TRB)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 3, 64));                          // Short on 1st TRB
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 3, TRB_DATA_SIZE + 64));          // Short on 2nd TRB
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 3, (TRB_DATA_SIZE * 2) + 64));    // Short on 3rd TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, 2_LINKTRB)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE, ((BULK_TRBS_IN_RING - 2) * TRB_DATA_SIZE) + 64));    // Short on 2 TRBs before link TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, 1_LINKTRB)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort((BULK_TRBS_IN_RING) * TRB_DATA_SIZE, ((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE) + 64));        // Short on 1 TRB before link TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, LINKTRB_1)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE, (BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE));           // Get to 1 TRB before link TRB
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 2, TRB_DATA_SIZE + 64));                                                      // Short on 1st TRB after link TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

TEST(DsShortTransfer, LINKTRB_2)
{
    NNT_EXPECT_RESULT_SUCCESS(SetupDevice());
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE, (BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE));           // Get to 1 TRB before link TRB
    NNT_EXPECT_RESULT_SUCCESS(PostReadExpectShort(TRB_DATA_SIZE * 3, (TRB_DATA_SIZE * 2) + 64));                                                // Short on 2nd TRB after link TRB
    NNT_EXPECT_RESULT_SUCCESS(TeardownDevice());
}

// TODO:
//  - short on 1TRB after link TRB
//  - short on 2TRB after link TRB

} // namespace ds
} // namespace usb
} // namespace nnt
