﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/usb/pd/usb_PdResult.h>
#include <nn/usb/pd/usb_PdCradle.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneGetCradleVdo.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        // API を呼ぶ間隔 (frame)
        const int64_t UpdateInterval = 20;

        nn::TimeSpan g_ApiExecutionTime;
        nn::os::Tick g_SceneStart;

        nn::usb::pd::CradleSession g_Session;

        nn::Result g_Dp2HdmiFwVersionResult;
        nn::usb::pd::VdmDp2HdmiFwVersion g_Dp2HdmiFwVersion;

        nn::Result g_PdcHFwVersionResult;
        nn::usb::pd::VdmPdcHFwVersion g_PdcHFwVersion;

        nn::Result g_PdcAFwVersionResult;
        nn::usb::pd::VdmPdcAFwVersion g_PdcAFwVersion;

        nn::Result g_McuFwVersionResult;
        nn::Bit32 g_McuFwVersion;

        nn::Result g_DeviceStateResult;
        nn::usb::pd::VdmDeviceState g_DeviceState;

        void Print() NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            // ヘッダ表示
            ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;
            ss << std::endl;

            ss << "Dp2HdmiFwVersionResult: ";
            if (g_Dp2HdmiFwVersionResult.IsSuccess())
            {
                ss << g_Dp2HdmiFwVersion <<  std::endl;
            }
            else
            {
                ss << ToString(g_Dp2HdmiFwVersionResult) << std::endl;
            }

            ss << "PdcHFwVersion: ";
            if (g_PdcHFwVersionResult.IsSuccess())
            {
                ss << g_PdcHFwVersion <<  std::endl;
            }
            else
            {
                ss << ToString(g_PdcHFwVersionResult) << std::endl;
            }

            ss << "PdcAFwVersion: ";
            if (g_PdcAFwVersionResult.IsSuccess())
            {
                ss << g_PdcAFwVersion <<  std::endl;
            }
            else
            {
                ss << ToString(g_PdcAFwVersionResult) << std::endl;
            }

            ss << "McuFwVersion: ";
            if (g_McuFwVersionResult.IsSuccess())
            {
                ss << g_McuFwVersion <<  std::endl;
            }
            else
            {
                ss << ToString(g_McuFwVersionResult) << std::endl;
            }

            ss << "DeviceState: ";
            if (g_DeviceStateResult.IsSuccess())
            {
                ss << "  DeviceType:" << g_DeviceState.Get<nn::usb::pd::VdmDeviceState::DeviceType>() <<  std::endl;
                ss << "  PowerDeriveryWarning:" << g_DeviceState.Get<nn::usb::pd::VdmDeviceState::PowerDeriveryWarning>() <<  std::endl;
                ss << "  RelayBoxUsbPower:" << g_DeviceState.Get<nn::usb::pd::VdmDeviceState::RelayBoxUsbPower>() <<  std::endl;
                ss << "  FatalError:" << g_DeviceState.Get<nn::usb::pd::VdmDeviceState::FatalError>() <<  std::endl;
            }
            else
            {
                ss << ToString(g_Dp2HdmiFwVersionResult) << std::endl;
            }

            ss << std::endl;

            // フッタ表示
            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;
            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }

        void Update() NN_NOEXCEPT
        {
            nn::usb::pd::Vdo vdo;
            auto start = nn::os::GetSystemTick();

            // TODO: vdo.storage へのアクセスは非推奨なので、適切な変換関数を用意する。
            {
                auto result = nn::usb::pd::GetCradleVdo(&vdo, &g_Session, nn::usb::pd::CradleVdmCommand_Dp2HdmiFwVersion);
                if (result.IsFailure())
                {
                    NN_LOG("[error] CradleVdmCommand_Dp2HdmiFwVersion module:%d description:%d\n", result.GetModule(), result.GetDescription());
                }
                g_Dp2HdmiFwVersionResult = result;
                g_Dp2HdmiFwVersion = static_cast<nn::usb::pd::VdmDp2HdmiFwVersion>(vdo.storage);
            }

            {
                auto result = nn::usb::pd::GetCradleVdo(&vdo, &g_Session, nn::usb::pd::CradleVdmCommand_PdcHFwVersion);
                if (result.IsFailure())
                {
                    NN_LOG("[error] CradleVdmCommand_PdcHFwVersion module:%d description:%d\n", result.GetModule(), result.GetDescription());
                }
                g_PdcHFwVersionResult = result;
                g_PdcHFwVersion = static_cast<nn::usb::pd::VdmDp2HdmiFwVersion>(vdo.storage);
            }

            {
                auto result = nn::usb::pd::GetCradleVdo(&vdo, &g_Session, nn::usb::pd::CradleVdmCommand_PdcAFwVersion);
                if (result.IsFailure())
                {
                    NN_LOG("[error] CradleVdmCommand_PdcAFwVersion module:%d description:%d\n", result.GetModule(), result.GetDescription());
                }
                g_PdcAFwVersionResult = result;
                g_PdcAFwVersion = static_cast<nn::usb::pd::VdmDp2HdmiFwVersion>(vdo.storage);
            }

            {
                auto result = nn::usb::pd::GetCradleVdo(&vdo, &g_Session, nn::usb::pd::CradleVdmCommand_McuFwVersion);
                if (result.IsFailure())
                {
                    NN_LOG("[error] CradleVdmCommand_McuFwVersion module:%d description:%d\n", result.GetModule(), result.GetDescription());
                }
                g_McuFwVersionResult = result;
                g_McuFwVersion = static_cast<nn::usb::pd::VdmDp2HdmiFwVersion>(vdo.storage);
            }

            {
                auto result = nn::usb::pd::GetCradleVdo(&g_DeviceState, &g_Session, nn::usb::pd::CradleVdmCommand_DeviceState);
                if (result.IsFailure())
                {
                    NN_LOG("[error] CradleVdmCommand_McuFwVersion module:%d description:%d\n", result.GetModule(), result.GetDescription());
                }
                g_DeviceStateResult = result;
            }

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();
        }

    } // namespace

    void SceneGetCradleVdo(SceneResult *pOutResult)
    {
        g_SceneStart = nn::os::GetSystemTick();

        nn::usb::pd::OpenCradleSession( &g_Session );

        for(int64_t frame = 0;; frame++)
        {
            input::Update();
            Print();

            if (frame % UpdateInterval == 0)
            {
                Update();
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        nn::usb::pd::CloseCradleSession(&g_Session);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_Top;
    }
}}}
