﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nnt/usbPdUtil/testUsbPd_utilState.h>

namespace nnt { namespace usb { namespace pd {

    bool IsCradleFamily( nn::usb::pd::StatusDeviceType deviceType ) NN_NOEXCEPT
    {
        return deviceType == nn::usb::pd::StatusDeviceType_Cradle ||
               deviceType == nn::usb::pd::StatusDeviceType_RelayBox ||
               deviceType == nn::usb::pd::StatusDeviceType_TableDock;
    }

    bool IsCradleLedControlable( nn::usb::pd::StatusDeviceType deviceType ) NN_NOEXCEPT
    {
        return deviceType == nn::usb::pd::StatusDeviceType_Cradle ||
               deviceType == nn::usb::pd::StatusDeviceType_RelayBox;
    }

    namespace {
        StateAssertMap* g_pStateAssertMap = nullptr;

        void MakeStateAssertMap()
        {
            auto pMap = new nnt::usb::pd::StateAssertMap();

            (*pMap)[State_NoConnection] =
                {
                    std::make_pair(true, false), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0), // currentPdo (0V / 0A)
                    std::make_pair(true, 0), // currentRdo (0th, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 0), // activeNoticeCountMin
                    std::make_pair(true, 0), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_Unconnected),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_OfficialAcAdaptor] =
                {
                    std::make_pair(true, true), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b104), // currentPdo (15V / 2.6A)
                    std::make_pair(true, 0x20041104), // currentRdo (2th, 2.6A / 2.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_AcAdaptor),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_AnkerMobileBattery] =
                {
                    std::make_pair(true, true), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b0a0), // currentPdo (15V / 1.6A)
                    std::make_pair(true, 0x400280a0), // currentRdo (4th, 1.6A / 1.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_AnkerPowerPort] =
                {
                    std::make_pair(true, true), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b12c), // currentPdo (15V / 3A)
                    std::make_pair(true, 0x3004b12c), // currentRdo (3nd, 3A / 3A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 2), // activeNoticeCountMin
                    std::make_pair(true, 2), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_AppleCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0002d12c), // currentPdo (9V / 3A)
                    std::make_pair(true, 0x2004b12c), // currentRdo (2nd, 3A / 3A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_GoogleDualPowerPort] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0001912C), // currentPdo (5V / 3A)
                    std::make_pair(true, 0x00000000), // currentRdo (0th, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 2), // activeNoticeCountMin
                    std::make_pair(true, 2), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_GooglePixelCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0002d0c8), // currentPdo (9V / 2A)
                    std::make_pair(true, 0x200320c8), // currentRdo (2nd, 2A / 2A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_RavpowerMobileBattery] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b0d2), // currentPdo (15V / 2.1A)
                    std::make_pair(true, 0x400348d2), // currentRdo (4th, 2.1A / 2.1A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 2), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_RavpowerMobileBatteryAfterWakeByBatterySaving] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x26019005), // currentPdo (DualRolePower / USBCommunicationsCapable / DataRoleSwap, 5V / 50mA)
                    std::make_pair(true, 0x10001405), // currentRdo (1st, 50mA / 50mA)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 2), // activeNoticeCountMin
                    std::make_pair(true, 3), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // クレードルに電源を入れずに本体に接続
            (*pMap)[State_CradleWithNoAcAdaptor] = (*pMap)[State_NoConnection];

            (*pMap)[State_CradleWithOfficialAcAdaptor] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b104), // currentPdo (15V / 2.6A)
                    std::make_pair(true, 0x20041104), // currentRdo (2th, 2.6A / 2.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, true), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Cradle),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Console),
                };
            (*pMap)[State_CradleWithHdmiTvOn] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b104), // currentPdo (15V / 2.6A)
                    std::make_pair(true, 0x20041104), // currentRdo (2th, 2.6A / 2.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, true), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Cradle),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Connected),
                    std::make_pair(true, nn::omm::OperationMode::Console),
                };
            (*pMap)[State_CradleWithAppleCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004a0c8), // currentPdo (14.8V / 2A)
                    std::make_pair(true, 0x200320c8), // currentRdo (2nd, 2A / 2A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, true), // isCradlePowerShortage
                    std::make_pair(true, true), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Cradle),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_CradleUsbHubUndetected),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_CradleWithPc] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x08019000), // currentPdo (ExternallyPowered, 5V / 0A)
                    std::make_pair(true, 0x10000000), // currentRdo (1st, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, true), // isCradlePowerShortage
                    std::make_pair(true, true), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Cradle),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_CradleUsbHubUndetected),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // 常時過電圧タイプ治具でクレードルを給電
            (*pMap)[State_CradleOverVoltage] = (*pMap)[State_NoConnection];

            // Power contract 後に過電圧になる治具でクレードルを給電
            (*pMap)[State_CradleOverVoltageAfterContract] = (*pMap)[State_NoConnection];

            // リレーボックスに電源を入れずに本体に接続
            (*pMap)[State_RelayBoxWithNoAcAdaptor] = (*pMap)[State_NoConnection];

            (*pMap)[State_RelayBoxWithHdmiTvOn] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b104), // currentPdo (15V / 2.6A)
                    std::make_pair(true, 0x20041104), // currentRdo (2th, 2.6A / 2.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, true), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_RelayBox),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Connected),
                    std::make_pair(true, nn::omm::OperationMode::Console),
                };
            (*pMap)[State_RelayBoxWithAppleCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004a0c8), // currentPdo (14.8V / 2A)
                    std::make_pair(true, 0x200320c8), // currentRdo (2nd, 2A / 2A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, true), // isCradlePowerShortage
                    std::make_pair(true, true), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_RelayBox),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_RelayBoxWithPc] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x08019000), // currentPdo (ExternallyPowered, 5V / 0A)
                    std::make_pair(true, 0x10000000), // currentRdo (1st, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, true), // isCradlePowerShortage
                    std::make_pair(true, true), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_RelayBox),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // テーブルドックに電源を入れずに本体に接続
            (*pMap)[State_TableDockWithNoAcAdaptor] = (*pMap)[State_NoConnection];

            (*pMap)[State_TableDockWithHdmiTvOn] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004b104), // currentPdo (15V / 2.6A)
                    std::make_pair(true, 0x20041104), // currentRdo (2th, 2.6A / 2.6A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_TableDock),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_EnoughPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_TableDockWithAppleCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0004a06e), // currentPdo (14.8V / 1.1A)
                    std::make_pair(true, 0x2001b86e), // currentRdo (2nd, 1.1A / 1.1A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, true), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_TableDock),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_TableDockWithGooglePixelCharger] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x0002d0c8), // currentPdo (9V / 2A)
                    std::make_pair(true, 0x200320c8), // currentRdo (2nd, 2A / 2A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, true), // isTableDockPowerShortage
                    std::make_pair(true, true), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_TableDock),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_CradleUsbHubUndetected),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            (*pMap)[State_TableDockWithPc] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x08019000), // currentPdo (ExternallyPowered, 5V / 0A)
                    std::make_pair(true, 0x10000000), // currentRdo (1st, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, true), // isTableDockPowerShortage
                    std::make_pair(true, true), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_TableDock),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_CradleUsbHubUndetected),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // 常時過電圧タイプ治具でクレードルを給電
            (*pMap)[State_TableDockOverVoltage] = (*pMap)[State_NoConnection];

            // Power contract 後に過電圧になる治具でクレードルを給電
            (*pMap)[State_TableDockOverVoltageAfterContract] = (*pMap)[State_NoConnection];

            (*pMap)[State_Pc] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0x00019000), // currentPdo (5V / 0A)
                    std::make_pair(true, 0x00000000), // currentRdo (0th, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_LowPower),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // NX 本体同士の接続
            (*pMap)[State_Nx] =
                {
                    std::make_pair(true, true), // IsActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(false, nn::usb::pd::StatusDataRole_Ufp),
                    std::make_pair(false, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(false, 0), // currentPdo (0V / 0A)
                    std::make_pair(false, 0), // currentRdo (0th, 0A / 0A)
                    std::make_pair(false, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(false, 1), // activeNoticeCountMin
                    std::make_pair(false, 1), // activeNoticeCountMax
                    std::make_pair(false, nn::psm::ChargerType_Unconnected),
                    std::make_pair(false, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };

            // 常時過電圧タイプ治具
            (*pMap)[State_OverVoltage] =
                {
                    std::make_pair(true, false), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Sink),
                    std::make_pair(true, 0), // currentPdo (0V / 0A)
                    std::make_pair(true, 0), // currentRdo (0th, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_OverVoltage),
                    std::make_pair(true, 0), // activeNoticeCountMin
                    std::make_pair(true, 0), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_NotSupported),
                    std::make_pair(true, 0),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            // Power Contract 後に過電圧を与える治具
            (*pMap)[State_OverVoltageAfterContract] = (*pMap)[State_OverVoltage];

            (*pMap)[State_UsbEther] =
                {
                    std::make_pair(true, true), // isActive
                    std::make_pair(false, nn::usb::pd::StatusPlugOrientation_Cc1),
                    std::make_pair(true, nn::usb::pd::StatusDataRole_Dfp),
                    std::make_pair(true, nn::usb::pd::StatusPowerRole_Source),
                    std::make_pair(true, 0), // currentPdo (0V / 0A)
                    std::make_pair(true, 0), // currentRdo (0th, 0A / 0A)
                    std::make_pair(true, false), // isElectronicallyMarkedCable
                    std::make_pair(true, nn::usb::pd::StatusAccessoryMode_None),
                    std::make_pair(true, false), // isDisplayPortAlternateMode
                    std::make_pair(true, false), // isCradlePowerShortage
                    std::make_pair(true, false), // isCradleWithUnofficialAcAdaptor
                    std::make_pair(true, false), // isTableDockPowerShortage
                    std::make_pair(true, false), // isTableDockWithUnofficialAcAdaptor
                    std::make_pair(true, nn::usb::pd::StatusDeviceType_Unknown),
                    std::make_pair(true, nn::usb::pd::StatusRequest_None),
                    std::make_pair(true, nn::usb::pd::StatusError_None),
                    std::make_pair(true, 1), // activeNoticeCountMin
                    std::make_pair(true, 1), // activeNoticeCountMax
                    std::make_pair(true, nn::psm::ChargerType_Unconnected),
                    std::make_pair(true, 1),
                    std::make_pair(true, nn::vi::HotplugState_Disconnected),
                    std::make_pair(true, nn::omm::OperationMode::Handheld),
                };
            // RavPower モバイルバッテリ接続状態でスリープさせた後 USB-Ether を接続
            (*pMap)[State_UsbEtherAfterSleepWithRavpowerMobileBattery] = (*pMap)[State_UsbEther];

            g_pStateAssertMap = pMap;
        } // NOLINT(impl/function_size)
    }

    StateAssertMap* GetStateAssertMap()
    {
        if (g_pStateAssertMap == nullptr)
        {
            MakeStateAssertMap();
        }

        return g_pStateAssertMap;
    }

    bool TestState(State state, const nn::usb::pd::Status &status, int32_t activeNoticeCount, nn::psm::ChargerType chargerType, int32_t ifCount, nn::vi::HotplugState& hotplugState, nn::omm::OperationMode operationMode)
    {
        auto pMap = GetStateAssertMap();
        StateAssert assert = (*pMap)[state];

        if (assert.isActive.first && assert.isActive.second != status.IsActive())
        {
            return false;
        }
        if (assert.plugOrientation.first && assert.plugOrientation.second != status.GetPlugOrientation())
        {
            return false;
        }
        if (assert.dataRole.first && assert.dataRole.second != status.GetDataRole())
        {
            return false;
        }
        if (assert.powerRole.first && assert.powerRole.second != status.GetPowerRole())
        {
            return false;
        }
        if (assert.currentPdo.first && assert.currentPdo.second != status.m_CurrentPdo.storage)
        {
            return false;
        }
        if (assert.currentRdo.first && assert.currentRdo.second != status.m_CurrentRdo.storage)
        {
            return false;
        }
        if (assert.isElectronicallyMarkedCable.first && assert.isElectronicallyMarkedCable.second != status.IsElectronicallyMarkedCable())
        {
            return false;
        }
        if (assert.accessoryMode.first && assert.accessoryMode.second != status.GetAccessoryMode())
        {
            return false;
        }
        if (assert.isDisplayPortAlternateMode.first && assert.isDisplayPortAlternateMode.second != status.IsDisplayPortAlternateMode())
        {
            return false;
        }
        if (assert.isCradlePowerShortage.first && assert.isCradlePowerShortage.second != status.IsCradlePowerShortage())
        {
            return false;
        }
        if (assert.isCradleWithUnofficialAcAdaptor.first && assert.isCradleWithUnofficialAcAdaptor.second != status.IsCradleWithUnofficialAcAdaptor())
        {
            return false;
        }
        if (assert.isTableDockPowerShortage.first && assert.isTableDockPowerShortage.second != status.IsTableDockPowerShortage())
        {
            return false;
        }
        if (assert.isTableDockWithUnofficialAcAdaptor.first && assert.isTableDockWithUnofficialAcAdaptor.second != status.IsTableDockWithUnofficialAcAdaptor())
        {
            return false;
        }
        if (assert.deviceType.first && assert.deviceType.second != status.GetDeviceType())
        {
            return false;
        }
        if (assert.request.first && assert.request.second != status.GetRequest())
        {
            return false;
        }
        if (assert.error.first && assert.error.second != status.GetError())
        {
            return false;
        }
        if ((assert.activeNoticeCountMin.first && assert.activeNoticeCountMin.second > activeNoticeCount) ||
            (assert.activeNoticeCountMax.first && assert.activeNoticeCountMax.second < activeNoticeCount)
           )
        {
            return false;
        }
        if (assert.chargerType.first && assert.chargerType.second != chargerType)
        {
            return false;
        }
        if (assert.ifCount.first && assert.ifCount.second != ifCount)
        {
            return false;
        }
        if (assert.hotplugState.first && assert.hotplugState.second != hotplugState)
        {
            return false;
        }
        if (assert.operationMode.first && assert.operationMode.second != operationMode)
        {
            return false;
        }
        return true;
    }
}}}
