﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/uart/uart.h>

#include <nn/gpio/gpio.h>

#include <nnt/gtest/gtest.h>

#include "../Common/testUart_Helper.h"
#include "testUart_ExtCon_details.h"

namespace nnt {
namespace uart {
namespace extcon {
namespace details {

namespace
{

// テストに使用するデータ一式
struct TestDataInfo
{
    nn::os::SystemEventType receiveReadyEvent;  // 受信イベント
    nn::uart::BaudRate      baudRate;           // テスト時のボーレート
    nn::uart::PortSession*  pSession;           // UART ポート
    nn::TimeSpan            pollingInterval;    // データ受信時のポーリング間隔
    size_t                  payloadSize;        // ACL パケットのペイロード長
    size_t                  packetSize;         // ACL パケット全体のサイズ
    int                     testCount;          // テストで送受信する ACL パケットの数
    char*                   pTestData;          // ACL パケットのペイロードに格納するデータ
    char*                   pReturnData;        // ループバックテストで返ってくるデータ
};

// テストの想定時間 [秒]
const int ExpectedTestSeconds = 10;

// デバイスがいなくなったと判定する時間
const nn::TimeSpan  CommunicationTimeOut = nn::TimeSpan::FromMilliSeconds(100);

/*
 * パケットのレスポンスを確認します。
 */
void CheckResponse(size_t* pOutReceivedBytes,
                   int* pOutReceivedPacketCount,
                   int* pOutCompletedPacketCount,
                   TestDataInfo* pTestData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutReceivedBytes);
    NN_ASSERT_NOT_NULL(pOutReceivedPacketCount);
    NN_ASSERT_NOT_NULL(pOutCompletedPacketCount);
    NN_ASSERT_NOT_NULL(pTestData);

    const size_t loopbackPayloadSizeMax   = 384;
    const size_t loopbackPacketHeaderSize =  12;

    *pOutReceivedBytes        = 0;
    *pOutReceivedPacketCount  = 0;
    *pOutCompletedPacketCount = 0;

    char s_ReceiveBuffer[UartBufferSize];
    size_t receiveBytes = loopbackPayloadSizeMax + loopbackPacketHeaderSize;
    size_t doneBytes = 0;

    auto result = nn::uart::Receive(&doneBytes,
                                    s_ReceiveBuffer,
                                    receiveBytes,
                                    pTestData->pSession);

    ASSERT_TRUE(result.IsSuccess());

    ASSERT_TRUE(nnt::uart::CompareTestData(pTestData->pReturnData,
                                           s_ReceiveBuffer,
                                           receiveBytes));
    *pOutReceivedBytes += receiveBytes;
    (*pOutReceivedPacketCount)++;
}

/*
 * テスト用データを初期化します。
 */
void InitializeTestData(TestDataInfo* pTestData,
                        nn::uart::PortSession* pSession,
                        nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pTestData);

    const size_t loopbackPayloadSizeMax   = 384;
    const size_t loopbackPacketHeaderSize =  12;

    pTestData->payloadSize = loopbackPayloadSizeMax;
    pTestData->packetSize  = pTestData->payloadSize + loopbackPacketHeaderSize;
    pTestData->baudRate    = baudRate;
    pTestData->pSession    = pSession;

    // 通信に規定時間かかる程度のパケット数を算出
    // TORIAEZU : MAX の速度は出ないため、割る係数を大き目に
    pTestData->testCount = static_cast<int>(baudRate) * ExpectedTestSeconds / pTestData->payloadSize / 20;
    NN_LOG("Test packet size : %d\n", pTestData->packetSize);
    NN_LOG("Test packet count: %d\n", pTestData->testCount);

    // テスト用 Loopback パケットを作成 (ハンドルとサイズ情報は Little Endian で格納)
    static char s_TestLoopbackData[loopbackPacketHeaderSize + loopbackPayloadSizeMax] =
        {
            0x19, 0x01, 0x03,
            static_cast<char>((pTestData->payloadSize + 7) & 0xFF),               // サイズ下位バイト
            static_cast<char>(((pTestData->payloadSize + 7) >> 8) & 0xFF),        // サイズ上位バイト
            0x91, 0x40,
            static_cast<char>(pTestData->payloadSize & 0xFF),                     // サイズ下位バイト
            static_cast<char>((pTestData->payloadSize >> 8) & 0xFF),              // サイズ上位バイト
            0x00, 0x00,
            0x00,
        };
    nnt::uart::GenerateTestData(s_TestLoopbackData + loopbackPacketHeaderSize,
                                pTestData->payloadSize);
    pTestData->pTestData = s_TestLoopbackData;

    // 返ってくるパケットは 2 byte 目、6 byte 目だけ値が変わって返ってくる
    static char s_ReturnLoopbackData[loopbackPacketHeaderSize + loopbackPayloadSizeMax];
    memcpy(s_ReturnLoopbackData, s_TestLoopbackData, loopbackPacketHeaderSize + loopbackPayloadSizeMax);
    pTestData->pReturnData = s_ReturnLoopbackData;
    pTestData->pReturnData[1] = 0x81;
    pTestData->pReturnData[5] = 0x94;

    // 1パケット受信したらシグナルされるようにする
    NN_ABORT_UNLESS(nn::uart::BindPortEvent(
        &pTestData->receiveReadyEvent,
        pTestData->pSession,
        nn::uart::PortEventType_ReceiveBufferReady,
        pTestData->packetSize));
}

/*
 * テスト用データに対する終了処理を行います。
 */
void FinalizeTestData(TestDataInfo* pTestData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pTestData);

    nn::uart::UnbindPortEvent(&pTestData->receiveReadyEvent, pTestData->pSession);
    nn::os::DestroySystemEvent(&pTestData->receiveReadyEvent);

}

/*
 * Loopbackでの送受信テストを行います。
 */
bool TestLoopback(nn::uart::PortName portName,
                  nn::uart::PortSession* pSession,
                  nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    NN_LOG("## Starting Loopback test\n");

    TestDataInfo testInfo;
    InitializeTestData(&testInfo, pSession, baudRate);

    // 送信バッファの空きを通知するイベント
    nn::os::SystemEventType sendReadyEvent;
    NN_ASSERT(nn::uart::BindPortEvent(
        &sendReadyEvent,
        testInfo.pSession,
        nn::uart::PortEventType_SendBufferReady,
        testInfo.packetSize));

    nn::os::ClearSystemEvent(&testInfo.receiveReadyEvent);
    size_t totalReceivedBytes = 0;
    int totalReceivedPackets = 0;
    int totalCompletedPackets = 0;
    auto startTick = nn::os::GetSystemTick();

    // Send->Receive を 1セットずつ行う
    // 規定数のレスポンスを受け取ったら終了
    while (totalReceivedPackets  < testInfo.testCount)
    {
        // Send
        //-------------------------------------------------
        if (nn::os::TimedWaitSystemEvent(&sendReadyEvent, CommunicationTimeOut))
        {
            nn::os::ClearSystemEvent(&sendReadyEvent);

            size_t doneBytes;
            nn::uart::Send(&doneBytes,
                testInfo.pSession,
                testInfo.pTestData,
                testInfo.packetSize);
            NN_ASSERT(doneBytes == testInfo.packetSize);
        }
        else
        {
            // 通信先と通信ができなくなったと判断
            nn::uart::UnbindPortEvent(&sendReadyEvent, pSession);
            nn::os::DestroySystemEvent(&sendReadyEvent);
            FinalizeTestData(&testInfo);
            return false;
        }

        //--------------------------------------------------

        // Receive とデータの受信確認
        //--------------------------------------------------
        size_t receivedBytes;
        int receivedPacketCount;
        int completedPacketCount;

        if(nn::os::TimedWaitSystemEvent(&testInfo.receiveReadyEvent, CommunicationTimeOut))
        {
            nn::os::ClearSystemEvent(&testInfo.receiveReadyEvent);

            CheckResponse(&receivedBytes, &receivedPacketCount, &completedPacketCount, &testInfo);
            if (::testing::Test::HasFatalFailure())
            {
                // 本当のテストの失敗は true で返す
                return true;
            }
        }
        else
        {
            // 通信先と通信ができなくなったと判断
            nn::uart::UnbindPortEvent(&sendReadyEvent, pSession);
            nn::os::DestroySystemEvent(&sendReadyEvent);
            FinalizeTestData(&testInfo);
            return false;
        }
        //--------------------------------------------------

        totalReceivedBytes += receivedBytes;
        totalReceivedPackets += receivedPacketCount;
        totalCompletedPackets += completedPacketCount;

        /*
        NN_LOG("Total: %d bytes  ACL loopback: %d packets  Completed: %d packets\n",
        totalReceivedBytes,
        totalReceivedPackets,
        totalCompletedPackets);
        */


        if (!(totalReceivedPackets % 100))
        {
            auto diffTick = nn::os::GetSystemTick() - startTick;
            NN_LOG("Total: %d bytes  loopback: %d packets  passedMs : %d (%08x)\n",
                totalReceivedBytes,
                totalReceivedPackets,
                diffTick.ToTimeSpan().GetMilliSeconds(),
                testInfo.pSession);
        }
    }

    auto diffTick = nn::os::GetSystemTick() - startTick;

    NN_LOG("\n");
    NN_LOG("Done!\n");

    // 経過時間とスループットを計測
    auto passedMs = diffTick.ToTimeSpan().GetMilliSeconds();
    if (passedMs > 0)
    {
        int bps = static_cast<long long int>(totalReceivedBytes) * 8000 / passedMs;
        NN_LOG("[Result]\n");
        NN_LOG("  Total received: %u bytes\n", totalReceivedBytes);
        NN_LOG("  Test time     : %ld msec\n", passedMs);
        NN_LOG("  Throughput    : %d bps\n", bps);
    }

    // ゴミが残っていないことを確認
    EXPECT_EQ(nn::uart::GetReadableLength(pSession), 0);
    if (nn::uart::GetReadableLength(pSession) > 0)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        static char s_ReceiveBuffer[4096];
        size_t doneBytes;
        auto result = nn::uart::Receive(&doneBytes,
                          s_ReceiveBuffer,
                          sizeof(s_ReceiveBuffer),
                          pSession);
        if (result.IsSuccess())
        {
            NN_LOG("[Remain data: %d bytes]\n", doneBytes);
            nnt::uart::DumpBinary(s_ReceiveBuffer, doneBytes);
        }
    }

    // 後始末
    nn::uart::UnbindPortEvent(&sendReadyEvent, pSession);
    nn::os::DestroySystemEvent(&sendReadyEvent);
    FinalizeTestData(&testInfo);

    return true;
}

}

bool TEST_LoopbackBody(nn::uart::PortName portName,
                       nn::uart::PortSession* pSession,
                       nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    return TestLoopback(portName, pSession, baudRate);
}

}}}}
