﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/uart/uart.h>

#include <nn/gpio/gpio.h>
#include <nn/pinmux/pinmux.h>

#include <nnt/gtest/gtest.h>

#include <nn/dd.h>

#include "../Common/testUart_Helper.h"
#include "testUart_ExtCon_details.h"

namespace
{

// 内部送信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_SendBuffer[4][nnt::uart::extcon::details::UartBufferSize];

// 内部受信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_ReceiveBuffer[4][nnt::uart::extcon::details::UartBufferSize];

/*
 * SYNC を確認します。
 */
void TestSync(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    const char Command[] =
        {
            0x19, 0x01, 0x03, 0x07, 0x00,
            0xA5, 0x01, 0x01, 0x7E, 0x00, 0x00,
            0x00,
        };

    const char ExpectedResult[] =
        {
            0x19, 0x81, 0x03, 0x07, 0x00,
            0xA5, 0x01, 0x02, 0x7D, 0x00, 0x00,
            0x00,
        };

    NN_LOG("## Test [Sync]\n");

    size_t receivedBytes;
    char receiveBuffer[sizeof(ExpectedResult)];
    nnt::uart::extcon::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体とサイズを確認
    ASSERT_EQ(receivedBytes, sizeof(ExpectedResult));
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResult,
                                receiveBuffer,
                                sizeof(ExpectedResult)));
}

/*
 * Inquiry コマンドを確認します。
 */
void TestInquiry(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    const char Command[] =
        {
            0x19, 0x01, 0x03, 0x07, 0x00,
            0x91, 0x01, 0x00, 0x00, 0x00, 0x00,
            0x00,
        };

    // Inquiry の応答全体は 20 バイト
    const size_t ExpectedResultSize = 20;

    const char ExpectedResultHead[] =
        {
            0x19, 0x81, 0x03, 0x0F, 0x00,
            0x94, 0x01, 0x08, 0x00, 0x00, 0x00,
            0x00,
        };

    NN_LOG("## Test [Inquiry]\n");

    size_t receivedBytes;
    char receiveBuffer[ExpectedResultSize];
    nnt::uart::extcon::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体のサイズと先頭部分を比較
    // (13 バイト目以降は変わらない保証がなさそうなので、比較せず受信確認のみ)
    ASSERT_EQ(receivedBytes, ExpectedResultSize);
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResultHead,
                                receiveBuffer,
                                sizeof(ExpectedResultHead)));
}

/*
 * UART のボーレート設定を変更します。
 */
void UpdateBaudRate(nn::uart::PortSession* pSession, uint32_t baudRateHz) NN_NOEXCEPT
{
    // Update_UART_Baud_Rate
    const char Command[] =
        {
            0x19, 0x01, 0x03, 0x0F, 0x00,
            0x91, 0x20, 0x08, 0x00, 0x00, 0x00, 0x00,
            static_cast<char>( baudRateHz        & 0xFF),  // Baud rate (Little endian)
            static_cast<char>((baudRateHz >>  8) & 0xFF),
            static_cast<char>((baudRateHz >> 16) & 0xFF),
            static_cast<char>((baudRateHz >> 24) & 0xFF),
            0x00, 0x00, 0x00, 0x00
        };
    const char ExpectedResult[] =
        {
            0x19, 0x81, 0x03, 0x07, 0x00,
            0x94, 0x20, 0x00, 0x00, 0x00, 0x00,
            0x00,
        };

    NN_LOG("## Test [Update_UART_Baud_Rate]\n");

    size_t receivedBytes;
    char receiveBuffer[sizeof(ExpectedResult)];
    nnt::uart::extcon::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体とサイズを確認
    ASSERT_EQ(receivedBytes, sizeof(ExpectedResult));
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResult,
                                receiveBuffer,
                                sizeof(ExpectedResult)));
}

/*
 * ExtCon へのコマンド送信テストを行います。
 */
void TestCommandForExtCon(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    typedef void (*TestCommandFunctionType)(nn::uart::PortSession*);

    // テスト関数の一覧
    const TestCommandFunctionType TestFunctions[] =
    {
        TestSync,
        TestInquiry
    };

    for (const auto TestFunction : TestFunctions)
    {
        NN_ABORT_UNLESS_NOT_NULL(TestFunction);

        TestFunction(pSession);

        // 失敗していたらテストを中断
        if (::testing::Test::HasFatalFailure())
        {
            break;
        }
    }
}

/*
 * UART の設定を行い、バッファ情報を取得します。
 */
void ConfigureUart(nn::uart::PortName portName,
                   nn::uart::BaudRate baudRate,
                   nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    NN_LOG("Using 115200 baud for setting\n");

    nn::uart::PortConfig  portConfig(
        nn::uart::BaudRate_115200,  // 設定時は 115200 baud を使用
        g_SendBuffer[portName],
        sizeof(g_SendBuffer[portName]),
        g_ReceiveBuffer[portName],
        sizeof(g_ReceiveBuffer[portName])
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));

    // UART のボーレートを変更
    UpdateBaudRate(&portSession, static_cast<int>(baudRate));
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }
    nn::uart::ClosePort(&portSession);
}
void DeconfigureUart(nn::uart::PortName portName,
                   nn::uart::BaudRate baudRate,
                   nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    NN_LOG("Using 115200 baud for setting\n");

    nn::uart::PortConfig  portConfig(
        baudRate,
        g_SendBuffer[portName],
        sizeof(g_SendBuffer[portName]),
        g_ReceiveBuffer[portName],
        sizeof(g_ReceiveBuffer[portName])
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));

    // UART のボーレートを変更
    UpdateBaudRate(&portSession, static_cast<int>(nn::uart::BaudRate_115200));
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    nn::uart::ClosePort(&portSession);
}

}

namespace nnt {
namespace uart {
namespace extcon {

void TEST_BasicCommand(nn::uart::PortName portName,
                       nn::uart::BaudRate baudRate,
                       nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    // ReceiveEnd イベントを使用するので、対応可否を確認
    ASSERT_TRUE(nn::uart::IsSupportedPortEvent(portName,
                                               nn::uart::PortEventType_ReceiveEnd));

    nn::uart::PortConfig  portConfig(
        baudRate,
        g_SendBuffer[portName],
        sizeof(g_SendBuffer[portName]),
        g_ReceiveBuffer[portName],
        sizeof(g_ReceiveBuffer[portName])
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));

    TestCommandForExtCon(&portSession);

    nn::uart::ClosePort(&portSession);
}

void TEST_Loopback(nn::uart::PortName portName,
                   nn::uart::BaudRate baudRate,
                   nn::uart::FlowControlMode flowControlMode
                   ) NN_NOEXCEPT
{
    bool loopbackTestResult = false;

    // UART の初期化
    ConfigureUart(portName, baudRate, flowControlMode);

    NN_LOG("Using %d baud for testing\n", static_cast<int>(baudRate));

    nn::uart::PortConfig  portConfig(
        baudRate,
        g_SendBuffer[portName],
        sizeof(g_SendBuffer[portName]),
        g_ReceiveBuffer[portName],
        sizeof(g_ReceiveBuffer[portName])
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));

    NN_LOG("Port(%d)=%08x\n", portName, &portSession);

    // ループバックテストを実行
    loopbackTestResult = details::TEST_LoopbackBody(portName, &portSession, baudRate);

    while (!loopbackTestResult)
    {
        // 通信先のデバイスがいなくなった場合、接続を待って、もう一度ループバックテストを回す
        nn::uart::ClosePort(&portSession);

        nn::gpio::GpioPadSession gpioSession;
        nn::pinmux::PinmuxSession pinmuxSession;

        nn::gpio::Initialize();
        nn::pinmux::Initialize();

        if (portName == nn::uart::PortName_ExtConS)
        {
            nn::pinmux::OpenSession(&pinmuxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
            nn::pinmux::SetPinAssignment(&pinmuxSession, nn::pinmux::PinAssignment_ExtConTxGpio);
            nn::gpio::OpenSession(&gpioSession, nn::gpio::GpioPadName_ExtconDetS);
        }
        else
        {
            nn::pinmux::OpenSession(&pinmuxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
            nn::pinmux::SetPinAssignment(&pinmuxSession, nn::pinmux::PinAssignment_ExtConTxGpio);
            nn::gpio::OpenSession(&gpioSession, nn::gpio::GpioPadName_ExtconDetU);
        }
        nn::gpio::SetDirection(&gpioSession, nn::gpio::Direction_Input);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        while (1)
        {
            NN_LOG("Waiting for ExtCon detection... : ExtCon:%s\n",
                (nn::gpio::GetValue(&gpioSession) == nn::gpio::GpioValue_Low) ? "True" : "False");

            if (nn::gpio::GetValue(&gpioSession) == nn::gpio::GpioValue_Low)
            {
                NN_LOG("ExtConU is detected!\n");
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        nn::gpio::CloseSession(&gpioSession);

        nn::pinmux::CloseSession(&pinmuxSession);

        if (portName == nn::uart::PortName_ExtConS)
        {

            nn::pinmux::OpenSession(&pinmuxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
            nn::pinmux::SetPinAssignment(&pinmuxSession, nn::pinmux::PinAssignment_ExtConTxUart);
        }
        else
        {
            nn::pinmux::OpenSession(&pinmuxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
            nn::pinmux::SetPinAssignment(&pinmuxSession, nn::pinmux::PinAssignment_ExtConTxUart);
        }

        // UART の初期化
        ConfigureUart(portName, baudRate, flowControlMode);

        ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));
        loopbackTestResult = details::TEST_LoopbackBody(portName, &portSession, baudRate);
    }

    nn::uart::ClosePort(&portSession);
    DeconfigureUart(portName, baudRate, flowControlMode);
}

namespace details {

void TransferCommand(size_t* pReceiveBytes,
                     char* pOutBuffer,
                     size_t outBufferSize,
                     nn::uart::PortSession* pSession,
                     const char* pCommand,
                     size_t commandSize,
                     bool isDumpData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pReceiveBytes);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pSession);
    NN_ABORT_UNLESS_NOT_NULL(pCommand);

    nn::os::SystemEventType receiveEndEvent;
    // 受信完了イベントの登録
    ASSERT_TRUE(nn::uart::BindPortEvent(&receiveEndEvent,
                                        pSession,
                                        nn::uart::PortEventType_ReceiveEnd,
                                        0));

    nn::os::ClearSystemEvent(&receiveEndEvent);

    // コマンドを送信
    size_t doneBytes;
    nn::uart::Send(&doneBytes,
                   pSession,
                   pCommand,
                   commandSize);
    ASSERT_EQ(commandSize, doneBytes);

    if (isDumpData)
    {
        NN_LOG("Sent data (%d bytes):\n  ", doneBytes);
        nnt::uart::DumpBinary(pCommand, doneBytes);
    }

    // 受信完了待ち
    nn::os::WaitSystemEvent(&receiveEndEvent);

    // コマンドの応答を受信
    size_t readableLength = nn::uart::GetReadableLength(pSession);
    auto result = nn::uart::Receive(pReceiveBytes, pOutBuffer, readableLength, pSession);
    ASSERT_TRUE(result.IsSuccess());

    if (isDumpData)
    {
        NN_LOG("Received data (%d bytes):\n  ", *pReceiveBytes);
        nnt::uart::DumpBinary(pOutBuffer, *pReceiveBytes);
    }

    // 受信完了イベントの削除
    ASSERT_TRUE(nn::uart::UnbindPortEvent(&receiveEndEvent, pSession));
    nn::os::DestroySystemEvent(&receiveEndEvent);
}

}  // details

}}}
